<?php
/**
 * Directorist Adapter for Universal Directory Map
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__FILE__) . 'base-adapter.php';

class SFRDM_Directorist_Adapter extends SFRDM_Base_Adapter {
    
    protected function init() {
        $this->post_type = 'at_biz_dir';
        $this->category_taxonomy = 'at_biz_dir-category';
        $this->tag_taxonomy = 'at_biz_dir-tags';
        $this->plugin_name = 'Directorist';
    }
    
    public function get_listings_query_args($category = '', $post_type = '') {
        $args = array(
            'post_type' => $this->post_type,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'meta_query' => array(
                'relation' => 'AND',
                array(
                    'key' => '_manual_lat',
                    'value' => '',
                    'compare' => '!='
                ),
                array(
                    'key' => '_manual_lng',
                    'value' => '',
                    'compare' => '!='
                )
            )
        );
        
        if (!empty($category)) {
            $args['tax_query'] = array(
                array(
                    'taxonomy' => $this->category_taxonomy,
                    'field' => 'slug',
                    'terms' => $category
                )
            );
        }
        
        return $args;
    }
    
    public function extract_listing_data($post) {
        $data = array(
            'id' => $post->ID,
            'title' => html_entity_decode($post->post_title),
            'lat' => (float) get_post_meta($post->ID, '_manual_lat', true),
            'lng' => (float) get_post_meta($post->ID, '_manual_lng', true),
            'address' => get_post_meta($post->ID, '_address', true),
            'phone' => get_post_meta($post->ID, '_phone', true),
            'email' => get_post_meta($post->ID, '_email', true),
            'website' => get_post_meta($post->ID, '_website', true),
            'url' => get_permalink($post->ID),
            'categories' => array(),
            'tags' => array()
        );
        
        // Get categories
        $categories = wp_get_post_terms($post->ID, $this->category_taxonomy, array('fields' => 'names'));
        $data['categories'] = is_array($categories) ? $categories : array();
        
        // Get tags
        $tags = wp_get_post_terms($post->ID, $this->tag_taxonomy, array('fields' => 'names'));
        $data['tags'] = is_array($tags) ? $tags : array();
        
        // Get image
        $data['image'] = $this->get_listing_image($post->ID);
        
        // Get category icon
        $data['category_icon'] = $this->get_category_icon($post->ID);
        
        // Get primary category
        $primary_category = $this->get_primary_category($post->ID);
        $data['category_name'] = $primary_category ? $primary_category->name : '';
        $data['category_slug'] = $primary_category ? $primary_category->slug : '';
        
        return $data;
    }
    
    public function get_available_categories() {
        $categories = get_terms(array(
            'taxonomy' => $this->category_taxonomy,
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC'
        ));
        
        if (is_wp_error($categories)) {
            return array();
        }
        
        return $this->organize_categories_hierarchically($categories);
    }
    
    public function get_listing_tags($post_id) {
        $tags = array();
        $directorist_tags = get_the_terms($post_id, $this->tag_taxonomy);
        
        if (!is_wp_error($directorist_tags) && $directorist_tags) {
            foreach ($directorist_tags as $tag) {
                $tags[] = array(
                    'name' => $tag->name,
                    'slug' => $tag->slug,
                    'url' => get_term_link($tag)
                );
            }
        }
        
        return $tags;
    }
    
    public function get_primary_category($listing_id) {
        $categories = get_the_terms($listing_id, $this->category_taxonomy);
        
        if (is_wp_error($categories) || empty($categories)) {
            return null;
        }
        
        // Smart category selection: prefer categories with more listings
        $selected_category = $categories[0];
        $max_count = $selected_category->count;
        
        foreach ($categories as $category) {
            if ($category->count > $max_count) {
                $selected_category = $category;
                $max_count = $category->count;
            }
        }
        
        return $selected_category;
    }
    
    public function get_category_icon($listing_id) {
        $primary_category = $this->get_primary_category($listing_id);
        
        if (!$primary_category) {
            return '';
        }
        
        // Get Directorist category icon
        $icon_class = get_term_meta($primary_category->term_id, 'category_icon', true);
        
        if ($icon_class) {
            return $icon_class;
        }
        
        return '';
    }
    
    public function get_listing_image($post_id) {
        // Try Directorist's preview image first
        $preview_img = get_post_meta($post_id, '_listing_prv_img', true);
        if (!empty($preview_img) && is_numeric($preview_img)) {
            return wp_get_attachment_image_url($preview_img, 'thumbnail');
        }
        
        // Try Directorist's listing images array
        $listing_images = get_post_meta($post_id, '_listing_img', true);
        if (!empty($listing_images) && is_array($listing_images)) {
            $first_image = reset($listing_images);
            if (is_numeric($first_image)) {
                return wp_get_attachment_image_url($first_image, 'thumbnail');
            }
        }
        
        // Fallback to featured image
        return get_the_post_thumbnail_url($post_id, 'medium');
    }
    
    public function is_map_hidden($post_id) {
        return get_post_meta($post_id, '_hide_map', true);
    }
    
    public function get_category_analytics($category, $post_type) {
        $category_analytics = array(
            'name' => $category->name,
            'total' => 0,
            'with_coordinates' => 0,
            'with_images' => 0,
            'with_hidden_maps' => 0
        );
        
        $listings = get_posts(array(
            'post_type' => $this->post_type,
            'posts_per_page' => -1,
            'tax_query' => array(
                array(
                    'taxonomy' => $this->category_taxonomy,
                    'field' => 'term_id',
                    'terms' => $category->term_id
                )
            )
        ));
        
        foreach ($listings as $listing) {
            $category_analytics['total']++;
            
            $lat = get_post_meta($listing->ID, '_manual_lat', true);
            $lng = get_post_meta($listing->ID, '_manual_lng', true);
            
            if ($lat && $lng) {
                $category_analytics['with_coordinates']++;
            }
            
            if ($this->get_listing_image($listing->ID)) {
                $category_analytics['with_images']++;
            }
            
            if ($this->is_map_hidden($listing->ID)) {
                $category_analytics['with_hidden_maps']++;
            }
        }
        
        return $category_analytics;
    }
    
    public function is_plugin_active() {
        return class_exists('Directorist_Base') || class_exists('Directorist\Directorist') || class_exists('ATBDP');
    }
    
    /**
     * Organize categories hierarchically
     */
    private function organize_categories_hierarchically($categories) {
        $parent_categories = array();
        $child_categories = array();
        
        foreach ($categories as $category) {
            if ($category->parent == 0) {
                $parent_categories[] = $category;
            } else {
                $child_categories[] = $category;
            }
        }
        
        // Sort parent categories alphabetically
        usort($parent_categories, function($a, $b) {
            return strcasecmp($a->name, $b->name);
        });
        
        // Sort child categories alphabetically
        usort($child_categories, function($a, $b) {
            return strcasecmp($a->name, $b->name);
        });
        
        // Build hierarchical structure
        $hierarchical_categories = array();
        
        foreach ($parent_categories as $parent) {
            $hierarchical_categories[] = $parent;
            
            // Add children of this parent
            foreach ($child_categories as $child) {
                if ($child->parent == $parent->term_id) {
                    $hierarchical_categories[] = $child;
                }
            }
        }
        
        return $hierarchical_categories;
    }
}
