<?php
/**
 * Business Directory Adapter for Universal Directory Map
 */

if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__FILE__) . 'base-adapter.php';

class SFRDM_Business_Directory_Adapter extends SFRDM_Base_Adapter {
    
    protected function init() {
        $this->post_type = 'wpbdp_listing';
        $this->category_taxonomy = 'wpbdp-category';
        $this->tag_taxonomy = 'wpbdp-tags';
        $this->plugin_name = 'Business Directory';
    }
    
    public function get_listings_query_args($category = '', $post_type = '') {
        // Business Directory without Google Maps add-on has limited mapping support
        // Return empty results to indicate no map support
        return array(
            'post_type' => $this->post_type,
            'post_status' => 'publish',
            'posts_per_page' => 0, // Return no results
            'meta_query' => array(
                array(
                    'key' => '_wpbdp_no_mapping_support',
                    'value' => '1',
                    'compare' => '='
                )
            )
        );
    }
    
    public function extract_listing_data($post) {
        // Check if Google Maps add-on is active
        $has_google_maps = $this->is_google_maps_addon_active();
        
        // Try to get coordinates from Google Maps add-on
        $lat = 0;
        $lng = 0;
        $has_coordinates = false;
        
        if ($has_google_maps) {
            // Try different possible meta field names for Google Maps add-on
            $lat = (float) get_post_meta($post->ID, '_wpbdp_google_maps_lat', true);
            $lng = (float) get_post_meta($post->ID, '_wpbdp_google_maps_lng', true);
            
            // If not found, try alternative field names
            if (!$lat && !$lng) {
                $lat = (float) get_post_meta($post->ID, '_wpbdp[fields][google_maps_lat]', true);
                $lng = (float) get_post_meta($post->ID, '_wpbdp[fields][google_maps_lng]', true);
            }
            
            // If still not found, try other possible variations
            if (!$lat && !$lng) {
                $lat = (float) get_post_meta($post->ID, '_wpbdp_latitude', true);
                $lng = (float) get_post_meta($post->ID, '_wpbdp_longitude', true);
            }
            
            $has_coordinates = ($lat != 0 || $lng != 0);
        }
        
        $data = array(
            'id' => $post->ID,
            'title' => html_entity_decode($post->post_title),
            'lat' => $lat,
            'lng' => $lng,
            'address' => get_post_meta($post->ID, '_wpbdp[fields][address]', true),
            'phone' => get_post_meta($post->ID, '_wpbdp[fields][phone]', true),
            'categories' => array(),
            'tags' => array(),
            'has_coordinates' => $has_coordinates,
            'has_google_maps_addon' => $has_google_maps
        );
        
        // Get categories
        $categories = wp_get_post_terms($post->ID, $this->category_taxonomy, array('fields' => 'names'));
        $data['categories'] = is_array($categories) ? $categories : array();
        
        // Get tags
        $tags = wp_get_post_terms($post->ID, $this->tag_taxonomy, array('fields' => 'names'));
        $data['tags'] = is_array($tags) ? $tags : array();
        
        // Get image
        $data['image'] = $this->get_listing_image($post->ID);
        
        // Get primary category
        $primary_category = $this->get_primary_category($post->ID);
        $data['category_name'] = $primary_category ? $primary_category->name : '';
        $data['category_slug'] = $primary_category ? $primary_category->slug : '';
        
        return $data;
    }
    
    public function get_available_categories() {
        // Business Directory without Google Maps add-on doesn't support mapping
        // Return empty array to indicate no categories available for maps
        return array();
    }
    
    public function get_listing_tags($post_id) {
        $tags = array();
        $wpbdp_tags = get_the_terms($post_id, $this->tag_taxonomy);
        
        if (!is_wp_error($wpbdp_tags) && $wpbdp_tags) {
            foreach ($wpbdp_tags as $tag) {
                $tags[] = array(
                    'name' => $tag->name,
                    'slug' => $tag->slug,
                    'url' => get_term_link($tag)
                );
            }
        }
        
        return $tags;
    }
    
    public function get_primary_category($listing_id) {
        $categories = get_the_terms($listing_id, $this->category_taxonomy);
        
        if (is_wp_error($categories) || empty($categories)) {
            return null;
        }
        
        // Smart category selection: prefer categories with more listings
        $selected_category = $categories[0];
        $max_count = $selected_category->count;
        
        foreach ($categories as $category) {
            if ($category->count > $max_count) {
                $selected_category = $category;
                $max_count = $category->count;
            }
        }
        
        return $selected_category;
    }
    
    public function get_category_icon($listing_id) {
        $primary_category = $this->get_primary_category($listing_id);
        
        if (!$primary_category) {
            return '';
        }
        
        // Business Directory doesn't have built-in category icons
        // This could be extended to support custom icons if needed
        return '';
    }
    
    public function get_listing_image($post_id) {
        // Business Directory uses standard WordPress featured image
        return get_the_post_thumbnail_url($post_id, 'medium');
    }
    
    public function is_map_hidden($post_id) {
        // Business Directory doesn't have a built-in hide map feature
        // This could be extended to support custom meta fields if needed
        return false;
    }
    
    public function get_category_analytics($category, $post_type) {
        $category_analytics = array(
            'name' => $category->name,
            'total' => 0,
            'with_coordinates' => 0,
            'with_images' => 0,
            'with_hidden_maps' => 0
        );
        
        $listings = get_posts(array(
            'post_type' => $this->post_type,
            'posts_per_page' => -1,
            'tax_query' => array(
                array(
                    'taxonomy' => $this->category_taxonomy,
                    'field' => 'term_id',
                    'terms' => $category->term_id
                )
            )
        ));
        
        foreach ($listings as $listing) {
            $category_analytics['total']++;
            
            $lat = get_post_meta($listing->ID, '_wpbdp[fields][_x_geolocation-lat]', true);
            $lng = get_post_meta($listing->ID, '_wpbdp[fields][_x_geolocation-lng]', true);
            
            if ($lat && $lng) {
                $category_analytics['with_coordinates']++;
            }
            
            if ($this->get_listing_image($listing->ID)) {
                $category_analytics['with_images']++;
            }
            
            if ($this->is_map_hidden($listing->ID)) {
                $category_analytics['with_hidden_maps']++;
            }
        }
        
        return $category_analytics;
    }
    
    public function is_plugin_active() {
        return class_exists('WPBDP');
    }
    
    /**
     * Check if the Google Maps add-on is active
     */
    public function is_google_maps_addon_active() {
        // Check for various possible class names for the Google Maps add-on
        return class_exists('WPBDP_Google_Maps') || 
               class_exists('WPBDP_GoogleMaps') || 
               class_exists('WPBDP_GoogleMapsModule') ||
               function_exists('wpbdp_google_maps') ||
               is_plugin_active('business-directory-plugin-google-maps/business-directory-plugin-google-maps.php');
    }
    
    /**
     * Get plugin compatibility message
     */
    public function get_compatibility_message() {
        return 'Business Directory Plugin detected, but it requires the premium Google Maps add-on for mapping functionality. For free mapping features, consider using Directorist or GeoDirectory instead.';
    }
    
    /**
     * Organize categories hierarchically
     */
    private function organize_categories_hierarchically($categories) {
        $parent_categories = array();
        $child_categories = array();
        
        foreach ($categories as $category) {
            if ($category->parent == 0) {
                $parent_categories[] = $category;
            } else {
                $child_categories[] = $category;
            }
        }
        
        // Sort parent categories alphabetically
        usort($parent_categories, function($a, $b) {
            return strcasecmp($a->name, $b->name);
        });
        
        // Sort child categories alphabetically
        usort($child_categories, function($a, $b) {
            return strcasecmp($a->name, $b->name);
        });
        
        // Build hierarchical structure
        $hierarchical_categories = array();
        
        foreach ($parent_categories as $parent) {
            $hierarchical_categories[] = $parent;
            
            // Add children of this parent
            foreach ($child_categories as $child) {
                if ($child->parent == $parent->term_id) {
                    $hierarchical_categories[] = $child;
                }
            }
        }
        
        return $hierarchical_categories;
    }
}
