<?php
/**
 * Base Adapter Class for SFR Directory Map
 * 
 * This class provides the interface that all directory plugin adapters must implement.
 * Each directory plugin (Directorist, GeoDirectory, Business Directory, etc.) should
 * have its own adapter that extends this base class.
 */

if (!defined('ABSPATH')) {
    exit;
}

abstract class SFRDM_Base_Adapter {
    
    /**
     * The post type this adapter handles
     */
    protected $post_type;
    
    /**
     * The taxonomy for categories
     */
    protected $category_taxonomy;
    
    /**
     * The taxonomy for tags (if supported)
     */
    protected $tag_taxonomy;
    
    /**
     * Plugin display name
     */
    protected $plugin_name;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->init();
    }
    
    /**
     * Initialize the adapter
     */
    abstract protected function init();
    
    /**
     * Get the post type this adapter handles
     */
    public function get_post_type() {
        return $this->post_type;
    }
    
    /**
     * Get the category taxonomy
     */
    public function get_category_taxonomy() {
        return $this->category_taxonomy;
    }
    
    /**
     * Get the tag taxonomy
     */
    public function get_tag_taxonomy() {
        return $this->tag_taxonomy;
    }
    
    /**
     * Get the plugin display name
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }
    
    /**
     * Check if this adapter can handle the given post type
     */
    public function can_handle($post_type) {
        return $post_type === $this->post_type;
    }
    
    /**
     * Get query arguments for listings with coordinates
     */
    abstract public function get_listings_query_args($category = '', $post_type = '');
    
    /**
     * Extract listing data from a post
     */
    abstract public function extract_listing_data($post);
    
    /**
     * Get available categories
     */
    abstract public function get_available_categories();
    
    /**
     * Get listing tags
     */
    abstract public function get_listing_tags($post_id);
    
    /**
     * Get primary category for a listing
     */
    abstract public function get_primary_category($listing_id);
    
    /**
     * Get category icon
     */
    abstract public function get_category_icon($listing_id);
    
    /**
     * Get listing image
     */
    abstract public function get_listing_image($post_id);
    
    /**
     * Check if map is hidden for a listing
     */
    abstract public function is_map_hidden($post_id);
    
    /**
     * Get analytics data for a category
     */
    abstract public function get_category_analytics($category, $post_type);
    
    /**
     * Check if the plugin is active
     */
    abstract public function is_plugin_active();
}


