<?php
/**
 * Adapter Manager for Universal Directory Map
 * 
 * This class manages all directory plugin adapters and provides a unified interface
 * for working with different directory plugins.
 */

if (!defined('ABSPATH')) {
    exit;
}

class SFRDM_Adapter_Manager {
    
    /**
     * Array of loaded adapters
     */
    private $adapters = array();
    
    /**
     * Current active adapter
     */
    private $active_adapter = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->load_adapters();
        $this->detect_active_plugin();
    }
    
    /**
     * Load all available adapters
     */
    private function load_adapters() {
        $adapter_files = array(
            'directorist-adapter.php',
            'business-directory-adapter.php'
        );
        
        foreach ($adapter_files as $file) {
            $file_path = plugin_dir_path(__FILE__) . $file;
            if (file_exists($file_path)) {
                require_once $file_path;
            }
        }
        
        // Register adapters
        $this->register_adapter('directorist', 'SFRDM_Directorist_Adapter');
        $this->register_adapter('business_directory', 'SFRDM_Business_Directory_Adapter');
    }
    
    /**
     * Register an adapter
     */
    private function register_adapter($key, $class_name) {
        if (class_exists($class_name)) {
            $this->adapters[$key] = new $class_name();
        }
    }
    
    /**
     * Detect which directory plugin is active and set the active adapter
     */
    private function detect_active_plugin() {
        foreach ($this->adapters as $key => $adapter) {
            if ($adapter->is_plugin_active()) {
                $this->active_adapter = $adapter;
                break;
            }
        }
    }
    
    /**
     * Get the active adapter
     */
    public function get_active_adapter() {
        return $this->active_adapter;
    }
    
    /**
     * Get adapter by key
     */
    public function get_adapter($key) {
        return isset($this->adapters[$key]) ? $this->adapters[$key] : null;
    }
    
    /**
     * Get all adapters
     */
    public function get_all_adapters() {
        return $this->adapters;
    }
    
    /**
     * Get the detected plugin name
     */
    public function get_detected_plugin_name() {
        if ($this->active_adapter) {
            return $this->active_adapter->get_plugin_name();
        }
        return 'Unknown';
    }
    
    /**
     * Get compatibility message for the detected plugin
     */
    public function get_compatibility_message() {
        if ($this->active_adapter && method_exists($this->active_adapter, 'get_compatibility_message')) {
            return $this->active_adapter->get_compatibility_message();
        }
        return '';
    }
    
    /**
     * Get the detected plugin post type
     */
    public function get_detected_plugin_post_type() {
        if ($this->active_adapter) {
            return $this->active_adapter->get_post_type();
        }
        return 'at_biz_dir'; // Default to Directorist
    }
    
    /**
     * Check if a plugin is detected
     */
    public function is_plugin_detected() {
        return $this->active_adapter !== null;
    }
    
    /**
     * Get query arguments for listings
     */
    public function get_listings_query_args($category = '', $post_type = '') {
        if ($this->active_adapter) {
            return $this->active_adapter->get_listings_query_args($category, $post_type);
        }
        return array();
    }
    
    /**
     * Extract listing data
     */
    public function extract_listing_data($post) {
        if ($this->active_adapter) {
            return $this->active_adapter->extract_listing_data($post);
        }
        return array();
    }
    
    /**
     * Get available categories
     */
    public function get_available_categories() {
        if ($this->active_adapter) {
            return $this->active_adapter->get_available_categories();
        }
        return array();
    }
    
    /**
     * Get listing tags
     */
    public function get_listing_tags($post_id) {
        if ($this->active_adapter) {
            return $this->active_adapter->get_listing_tags($post_id);
        }
        return array();
    }
    
    /**
     * Get primary category
     */
    public function get_primary_category($listing_id) {
        if ($this->active_adapter) {
            return $this->active_adapter->get_primary_category($listing_id);
        }
        return null;
    }
    
    /**
     * Get category icon
     */
    public function get_category_icon($listing_id) {
        if ($this->active_adapter) {
            return $this->active_adapter->get_category_icon($listing_id);
        }
        return '';
    }
    
    /**
     * Get listing image
     */
    public function get_listing_image($post_id) {
        if ($this->active_adapter) {
            return $this->active_adapter->get_listing_image($post_id);
        }
        return '';
    }
    
    /**
     * Check if map is hidden
     */
    public function is_map_hidden($post_id) {
        if ($this->active_adapter) {
            return $this->active_adapter->is_map_hidden($post_id);
        }
        return false;
    }
    
    /**
     * Get category analytics
     */
    public function get_category_analytics($category, $post_type) {
        if ($this->active_adapter) {
            return $this->active_adapter->get_category_analytics($category, $post_type);
        }
        return array();
    }
}
