<?php
/**
 * 
 * @link              https://www.stefanofattori.it
 * @package           SF_Restrict_FSE_Site_Editor
 * 
 * @wordpress-plugin
 * 
 * Plugin Name: 	  SF Restrict FSE Site Editor
 * Plugin URI:        https://www.stefanofattori.it/wordpress/plugins/
 * Description: 	  Only allows specific administrators to access the site editor (FSE). Useful for developers, freelancers or web agencies when the client needs an administrator user, but without the possibility of damaging the website template.
 * Requires at least: 6.0
 * Requires PHP:      7.0
 * Version:			  1.0.1
 * Author:            Stefano Fattori <info@stefanofattori.it>
 * Author URI:		  https://www.stefanofattori.it
 * License:    	      GNU General Public License v3.0
 * License URI:	   	  http://www.gnu.org/licenses/gpl-3.0.html
 * Text Domain:       sf-restrict-fse-site-editor
 * Domain Path: 	  /languages
 * 
 */


if ( ! defined( 'ABSPATH' ) ) {
	exit; // Avoid direct access
}

/**
 * Set Plugin Activation Hook
 */
register_activation_hook( __FILE__, 'SFRFSE__activate_plugin' );


/**
 * Set Plugin Uninstall Hook that deletes plugin option from DB
 * @return void
 */
function SFRFSE_activate_plugin(): void {
	register_uninstall_hook( __FILE__, array( 'SFRFSE_Restrict_FSE_Site_Editor', 'uninstall_plugin' ) );
}

/**
 * SFRFSE_PLUGIN_BASENAME Constant
 */
define( 'SFRFSE_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

if ( ! class_exists( 'SFRFSE_Restrict_FSE_Site_Editor' ) ) {
	class SFRFSE_Restrict_FSE_Site_Editor {
		/**
		 * Instance of class
		 * @var $instance
		 */
		private static $instance = null;

		/**
		 * Name of group settings
		 * @var string
		 */
		private static $settings_group_name = 'restrict_fse_group';

		/**
		 * DB Option name
		 * @var string
		 */
		private static $option_name = 'restrict_fse_allowed_admins';

		/**
		 * __construct of class SFRFSE_Restrict_FSE_Site_Editor
		 */
		private function __construct() {
			add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
			add_action( 'admin_init', [ $this, 'register_settings' ] );
			add_action( 'admin_menu', [ $this, 'hide_site_editor_menu' ], 999 );
			add_action( 'admin_bar_menu', [ $this, 'hide_edit_site_button' ], 999 );

			add_filter( 'map_meta_cap', [ $this, 'restrict_site_editor_access' ], 10, 3 );
			add_filter( 'plugin_action_links', [ $this, 'restrict_plugin_actions' ], 10, 2 );
			add_filter( 'map_meta_cap', [ $this, 'restrict_plugin_management' ], 10, 4 );
		}

		/**
		 * Singleton method get_instance of the class
		 * @return SFRFSE_Restrict_FSE_Site_Editor
		 */
		public static function get_instance() {
			if ( self::$instance === null ) {
				self::$instance = new self();
			}
			return self::$instance;
		}



		/**
		 * Adds plugin page to WP Admin Menu
		 * @return void
		 */
		public function add_settings_page() {
			$allowed_admins = get_option( $this::$option_name, [] );
			if ( ! is_array( $allowed_admins ) ) {
				$allowed_admins = [];
			}
			if ( empty( $allowed_admins ) || in_array( get_current_user_id(), $allowed_admins ) ) {
				add_submenu_page(
					'options-general.php',
					__( 'Restrict FSE Site Editor', 'sf-restrict-fse-site-editor' ),
					__( 'Restrict FSE Site Editor', 'sf-restrict-fse-site-editor' ),
					'manage_options',
					'restrict-fse-site-editor',
					[ $this, 'settings_page_html' ]
				);
			}
		}

		/**
		 * Renders HTML Admin Page
		 * Callback function called from add_settings_page() > add_submenu_page()
		 * @return void
		 */
		public function settings_page_html() {
			$allowed_admins = get_option( $this::$option_name, [] );
			if ( ! is_array( $allowed_admins ) ) {
				$allowed_admins = [];
			}

			$current_user_id = get_current_user_id();
			if ( ! empty( $allowed_admins ) && ! in_array( $current_user_id, $allowed_admins ) ) {
				wp_die( esc_html__( 'You do not have permissions to access this page.', 'sf-restrict-fse-site-editor' ) );
			}

			$users = get_users( [ 'role' => 'administrator' ] );
			?>
			<div class="wrap">
				<h1><?php esc_html_e( 'Restrict FSE Site Editor', 'sf-restrict-fse-site-editor' ) ?></h1>
				<form method="post" action="options.php">
					<?php settings_fields( 'restrict_fse_group' ); ?>
					<table class="form-table">
						<tr>
							<th><label><?php esc_html_e( 'Select administrator users to enable the FSE Site Editor', 'sf-restrict-fse-site-editor' ) ?></label>
							</th>
							<td>
								<?php foreach ( $users as $user ) : ?>
									<label>
										<input type="checkbox" name="<?php echo esc_attr( $this::$option_name ); ?>[]"
											value="<?php echo esc_attr( $user->ID ); ?>" <?php checked( in_array( $user->ID, (array) $allowed_admins ) ); ?>>
										<?php echo esc_html( $user->display_name ); ?>
										<?php if ( $user->ID == $current_user_id ) : ?>
											<span
												style="font-size:12px; color: #000000; background:rgb(91, 183, 54); border-radius: 25px; padding: 2px 5px; font-weight: bold;"><?php esc_html_e( 'Current User', 'sf-restrict-fse-site-editor' ) ?></span>
										<?php endif; ?>
									</label><br>
								<?php endforeach; ?>
							</td>
						</tr>
					</table>
					<?php submit_button(); ?>
				</form>
			</div>
			<?php
		}


		/**
		 * Registers settings for admin page
		 * @return void
		 */
		public function register_settings() {
			register_setting(
					$this::$settings_group_name,
					$this::$option_name,
				array(
					'type' => 'array',
					'show_in_rest' => false,
					'sanitize_callback' => function ($array) {
						return map_deep( $array, 'sanitize_text_field' );
					}
				)
			);
		}


		/**
		 * Restrict FSE Site Editor access from unauthorized administrator users
		 * @param mixed $caps
		 * @param mixed $cap
		 * @param mixed $user_id
		 */
		public function restrict_site_editor_access( $caps, $cap, $user_id ) {
			$allowed_admins = get_option( $this::$option_name, [] );
			if ( ! is_array( $allowed_admins ) ) {
				$allowed_admins = [];
			}

			if ( $cap === 'edit_theme_options' && ! in_array( $user_id, $allowed_admins ) ) {
				return [ 'do_not_allow' ];
			}
			return $caps;
		}


		/**
		 * Hide FSE Site Editor menu link from unauthorized administrator users
		 * @return void
		 */
		public function hide_site_editor_menu() {
			$allowed_admins = get_option( $this::$option_name, [] );
			if ( ! is_array( $allowed_admins ) ) {
				$allowed_admins = [];
			}

			if ( ! in_array( get_current_user_id(), $allowed_admins ) ) {
				remove_submenu_page( 'themes.php', 'site-editor.php' );
			}
		}

		/**
		 * Hide FSE Site Editor button on WP Admin Bar from unauthorized administrator users
		 * @param mixed $wp_admin_bar
		 * @return void
		 */
		public function hide_edit_site_button( $wp_admin_bar ) {
			$allowed_admins = get_option( $this::$option_name, [] );
			if ( ! is_array( $allowed_admins ) ) {
				$allowed_admins = [];
			}

			if ( ! in_array( get_current_user_id(), $allowed_admins ) ) {
				$wp_admin_bar->remove_node( 'site-editor' );
			}
		}


		/**
		 * Removes 'deactivate' and 'delete' actions for this plugin on management plugin page from unauthorized administrator users
		 * @param mixed $actions
		 * @param mixed $plugin_file
		 */
		public function restrict_plugin_actions( $actions, $plugin_file ) {
			// Plugin to restrict
			$protected_plugin = SFRFSE_PLUGIN_BASENAME;

			if ( $plugin_file === $protected_plugin ) {
				$allowed_admins = get_option( $this::$option_name, [] );
				if ( ! is_array( $allowed_admins ) ) {
					$allowed_admins = [];
				}
				if ( empty( $allowed_admins ) || in_array( get_current_user_id(), $allowed_admins ) ) {
					return $actions;
				}
				unset( $actions['deactivate'] );
				unset( $actions['delete'] );

				// Custom notice for this plugin on management plugin page
				$actions['custom_notice'] = '<div style="color: red;">' . esc_html__( '⚠ Only enabled administrators can deactivate or delete this plugin', 'sf-restrict-fse-site-editor' ) . '</div>';

			}
			return $actions;
		}


		/**
		 * Deny to deactivate or delete this plugin directly for unauthorized administrator users
		 * @param mixed $caps
		 * @param mixed $cap
		 * @param mixed $user_id
		 * @param mixed $args
		 */
		public function restrict_plugin_management( $caps, $cap, $user_id, $args ) {
			// Plugin to restrict
			$protected_plugin = SFRFSE_PLUGIN_BASENAME;

			if ( isset( $args[0] ) && $args[0] === plugin_basename( $protected_plugin ) ) {
				$allowed_admins = get_option( $this::$option_name, [] );
				if ( ! is_array( $allowed_admins ) ) {
					$allowed_admins = [];
				}
				if ( ! in_array( $user_id, $allowed_admins ) && in_array( $cap, [ 'deactivate_plugin', 'delete_plugin' ] ) ) {
					return [ 'do_not_allow' ];
				}
			}
			return $caps;
		}


		/**
		 * Uninstall method that fire with uninstall hook
		 * Deletes plugin option from DB to mantain a clean DB
		 * @return void
		 */
		public static function uninstall_plugin() {
			unregister_setting( 'restrict_fse_group', SFRFSE_Restrict_FSE_Site_Editor::$option_name );
			delete_option( SFRFSE_Restrict_FSE_Site_Editor::$option_name );
		}
	}
}
/**
 * Get Instance of the class SFRFSE_Restrict_FSE_Site_Editor
 */
SFRFSE_Restrict_FSE_Site_Editor::get_instance();