<?php

namespace SEMFE;

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * REST API
 * Handles all REST endpoints
 */
class REST_API {
    /**
     * Export Manager
     */
    private $export_manager;

    /**
     * Import Manager
     */
    private $import_manager;

    /**
     * Network Manager (PRO only, multisite)
     */
    private $network_manager;

    /**
     * Constructor
     */
    public function __construct( $export_manager, $import_manager, $network_manager = null ) {
        $this->export_manager = $export_manager;
        $this->import_manager = $import_manager;
        $this->network_manager = $network_manager;
        add_action( 'rest_api_init', [$this, 'register_routes'] );
    }

    /**
     * Register REST routes
     */
    public function register_routes() {
        // Export endpoint
        register_rest_route( 'semfe/v1', '/export', [
            'methods'             => 'POST',
            'callback'            => [$this, 'export'],
            'permission_callback' => [$this, 'check_permissions'],
            'args'                => [
                'categories'  => [
                    'required' => false,
                    'type'     => 'array',
                    'default'  => [],
                ],
                'description' => [
                    'required'          => false,
                    'type'              => 'string',
                    'default'           => '',
                    'sanitize_callback' => 'sanitize_textarea_field',
                ],
            ],
        ] );
        // Import preview endpoint
        register_rest_route( 'semfe/v1', '/import-preview', [
            'methods'             => 'POST',
            'callback'            => [$this, 'import_preview'],
            'permission_callback' => [$this, 'check_permissions'],
        ] );
        // Import endpoint
        register_rest_route( 'semfe/v1', '/import', [
            'methods'             => 'POST',
            'callback'            => [$this, 'import'],
            'permission_callback' => [$this, 'check_permissions'],
            'args'                => [
                'data'       => [
                    'required' => true,
                ],
                'categories' => [
                    'required' => false,
                    'type'     => 'array',
                    'default'  => [],
                ],
                'strategy'   => [
                    'required' => false,
                    'type'     => 'string',
                    'default'  => 'merge',
                    'enum'     => ['merge', 'replace'],
                ],
            ],
        ] );
        // Network sites list endpoint (available for all multisite super admins).
        if ( is_multisite() ) {
            register_rest_route( 'semfe/v1', '/network/sites', [
                'methods'             => 'GET',
                'callback'            => [$this, 'get_network_sites'],
                'permission_callback' => [$this, 'check_network_sites_permissions'],
            ] );
        }
    }

    /**
     * Export endpoint handler
     */
    public function export( $request ) {
        $categories = $request->get_param( 'categories' );
        $description = $request->get_param( 'description' );
        try {
            $export_data = $this->export_manager->export( $categories, $description );
            return rest_ensure_response( [
                'success' => true,
                'data'    => $export_data,
            ] );
        } catch ( \Exception $e ) {
            return new \WP_Error('export_failed', $e->getMessage(), [
                'status' => 500,
            ]);
        }
    }

    /**
     * Import preview endpoint handler
     */
    public function import_preview( $request ) {
        $import_data = $request->get_json_params();
        try {
            $preview = $this->import_manager->preview( $import_data );
            return rest_ensure_response( [
                'success' => true,
                'preview' => $preview,
            ] );
        } catch ( \Exception $e ) {
            return new \WP_Error('preview_failed', $e->getMessage(), [
                'status' => 500,
            ]);
        }
    }

    /**
     * Import endpoint handler
     */
    public function import( $request ) {
        $import_data = $request->get_param( 'data' );
        $categories = $request->get_param( 'categories' );
        $strategy = $request->get_param( 'strategy' );
        try {
            $result = $this->import_manager->import( $import_data, $categories, $strategy );
            return rest_ensure_response( $result );
        } catch ( \Exception $e ) {
            return new \WP_Error('import_failed', $e->getMessage(), [
                'status' => 500,
            ]);
        }
    }

    /**
     * Check permissions
     */
    public function check_permissions() {
        return current_user_can( 'manage_options' );
    }

    /**
     * Check network sites permissions (for listing sites - no PRO check)
     * Available for all multisite super admins.
     *
     * @return bool|\WP_Error
     */
    public function check_network_sites_permissions() {
        if ( !is_multisite() ) {
            return new \WP_Error('not_multisite', __( 'This feature requires WordPress Multisite.', 'settings-manager-for-elementor' ), [
                'status' => 400,
            ]);
        }
        if ( !current_user_can( 'manage_network' ) ) {
            return new \WP_Error('forbidden', __( 'You do not have permission to view network sites.', 'settings-manager-for-elementor' ), [
                'status' => 403,
            ]);
        }
        return true;
    }

    /**
     * Check network permissions (PRO only, multisite)
     *
     * @return bool|\WP_Error
     */
    public function check_network_permissions() {
        if ( !is_multisite() ) {
            return new \WP_Error('not_multisite', __( 'This feature requires WordPress Multisite.', 'settings-manager-for-elementor' ), [
                'status' => 400,
            ]);
        }
        if ( !current_user_can( 'manage_network' ) ) {
            return new \WP_Error('forbidden', __( 'You do not have permission to perform network operations.', 'settings-manager-for-elementor' ), [
                'status' => 403,
            ]);
        }
        if ( !semfe_fs()->can_use_premium_code() ) {
            return new \WP_Error('pro_required', __( 'This feature requires a PRO license.', 'settings-manager-for-elementor' ), [
                'status' => 403,
            ]);
        }
        return true;
    }

    /**
     * Get network sites endpoint handler
     * Available for all multisite super admins (free and PRO).
     *
     * @return \WP_REST_Response|\WP_Error
     */
    public function get_network_sites() {
        // Use Network_Manager if available (PRO), otherwise use inline implementation.
        if ( $this->network_manager ) {
            $sites = $this->network_manager->get_network_sites();
        } else {
            $sites = $this->get_network_sites_inline();
        }
        return rest_ensure_response( [
            'success' => true,
            'sites'   => $sites,
        ] );
    }

    /**
     * Get network sites (inline implementation for free users)
     * Uses single blog switch per site for performance.
     *
     * @return array
     */
    private function get_network_sites_inline() {
        $current_blog_id = get_current_blog_id();
        $sites = get_sites( [
            'number'   => 100,
            'deleted'  => 0,
            'archived' => 0,
            'spam'     => 0,
        ] );
        $result = [];
        foreach ( $sites as $site ) {
            // Skip current site.
            if ( (int) $site->blog_id === $current_blog_id ) {
                continue;
            }
            // Get name and Elementor status in a single switch.
            switch_to_blog( $site->blog_id );
            $name = get_bloginfo( 'name' );
            if ( empty( $name ) ) {
                /* translators: %d is replaced with the site id */
                $name = sprintf( __( 'Site %d', 'settings-manager-for-elementor' ), $site->blog_id );
            }
            $elementor_active = false;
            $active_plugins = get_option( 'active_plugins', [] );
            foreach ( $active_plugins as $plugin ) {
                if ( strpos( $plugin, 'elementor.php' ) !== false ) {
                    $elementor_active = true;
                    break;
                }
            }
            // Also check network-activated plugins.
            if ( !$elementor_active ) {
                $network_plugins = get_site_option( 'active_sitewide_plugins', [] );
                foreach ( array_keys( $network_plugins ) as $plugin ) {
                    if ( strpos( $plugin, 'elementor.php' ) !== false ) {
                        $elementor_active = true;
                        break;
                    }
                }
            }
            restore_current_blog();
            $result[] = [
                'id'               => (int) $site->blog_id,
                'name'             => $name,
                'url'              => $site->siteurl,
                'elementor_active' => $elementor_active,
            ];
        }
        return $result;
    }

    /**
     * Import to network site endpoint handler (PRO only)
     *
     * @param \WP_REST_Request $request Request object.
     * @return \WP_REST_Response|\WP_Error
     */
    public function network_import_to_site( $request ) {
        if ( !$this->network_manager ) {
            return new \WP_Error('not_available', __( 'Network manager not available.', 'settings-manager-for-elementor' ), [
                'status' => 500,
            ]);
        }
        $import_data = $request->get_param( 'data' );
        $categories = $request->get_param( 'categories' );
        $strategy = $request->get_param( 'strategy' );
        $site_id = $request->get_param( 'site_id' );
        $result = $this->network_manager->import_to_site(
            $import_data,
            $categories,
            $strategy,
            $site_id
        );
        if ( !$result['success'] ) {
            return new \WP_Error('import_failed', $result['error'], [
                'status' => 500,
            ]);
        }
        return rest_ensure_response( $result );
    }

}
