<?php

namespace SEMFE;

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Import Manager
 * Handles importing settings by category
 */
class Import_Manager {
    /**
     * Category Manager
     */
    private $category_manager;

    /**
     * Constructor
     */
    public function __construct( $category_manager ) {
        $this->category_manager = $category_manager;
    }

    /**
     * Preview import data
     *
     * @param array $import_data
     * @return array
     */
    public function preview( $import_data ) {
        if ( !$this->validate_import_data( $import_data ) ) {
            return [
                'valid' => false,
                'error' => esc_html__( 'Invalid import data format', 'settings-manager-for-elementor' ),
            ];
        }
        $preview = [
            'valid'             => true,
            'version'           => $import_data['version'] ?? 'unknown',
            'plugin_version'    => $import_data['plugin_version'] ?? 'unknown',
            'export_date'       => $import_data['export_date'] ?? 'unknown',
            'site_url'          => $import_data['site_url'] ?? 'unknown',
            'elementor_version' => $import_data['elementor_version'] ?? 'unknown',
            'theme_name'        => $import_data['theme_name'] ?? 'unknown',
            'theme_version'     => $import_data['theme_version'] ?? 'unknown',
            'description'       => $import_data['description'] ?? '',
            'categories'        => $import_data['categories'] ?? [],
            'counts'            => [],
        ];
        // Count items in each category
        foreach ( $import_data['categories'] as $category_id ) {
            if ( isset( $import_data['data'][$category_id] ) ) {
                $preview['counts'][$category_id] = $this->count_category_items( $category_id, $import_data['data'][$category_id] );
            }
        }
        $preview['total_items'] = array_sum( $preview['counts'] );
        return $preview;
    }

    /**
     * Import settings with selected categories
     *
     * @param array $import_data
     * @param array $selected_categories
     * @param string $strategy 'merge' or 'replace'
     * @return array
     */
    public function import( $import_data, $selected_categories = [], $strategy = 'merge' ) {
        if ( !$this->validate_import_data( $import_data ) ) {
            return [
                'success' => false,
                'error'   => esc_html__( 'Invalid import data', 'settings-manager-for-elementor' ),
            ];
        }
        // Validate categories
        $selected_categories = $this->category_manager->validate_categories( $selected_categories );
        // Create backup before any modifications
        Backup_Manager::get_instance()->create_backup();
        $results = [
            'success'  => true,
            'imported' => [],
            'skipped'  => [],
            'errors'   => [],
        ];
        // Import each selected category
        foreach ( $selected_categories as $category_id ) {
            if ( !isset( $import_data['data'][$category_id] ) ) {
                $results['skipped'][] = $category_id;
                continue;
            }
            try {
                $this->import_category( $category_id, $import_data['data'][$category_id], $strategy );
                $results['imported'][] = $category_id;
            } catch ( \Exception $e ) {
                $results['errors'][$category_id] = $e->getMessage();
            }
        }
        // Clear cache after import
        \Elementor\Plugin::$instance->files_manager->clear_cache();
        return $results;
    }

    /**
     * Import specific category
     *
     * @param string $category_id
     * @param mixed $data
     * @param string $strategy
     */
    private function import_category( $category_id, $data, $strategy ) {
        $method = 'import_' . $category_id;
        if ( method_exists( $this, $method ) ) {
            $this->{$method}( $data, $strategy );
        }
    }

    /**
     * Import Global Colors
     */
    private function import_global_colors( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $existing_settings = $kit->get_settings();
        if ( $strategy === 'replace' ) {
            // Delete autosave to prevent stale cached data from being displayed
            // Colors endpoint uses get_settings_for_display() which caches data from autosave
            $autosave = $kit->get_autosave();
            if ( $autosave ) {
                wp_delete_post_revision( $autosave->get_id() );
            }
            // For replace strategy, use Page Settings Manager directly
            // update_settings() uses array_replace_recursive() which merges numeric arrays
            $existing_settings['system_colors'] = $data['system'] ?? [];
            $existing_settings['custom_colors'] = $data['custom'] ?? [];
            $page_settings_manager = \Elementor\Core\Settings\Manager::get_settings_managers( 'page' );
            $page_settings_manager->save_settings( $existing_settings, $kit->get_post()->ID );
            // Force document cache refresh - clears parsed_active_settings object cache
            // Without this, get_settings_for_display() returns stale cached data
            \Elementor\Plugin::$instance->documents->get( $kit->get_post()->ID, false );
        } else {
            // Merge strategy
            $settings = [];
            $settings['system_colors'] = $this->merge_colors( $existing_settings['system_colors'] ?? [], $data['system'] ?? [] );
            $settings['custom_colors'] = $this->merge_colors( $existing_settings['custom_colors'] ?? [], $data['custom'] ?? [] );
            $kit->update_settings( $settings );
        }
    }

    /**
     * Import Global Fonts
     */
    private function import_global_typography( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $existing_settings = $kit->get_settings();
        if ( $strategy === 'replace' ) {
            // Delete autosave to prevent stale cached data from being displayed
            $autosave = $kit->get_autosave();
            if ( $autosave ) {
                wp_delete_post_revision( $autosave->get_id() );
            }
            // For replace strategy, use Page Settings Manager directly
            // update_settings() uses array_replace_recursive() which merges numeric arrays
            $existing_settings['system_typography'] = $data['system'] ?? [];
            $existing_settings['custom_typography'] = $data['custom'] ?? [];
            $existing_settings['default_generic_fonts'] = $data['default_generic_fonts'] ?? 'Sans-serif';
            $page_settings_manager = \Elementor\Core\Settings\Manager::get_settings_managers( 'page' );
            $page_settings_manager->save_settings( $existing_settings, $kit->get_post()->ID );
            // Force document cache refresh - clears cached object instances
            \Elementor\Plugin::$instance->documents->get( $kit->get_post()->ID, false );
        } else {
            // Merge strategy
            $settings = [];
            $settings['system_typography'] = $this->merge_typography( $existing_settings['system_typography'] ?? [], $data['system'] ?? [] );
            $settings['custom_typography'] = $this->merge_typography( $existing_settings['custom_typography'] ?? [], $data['custom'] ?? [] );
            $settings['default_generic_fonts'] = $data['default_generic_fonts'] ?? 'Sans-serif';
            $kit->update_settings( $settings );
        }
    }

    /**
     * Import Theme Typography
     */
    private function import_theme_typography( $data, $strategy ) {
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Theme Buttons
     */
    private function import_theme_buttons( $data, $strategy ) {
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Theme Images
     */
    private function import_theme_images( $data, $strategy ) {
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Theme Form Fields
     */
    private function import_theme_form_fields( $data, $strategy ) {
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Layout Settings
     */
    private function import_layout_settings( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Breakpoints
     */
    private function import_breakpoints( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Site Identity (Name & Description)
     */
    private function import_site_identity_text( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Site Identity (Logo & Favicon)
     */
    private function import_site_identity_media( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Background Style (Color & Style)
     */
    private function import_background_style( $data, $strategy ) {
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Background Image
     */
    private function import_background_image( $data, $strategy ) {
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Lightbox Settings
     */
    private function import_lightbox_settings( $data, $strategy ) {
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Page Transitions
     */
    private function import_page_transitions( $data, $strategy ) {
    }

    /**
     * Import Preloader Settings (non-media)
     */
    private function import_preloader_settings( $data, $strategy ) {
    }

    /**
     * Import Preloader Media (Icon & Image)
     */
    private function import_preloader_media( $data, $strategy ) {
    }

    /**
     * Import Custom CSS
     */
    private function import_custom_css( $data, $strategy ) {
    }

    /**
     * Import Hello Theme Header Style
     */
    private function import_hello_header_style( $data, $strategy ) {
        // Only import if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return;
        }
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Hello Theme Header Background Image
     */
    private function import_hello_header_image( $data, $strategy ) {
        // Only import if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return;
        }
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Hello Theme Footer Style
     */
    private function import_hello_footer_style( $data, $strategy ) {
        // Only import if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return;
        }
        $data = $this->extract_and_import_globals( $data );
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Hello Theme Footer Background Image
     */
    private function import_hello_footer_image( $data, $strategy ) {
        // Only import if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return;
        }
        $kit = $this->get_active_kit();
        $kit->update_settings( $data );
    }

    /**
     * Import Hello Theme Backend Settings
     *
     * Imports Hello theme settings to admin.php?page=hello-elementor-settings
     *
     * @param array $data Settings data
     * @param string $strategy Import strategy (merge/replace) - not used for simple options
     */
    private function import_hello_backend( $data, $strategy ) {
        // Only import if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return;
        }
        // Map of export keys to WordPress option names
        $options_map = [
            'description_meta_tag' => 'hello_elementor_settings_description_meta_tag',
            'skip_link'            => 'hello_elementor_settings_skip_link',
            'header_footer'        => 'hello_elementor_settings_header_footer',
            'page_title'           => 'hello_elementor_settings_page_title',
            'hello_style'          => 'hello_elementor_settings_hello_style',
            'hello_theme'          => 'hello_elementor_settings_hello_theme',
        ];
        foreach ( $options_map as $key => $option_name ) {
            if ( isset( $data[$key] ) ) {
                update_option( $option_name, $data[$key] );
            }
        }
    }

    /**
     * Import Backend General Settings
     */
    private function import_backend_general( $data, $strategy ) {
        if ( isset( $data['cpt_support'] ) ) {
            update_option( 'elementor_cpt_support', $data['cpt_support'] );
        }
        if ( isset( $data['disable_color_schemes'] ) ) {
            update_option( 'elementor_disable_color_schemes', $data['disable_color_schemes'] );
        }
        if ( isset( $data['disable_typography_schemes'] ) ) {
            update_option( 'elementor_disable_typography_schemes', $data['disable_typography_schemes'] );
        }
        if ( isset( $data['allow_tracking'] ) ) {
            update_option( 'elementor_allow_tracking', $data['allow_tracking'] );
        }
    }

    /**
     * Import Backend Integrations
     */
    private function import_backend_integrations( $data, $strategy ) {
        // Map export keys to WordPress option names
        $option_map = [
            'google_maps_api_key'     => 'elementor_google_maps_api_key',
            'recaptcha_site_key'      => 'elementor_pro_recaptcha_site_key',
            'recaptcha_secret_key'    => 'elementor_pro_recaptcha_secret_key',
            'recaptcha_threshold'     => 'elementor_pro_recaptcha_threshold',
            'recaptcha_v3_site_key'   => 'elementor_pro_recaptcha_v3_site_key',
            'recaptcha_v3_secret_key' => 'elementor_pro_recaptcha_v3_secret_key',
            'recaptcha_v3_threshold'  => 'elementor_pro_recaptcha_v3_threshold',
            'facebook_app_id'         => 'elementor_pro_facebook_app_id',
            'typekit_kit_id'          => 'elementor_typekit-kit-id',
            'font_awesome_pro_kit_id' => 'elementor_font_awesome_pro_kit_id',
            'mailchimp_api_key'       => 'elementor_pro_mailchimp_api_key',
            'getresponse_api_key'     => 'elementor_pro_getresponse_api_key',
            'drip_api_token'          => 'elementor_pro_drip_api_token',
            'convertkit_api_key'      => 'elementor_pro_convertkit_api_key',
            'activecampaign_api_key'  => 'elementor_pro_activecampaign_api_key',
            'activecampaign_api_url'  => 'elementor_pro_activecampaign_api_url',
            'mailerlite_api_key'      => 'elementor_pro_mailerlite_api_key',
        ];
        // Import each integration setting
        foreach ( $data as $key => $value ) {
            if ( isset( $option_map[$key] ) ) {
                update_option( $option_map[$key], $value );
            }
        }
    }

    /**
     * Import Backend Advanced Settings
     */
    private function import_backend_advanced( $data, $strategy ) {
        foreach ( $data as $key => $value ) {
            update_option( 'elementor_' . $key, $value );
        }
    }

    /**
     * Import Backend Performance Settings
     */
    private function import_backend_performance( $data, $strategy ) {
        foreach ( $data as $key => $value ) {
            update_option( 'elementor_' . $key, $value );
        }
    }

    /**
     * Import Experiments
     */
    private function import_experiments( $data, $strategy ) {
        foreach ( $data as $feature_name => $feature_data ) {
            $option_key = 'elementor_experiment-' . $feature_name;
            $state = $feature_data['state'] ?? 'default';
            update_option( $option_key, $state );
        }
    }

    /**
     * Validate import data structure
     */
    private function validate_import_data( $data ) {
        return isset( $data['data'] ) && isset( $data['categories'] );
    }

    /**
     * Count items in category data
     */
    private function count_category_items( $category_id, $data ) {
        if ( $category_id === 'global_colors' ) {
            return count( $data['system'] ?? [] ) + count( $data['custom'] ?? [] );
        }
        if ( $category_id === 'global_typography' ) {
            return count( $data['system'] ?? [] ) + count( $data['custom'] ?? [] );
        }
        if ( is_array( $data ) ) {
            return count( $data );
        }
        return 1;
    }

    /**
     * Merge colors (merge by _id)
     */
    private function merge_colors( $existing, $new ) {
        $merged = $existing;
        foreach ( $new as $new_item ) {
            $found = false;
            foreach ( $merged as &$existing_item ) {
                if ( $existing_item['_id'] === $new_item['_id'] ) {
                    $existing_item = $new_item;
                    $found = true;
                    break;
                }
            }
            if ( !$found ) {
                $merged[] = $new_item;
            }
        }
        return $merged;
    }

    /**
     * Merge typography (merge by _id)
     */
    private function merge_typography( $existing, $new ) {
        return $this->merge_colors( $existing, $new );
        // Same logic
    }

    /**
     * Helper: Get active kit
     */
    private function get_active_kit() {
        return \Elementor\Plugin::$instance->kits_manager->get_active_kit();
    }

    /**
     * Helper: Import __globals__ entries
     *
     * Merges imported __globals__ entries with existing ones.
     * This preserves global color/typography links for the imported settings.
     *
     * @param array $globals The __globals__ entries to import
     */
    private function import_globals( $globals ) {
        if ( empty( $globals ) || !is_array( $globals ) ) {
            return;
        }
        $kit = $this->get_active_kit();
        $existing_settings = $kit->get_settings();
        $existing_globals = $existing_settings['__globals__'] ?? [];
        // Merge: imported globals override existing ones for the same keys
        $merged_globals = array_merge( $existing_globals, $globals );
        $kit->update_settings( [
            '__globals__' => $merged_globals,
        ] );
    }

    /**
     * Helper: Extract __globals__ from category data and import remaining settings
     *
     * @param array $data Category data that may contain __globals__ key
     * @return array Data without __globals__ key (for normal import)
     */
    private function extract_and_import_globals( $data ) {
        if ( isset( $data['__globals__'] ) ) {
            $this->import_globals( $data['__globals__'] );
            unset($data['__globals__']);
        }
        return $data;
    }

    /**
     * Check if Hello Elementor theme is active
     *
     * @return bool
     */
    private function is_hello_theme_active() {
        $theme = wp_get_theme();
        return 'Hello Elementor' === $theme->name || 'Hello Elementor' === $theme->parent_theme;
    }

}
