<?php

namespace SEMFE;

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Export Manager
 * Handles exporting settings by category
 */
class Export_Manager {
    /**
     * Category Manager
     */
    private $category_manager;

    /**
     * Constructor
     */
    public function __construct( $category_manager ) {
        $this->category_manager = $category_manager;
    }

    /**
     * Export complete settings with selected categories
     *
     * @param array  $selected_categories Categories to export.
     * @param string $description         Optional description for the export.
     * @return array
     */
    public function export( $selected_categories = [], $description = '' ) {
        // If no categories selected, use defaults
        if ( empty( $selected_categories ) ) {
            $selected_categories = $this->category_manager->get_default_selected();
        }
        // Validate categories
        $selected_categories = $this->category_manager->validate_categories( $selected_categories );
        // Get theme information
        $theme = wp_get_theme();
        $export_data = [
            'export_date'       => current_time( 'mysql' ),
            'export_timestamp'  => current_time( 'Ymd-Hi' ),
            'site_url'          => get_site_url(),
            'plugin_version'    => SEMFE_VERSION,
            'elementor_version' => ELEMENTOR_VERSION,
            'theme_name'        => $theme->get( 'Name' ),
            'theme_version'     => $theme->get( 'Version' ),
            'description'       => $description,
            'categories'        => $selected_categories,
            'data'              => [],
        ];
        // Export each selected category
        foreach ( $selected_categories as $category_id ) {
            $export_data['data'][$category_id] = $this->export_category( $category_id );
        }
        return $export_data;
    }

    /**
     * Export specific category
     *
     * @param string $category_id
     * @return mixed
     */
    private function export_category( $category_id ) {
        $method = 'export_' . $category_id;
        if ( method_exists( $this, $method ) ) {
            return $this->{$method}();
        }
        return null;
    }

    /**
     * Export Global Colors
     */
    private function export_global_colors() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        return [
            'system' => $settings['system_colors'] ?? [],
            'custom' => $settings['custom_colors'] ?? [],
        ];
    }

    /**
     * Export Global Fonts
     */
    private function export_global_typography() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        return [
            'system'                => $settings['system_typography'] ?? [],
            'custom'                => $settings['custom_typography'] ?? [],
            'default_generic_fonts' => $settings['default_generic_fonts'] ?? 'Sans-serif',
        ];
    }

    /**
     * Export Theme Typography
     */
    private function export_theme_typography() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Build prefixes array for typography controls
        $prefixes = [
            // Body Typography - All typography properties and color
            'body_typography_',
            'body_color',
            // Paragraph Spacing - All responsive variants
            'paragraph_spacing',
            // Link Typography - All typography properties and colors
            'link_normal_typography_',
            'link_normal_color',
            'link_hover_color',
            'link_hover_typography_',
        ];
        // Add heading prefixes (H1-H6) - Typography properties and color
        for ($i = 1; $i <= 6; $i++) {
            $prefixes[] = "h{$i}_typography_";
            $prefixes[] = "h{$i}_color";
        }
        $data = $this->extract_by_prefix( $settings, $prefixes );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, $prefixes );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Theme Buttons
     *
     * Exports all button styling controls including:
     * - Typography (font family, size, weight, spacing, etc.)
     * - Text shadow and colors
     * - Background (classic and gradient) for normal and hover states
     * - Border styling with responsive variants
     * - Box shadow for normal and hover states
     * - Padding with responsive variants
     */
    private function export_theme_buttons() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract all button-related controls
        $data = $this->extract_by_prefix( $settings, 'button_' );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, 'button_' );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Theme Images
     *
     * Exports all image styling controls including:
     * - Border styling with responsive variants for normal and hover states
     * - Border radius with responsive variants for normal and hover states
     * - Opacity for normal and hover states
     * - Box shadow for normal and hover states
     * - CSS filters (blur, brightness, contrast, saturate, hue) for normal and hover states
     * - Hover transition duration
     */
    private function export_theme_images() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract all image-related controls
        $data = $this->extract_by_prefix( $settings, 'image_' );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, 'image_' );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Theme Form Fields
     *
     * Exports all form field styling controls including:
     * - Label typography with responsive variants
     * - Field typography with responsive variants
     * - Colors for normal and focus states
     * - Border styling with responsive variants for normal and focus states
     * - Box shadow for normal and focus states
     * - Padding with responsive variants
     * - Focus transition duration
     */
    private function export_theme_form_fields() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract all form-related controls
        $data = $this->extract_by_prefix( $settings, 'form_' );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, 'form_' );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Site Identity (Name & Description)
     */
    private function export_site_identity_text() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        $controls = ['site_name', 'site_description'];
        return $this->extract_settings( $settings, $controls );
    }

    /**
     * Export Site Identity (Logo & Favicon)
     */
    private function export_site_identity_media() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        $controls = ['site_logo', 'site_favicon'];
        return $this->extract_settings( $settings, $controls );
    }

    /**
     * Export Background Style (Color & Style)
     *
     * Exports all body background controls except images:
     * - Background type selector (classic/gradient)
     * - Classic background (color, position, attachment, repeat, size) with responsive variants
     * - Gradient background (colors, stops, angle, position) with responsive variants
     * - Mobile browser background color
     * - Body overscroll behavior
     */
    private function export_background_style() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract non-image background controls
        $prefixes = ['body_background_', 'mobile_browser_background', 'body_overscroll_behavior'];
        // Image-related keys to exclude
        $exclude_prefixes = ['body_background_image'];
        $data = $this->extract_by_prefix(
            $settings,
            $prefixes,
            [],
            $exclude_prefixes
        );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, $prefixes );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Background Image
     *
     * Exports body background image settings with responsive variants.
     */
    private function export_background_image() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract only image-related background controls
        $data = $this->extract_by_prefix( $settings, 'body_background_image' );
        return $data;
    }

    /**
     * Export Layout Settings
     *
     * Uses prefix-based extraction to automatically capture new responsive variants
     * or related settings that Elementor may add in future versions.
     */
    private function export_layout_settings() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        $prefixes = [
            'container_width',
            'container_padding',
            'space_between_widgets',
            'page_title_selector',
            'stretched_section_container',
            'default_page_template'
        ];
        return $this->extract_by_prefix( $settings, $prefixes );
    }

    /**
     * Export Breakpoints
     *
     * Uses prefix-based extraction to automatically capture new breakpoint
     * settings that Elementor may add in future versions.
     */
    private function export_breakpoints() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        $prefixes = ['active_breakpoints', 'viewport_'];
        return $this->extract_by_prefix( $settings, $prefixes );
    }

    /**
     * Export Lightbox Settings
     *
     * Exports all lightbox controls including:
     * - Global lightbox enable/disable
     * - Behavior settings (counter, fullscreen, zoom, share)
     * - Content settings (title source, description source)
     * - Style settings (colors, icon sizes)
     */
    private function export_lightbox_settings() {
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract all lightbox-related controls
        $prefixes = ['lightbox_', 'global_image_lightbox'];
        $data = $this->extract_by_prefix( $settings, $prefixes );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, $prefixes );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Page Transitions
     *
     * Uses prefix-based extraction to automatically capture new page transition
     * settings that Elementor may add in future versions.
     * Excludes preloader settings which are exported separately.
     */
    private function export_page_transitions() {
        return [];
    }

    /**
     * Export Preloader Settings (non-media)
     *
     * Uses prefix-based extraction to automatically capture preloader
     * settings except icon and image.
     */
    private function export_preloader_settings() {
        return [];
    }

    /**
     * Export Preloader Media (Icon & Image)
     *
     * Exports preloader icon and image settings.
     */
    private function export_preloader_media() {
        return [];
    }

    /**
     * Export Custom CSS
     */
    private function export_custom_css() {
        return '';
    }

    /**
     * Export Hello Theme Header Style (non-image)
     */
    private function export_hello_header_style() {
        // Only export if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return [];
        }
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Exclude background image settings
        $exclude_prefixes = ['hello_header_background_image'];
        $data = $this->extract_by_prefix(
            $settings,
            'hello_header_',
            [],
            $exclude_prefixes
        );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, 'hello_header_' );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Hello Theme Header Background Image
     */
    private function export_hello_header_image() {
        // Only export if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return [];
        }
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract only background image settings
        $data = $this->extract_by_prefix( $settings, 'hello_header_background_image' );
        return $data;
    }

    /**
     * Export Hello Theme Footer Style (non-image)
     */
    private function export_hello_footer_style() {
        // Only export if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return [];
        }
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Exclude background image settings
        $exclude_prefixes = ['hello_footer_background_image'];
        $data = $this->extract_by_prefix(
            $settings,
            'hello_footer_',
            [],
            $exclude_prefixes
        );
        // Include relevant __globals__ entries
        $globals = $this->extract_globals_by_prefix( $settings, 'hello_footer_' );
        if ( !empty( $globals ) ) {
            $data['__globals__'] = $globals;
        }
        return $data;
    }

    /**
     * Export Hello Theme Footer Background Image
     */
    private function export_hello_footer_image() {
        // Only export if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return [];
        }
        $kit = $this->get_active_kit();
        $settings = $kit->get_settings();
        // Extract only background image settings
        $data = $this->extract_by_prefix( $settings, 'hello_footer_background_image' );
        return $data;
    }

    /**
     * Export Hello Theme Backend Settings
     *
     * Exports Hello theme settings from admin.php?page=hello-elementor-settings
     * These control meta tag, skip link, header/footer, page title, and theme styles.
     *
     * @return array Settings data
     */
    private function export_hello_backend() {
        // Only export if Hello theme is active
        if ( !$this->is_hello_theme_active() ) {
            return [];
        }
        return [
            'description_meta_tag' => get_option( 'hello_elementor_settings_description_meta_tag', '' ),
            'skip_link'            => get_option( 'hello_elementor_settings_skip_link', '' ),
            'header_footer'        => get_option( 'hello_elementor_settings_header_footer', '' ),
            'page_title'           => get_option( 'hello_elementor_settings_page_title', '' ),
            'hello_style'          => get_option( 'hello_elementor_settings_hello_style', '' ),
            'hello_theme'          => get_option( 'hello_elementor_settings_hello_theme', '' ),
        ];
    }

    /**
     * Export Backend General Settings
     */
    private function export_backend_general() {
        return [
            'cpt_support'                => get_option( 'elementor_cpt_support', [] ),
            'disable_color_schemes'      => get_option( 'elementor_disable_color_schemes', '' ),
            'disable_typography_schemes' => get_option( 'elementor_disable_typography_schemes', '' ),
            'allow_tracking'             => get_option( 'elementor_allow_tracking', '' ),
        ];
    }

    /**
     * Export Backend Integrations
     */
    private function export_backend_integrations() {
        $integrations = [
            'google_maps_api_key'     => get_option( 'elementor_google_maps_api_key', '' ),
            'recaptcha_site_key'      => get_option( 'elementor_pro_recaptcha_site_key', '' ),
            'recaptcha_secret_key'    => get_option( 'elementor_pro_recaptcha_secret_key', '' ),
            'recaptcha_threshold'     => get_option( 'elementor_pro_recaptcha_threshold', '' ),
            'recaptcha_v3_site_key'   => get_option( 'elementor_pro_recaptcha_v3_site_key', '' ),
            'recaptcha_v3_secret_key' => get_option( 'elementor_pro_recaptcha_v3_secret_key', '' ),
            'recaptcha_v3_threshold'  => get_option( 'elementor_pro_recaptcha_v3_threshold', '' ),
            'facebook_app_id'         => get_option( 'elementor_pro_facebook_app_id', '' ),
            'typekit_kit_id'          => get_option( 'elementor_typekit-kit-id', '' ),
            'font_awesome_pro_kit_id' => get_option( 'elementor_font_awesome_pro_kit_id', '' ),
            'mailchimp_api_key'       => get_option( 'elementor_pro_mailchimp_api_key', '' ),
            'getresponse_api_key'     => get_option( 'elementor_pro_getresponse_api_key', '' ),
            'drip_api_token'          => get_option( 'elementor_pro_drip_api_token', '' ),
            'convertkit_api_key'      => get_option( 'elementor_pro_convertkit_api_key', '' ),
            'activecampaign_api_key'  => get_option( 'elementor_pro_activecampaign_api_key', '' ),
            'activecampaign_api_url'  => get_option( 'elementor_pro_activecampaign_api_url', '' ),
            'mailerlite_api_key'      => get_option( 'elementor_pro_mailerlite_api_key', '' ),
        ];
        // Only export configured integrations (non-empty values)
        return array_filter( $integrations, function ( $value ) {
            return !empty( $value );
        } );
    }

    /**
     * Export Backend Advanced Settings
     */
    private function export_backend_advanced() {
        return [
            'editor_break_lines'      => get_option( 'elementor_editor_break_lines', '' ),
            'unfiltered_files_upload' => get_option( 'elementor_unfiltered_files_upload', '' ),
            'google_font'             => get_option( 'elementor_google_font', '1' ),
            'font_display'            => get_option( 'elementor_font_display', 'auto' ),
            'load_fa4_shim'           => get_option( 'elementor_load_fa4_shim', '' ),
            'meta_generator_tag'      => get_option( 'elementor_meta_generator_tag', '' ),
            'form-submissions'        => get_option( 'elementor_form-submissions', '' ),
        ];
    }

    /**
     * Export Backend Performance Settings
     */
    private function export_backend_performance() {
        return [
            'css_print_method'            => get_option( 'elementor_css_print_method', 'external' ),
            'optimized_image_loading'     => get_option( 'elementor_optimized_image_loading', '' ),
            'optimized_gutenberg_loading' => get_option( 'elementor_optimized_gutenberg_loading', '' ),
            'lazy_load_background_images' => get_option( 'elementor_lazy_load_background_images', '' ),
            'local_google_fonts'          => get_option( 'elementor_local_google_fonts', '' ),
            'element_cache_ttl'           => get_option( 'elementor_element_cache_ttl', '24' ),
        ];
    }

    /**
     * Export Experiments
     */
    private function export_experiments() {
        $experiments_manager = \Elementor\Plugin::$instance->experiments;
        $features = $experiments_manager->get_features();
        $experiments = [];
        foreach ( $features as $feature_name => $feature_data ) {
            $experiments[$feature_name] = [
                'name'           => $feature_name,
                'state'          => $feature_data['state'] ?? 'default',
                'default'        => $feature_data['default'] ?? 'inactive',
                'title'          => $feature_data['title'] ?? $feature_name,
                'release_status' => $feature_data['release_status'] ?? 'alpha',
            ];
        }
        return $experiments;
    }

    /**
     * Helper: Get active kit
     */
    private function get_active_kit() {
        return \Elementor\Plugin::$instance->kits_manager->get_active_kit();
    }

    /**
     * Helper: Extract __globals__ entries matching given prefixes
     *
     * The __globals__ array stores references to global colors/typography.
     * For example: "__globals__" => ["body_background_color" => "globals/colors?id=text"]
     * This tells Elementor to use the "text" global color for body_background_color.
     *
     * @param array $settings All kit settings (must include __globals__ key)
     * @param string|array $prefixes Single prefix or array of prefixes to match
     * @return array Matching __globals__ entries
     */
    private function extract_globals_by_prefix( $settings, $prefixes ) {
        if ( empty( $settings['__globals__'] ) || !is_array( $settings['__globals__'] ) ) {
            return [];
        }
        // Normalize prefixes to array
        if ( !is_array( $prefixes ) ) {
            $prefixes = [$prefixes];
        }
        $extracted = [];
        foreach ( $settings['__globals__'] as $key => $value ) {
            // Skip empty values
            if ( empty( $value ) ) {
                continue;
            }
            // Check if key matches any prefix
            foreach ( $prefixes as $prefix ) {
                if ( strpos( $key, $prefix ) === 0 ) {
                    $extracted[$key] = $value;
                    break;
                }
            }
        }
        return $extracted;
    }

    /**
     * Helper: Extract __globals__ entries for exact keys
     *
     * @param array $settings All kit settings
     * @param array $keys Exact keys to extract
     * @return array Matching __globals__ entries
     */
    private function extract_globals_by_keys( $settings, $keys ) {
        if ( empty( $settings['__globals__'] ) || !is_array( $settings['__globals__'] ) ) {
            return [];
        }
        $extracted = [];
        foreach ( $keys as $key ) {
            if ( isset( $settings['__globals__'][$key] ) && !empty( $settings['__globals__'][$key] ) ) {
                $extracted[$key] = $settings['__globals__'][$key];
            }
        }
        return $extracted;
    }

    /**
     * Helper: Extract specific settings from array
     */
    private function extract_settings( $settings, $keys ) {
        $result = [];
        foreach ( $keys as $key ) {
            if ( isset( $settings[$key] ) ) {
                $result[$key] = $settings[$key];
            }
        }
        return $result;
    }

    /**
     * Helper: Extract settings by prefix with optional exclusions
     *
     * @param array $settings All kit settings
     * @param string|array $prefixes Single prefix or array of prefixes to match
     * @param array $exclude Optional array of exact keys to exclude
     * @param array $exclude_prefixes Optional array of prefixes to exclude
     * @return array Filtered settings
     */
    private function extract_by_prefix(
        $settings,
        $prefixes,
        $exclude = [],
        $exclude_prefixes = []
    ) {
        // Normalize prefixes to array
        if ( !is_array( $prefixes ) ) {
            $prefixes = [$prefixes];
        }
        $extracted = [];
        foreach ( $settings as $key => $value ) {
            // Check if key matches any prefix
            $matches = false;
            foreach ( $prefixes as $prefix ) {
                if ( strpos( $key, $prefix ) === 0 ) {
                    $matches = true;
                    break;
                }
            }
            if ( !$matches ) {
                continue;
            }
            // Check if key matches any exclude prefix
            if ( !empty( $exclude_prefixes ) ) {
                foreach ( $exclude_prefixes as $exclude_prefix ) {
                    if ( strpos( $key, $exclude_prefix ) === 0 ) {
                        continue 2;
                    }
                }
            }
            // Add if matches and not in exact exclude list
            if ( !in_array( $key, $exclude, true ) ) {
                $extracted[$key] = $value;
            }
        }
        return $extracted;
    }

    /**
     * Check if Hello Elementor theme is active
     *
     * @return bool
     */
    private function is_hello_theme_active() {
        $theme = wp_get_theme();
        return 'Hello Elementor' === $theme->name || 'Hello Elementor' === $theme->parent_theme;
    }

}
