<?php
namespace SEMFE;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Category Manager
 * Manages all available export/import categories
 */
class Category_Manager {

	/**
	 * Get all available categories
	 *
	 * @return array
	 */
	public function get_categories() {
		$categories = [
			// Design System Group
			'global_colors' => [
				'id' => 'global_colors',
				'label' => esc_html__( 'Global Colors', 'settings-manager-for-elementor' ),
				'group' => 'design_system',
				'group_label' => esc_html__( 'Design System', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Predefined and custom global colors', 'settings-manager-for-elementor' ),
			],
			'global_typography' => [
				'id' => 'global_typography',
				'label' => esc_html__( 'Global Fonts', 'settings-manager-for-elementor' ),
				'group' => 'design_system',
				'group_label' => esc_html__( 'Design System', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Predefined and custom global fonts', 'settings-manager-for-elementor' ),
			],

			// Theme Style Group
			'theme_typography' => [
				'id' => 'theme_typography',
				'label' => esc_html__( 'Typography', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Base typography for body, links and headings', 'settings-manager-for-elementor' ),
			],
			'theme_buttons' => [
				'id' => 'theme_buttons',
				'label' => esc_html__( 'Buttons', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Default button styling', 'settings-manager-for-elementor' ),
			],
			'theme_images' => [
				'id' => 'theme_images',
				'label' => esc_html__( 'Images', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Default image styling', 'settings-manager-for-elementor' ),
			],
			'theme_form_fields' => [
				'id' => 'theme_form_fields',
				'label' => esc_html__( 'Form Fields', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Default form field styling', 'settings-manager-for-elementor' ),
			],

			// Settings Group
			'site_identity_text' => [
				'id' => 'site_identity_text',
				'label' => esc_html__( 'Site Identity (Site Name & Site Description)', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Site name and description', 'settings-manager-for-elementor' ),
			],
			'site_identity_media' => [
				'id' => 'site_identity_media',
				'label' => esc_html__( 'Site Identity (Site Logo & Site Favicon)', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => false,
				'default_import' => false,
				'description' => esc_html__( 'Logo and favicon (usually require manual setup per site)', 'settings-manager-for-elementor' ),
			],
			'background_style' => [
				'id' => 'background_style',
				'label' => esc_html__( 'Background (Settings)', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Site background color, gradient, and style settings', 'settings-manager-for-elementor' ),
			],
			'background_image' => [
				'id' => 'background_image',
				'label' => esc_html__( 'Background (Image)', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => false,
				'default_import' => false,
				'description' => esc_html__( 'Site background image (usually requires manual setup per site)', 'settings-manager-for-elementor' ),
			],
			'layout_settings' => [
				'id' => 'layout_settings',
				'label' => esc_html__( 'Layout Settings', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Content width, widget spacing, page title selector, etc.', 'settings-manager-for-elementor' ),
			],
			'breakpoints' => [
				'id' => 'breakpoints',
				'label' => esc_html__( 'Breakpoints', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Responsive breakpoints for mobile, tablet, laptop, and widescreen', 'settings-manager-for-elementor' ),
			],
			'lightbox_settings' => [
				'id' => 'lightbox_settings',
				'label' => esc_html__( 'Lightbox', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Lightbox colors and other settings', 'settings-manager-for-elementor' ),
			],
			'page_transitions' => [
				'id' => 'page_transitions',
				'label' => esc_html__( 'Page Transitions', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'is_pro' => true,
				'description' => esc_html__( 'Page transition background and animation settings (Elementor Pro)', 'settings-manager-for-elementor' ),
			],
			'preloader_settings' => [
				'id' => 'preloader_settings',
				'label' => esc_html__( 'Preloader (Settings)', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'is_pro' => true,
				'description' => esc_html__( 'Preloader animation, colors, and style settings (Elementor Pro)', 'settings-manager-for-elementor' ),
			],
			'preloader_media' => [
				'id' => 'preloader_media',
				'label' => esc_html__( 'Preloader (Icon & Image)', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => false,
				'default_import' => false,
				'is_pro' => true,
				'description' => esc_html__( 'Preloader icon and image (usually require manual setup per site) (Elementor Pro)', 'settings-manager-for-elementor' ),
			],
			'custom_css' => [
				'id' => 'custom_css',
				'label' => esc_html__( 'Custom CSS', 'settings-manager-for-elementor' ),
				'group' => 'settings',
				'group_label' => esc_html__( 'Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'is_pro' => true,
				'description' => esc_html__( 'Global custom CSS code (Elementor Pro)', 'settings-manager-for-elementor' ),
			],

			// Backend Settings Group
			'backend_general' => [
				'id' => 'backend_general',
				'label' => esc_html__( 'General', 'settings-manager-for-elementor' ),
				'group' => 'backend',
				'group_label' => esc_html__( 'Additional (Backend) Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Post types, colors, fonts, tracking', 'settings-manager-for-elementor' ),
			],
			'backend_integrations' => [
				'id' => 'backend_integrations',
				'label' => esc_html__( 'Integrations', 'settings-manager-for-elementor' ),
				'group' => 'backend',
				'group_label' => esc_html__( 'Additional (Backend) Settings', 'settings-manager-for-elementor' ),
				'default_export' => false,
				'default_import' => false,
				'description' => esc_html__( 'API keys (Google Maps and other, Stripe excluded for security reasons) - handle with care', 'settings-manager-for-elementor' ),
			],
			'backend_advanced' => [
				'id' => 'backend_advanced',
				'label' => esc_html__( 'Advanced', 'settings-manager-for-elementor' ),
				'group' => 'backend',
				'group_label' => esc_html__( 'Additional (Backend) Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Editor settings, file uploads, Google Fonts, etc.', 'settings-manager-for-elementor' ),
			],
			'backend_performance' => [
				'id' => 'backend_performance',
				'label' => esc_html__( 'Performance', 'settings-manager-for-elementor' ),
				'group' => 'backend',
				'group_label' => esc_html__( 'Additional (Backend) Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'CSS print method, image optimization, etc.', 'settings-manager-for-elementor' ),
			],

			'experiments' => [
				'id' => 'experiments',
				'label' => esc_html__( 'Features', 'settings-manager-for-elementor' ),
				'group' => 'backend',
				'group_label' => esc_html__( 'Additional (Backend) Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Ongoing experiments and stable features', 'settings-manager-for-elementor' ),
			],
		];

		// Only add Hello categories if Hello theme is active
		if ( $this->is_hello_theme_active() ) {
			$categories['hello_header_style'] = [
				'id' => 'hello_header_style',
				'label' => esc_html__( 'Hello Theme Header (Settings)', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Hello theme header settings', 'settings-manager-for-elementor' ),
			];
			$categories['hello_header_image'] = [
				'id' => 'hello_header_image',
				'label' => esc_html__( 'Hello Theme Header (Background Image)', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => false,
				'default_import' => false,
				'description' => esc_html__( 'Hello theme header background image (usually requires manual setup per site)', 'settings-manager-for-elementor' ),
			];

			$categories['hello_footer_style'] = [
				'id' => 'hello_footer_style',
				'label' => esc_html__( 'Hello Theme Footer (Settings)', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Hello theme footer settings', 'settings-manager-for-elementor' ),
			];
			$categories['hello_footer_image'] = [
				'id' => 'hello_footer_image',
				'label' => esc_html__( 'Hello Theme Footer (Background Image)', 'settings-manager-for-elementor' ),
				'group' => 'theme_style',
				'group_label' => esc_html__( 'Theme Style', 'settings-manager-for-elementor' ),
				'default_export' => false,
				'default_import' => false,
				'description' => esc_html__( 'Hello theme footer background image (usually requires manual setup per site)', 'settings-manager-for-elementor' ),
			];

			$categories['hello_backend'] = [
				'id' => 'hello_backend',
				'label' => esc_html__( 'Hello Theme Settings', 'settings-manager-for-elementor' ),
				'group' => 'backend',
				'group_label' => esc_html__( 'Additional (Backend) Settings', 'settings-manager-for-elementor' ),
				'default_export' => true,
				'default_import' => true,
				'description' => esc_html__( 'Hello theme advanced settings', 'settings-manager-for-elementor' ),
			];
		}

		return $categories;
	}

	/**
	 * Get categories grouped by group
	 *
	 * @return array
	 */
	public function get_grouped_categories() {
		$categories = $this->get_categories();
		$grouped = [];

		foreach ( $categories as $category ) {
			$group = $category['group'];
			if ( ! isset( $grouped[ $group ] ) ) {
				$grouped[ $group ] = [
					'label' => $category['group_label'],
					'categories' => [],
				];
			}
			$grouped[ $group ]['categories'][] = $category;
		}

		return $grouped;
	}

	/**
	 * Get default export categories
	 *
	 * @return array
	 */
	public function get_default_export() {
		$categories = $this->get_categories();
		$selected = [];

		foreach ( $categories as $id => $category ) {
			if ( $category['default_export'] ) {
				$selected[] = $id;
			}
		}

		return $selected;
	}

	/**
	 * Get default import categories
	 *
	 * @return array
	 */
	public function get_default_import() {
		$categories = $this->get_categories();
		$selected = [];

		foreach ( $categories as $id => $category ) {
			if ( $category['default_import'] ) {
				$selected[] = $id;
			}
		}

		return $selected;
	}

	/**
	 * Get default selected categories (backward compatibility)
	 *
	 * @deprecated Use get_default_export() or get_default_import() instead
	 * @return array
	 */
	public function get_default_selected() {
		return $this->get_default_export();
	}

	/**
	 * Validate category IDs
	 *
	 * @param array $category_ids
	 * @return array Valid category IDs
	 */
	public function validate_categories( $category_ids ) {
		$valid_categories = array_keys( $this->get_categories() );
		return array_intersect( $category_ids, $valid_categories );
	}

	/**
	 * Check if Hello Elementor theme is active
	 *
	 * @return bool
	 */
	private function is_hello_theme_active() {
		$theme = wp_get_theme();
		return ( 'Hello Elementor' === $theme->name || 'Hello Elementor' === $theme->parent_theme );
	}
}
