<?php
namespace SEMFE;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Backup Manager
 * Handles automatic backups before import operations
 */
class Backup_Manager {

	/**
	 * Singleton instance
	 */
	private static $_instance = null;

	/**
	 * Retention period in days
	 */
	const RETENTION_DAYS = 30;

	/**
	 * Get singleton instance
	 *
	 * @return Backup_Manager
	 */
	public static function get_instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {}

	/**
	 * Log error message when debug mode is enabled
	 *
	 * @param string $message Error message to log.
	 */
	private function log_error( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( $message );
		}
	}

	/**
	 * Create a backup of all current settings
	 *
	 * Creates a JSON export of all settings before an import operation.
	 * On failure, logs an error but returns false to allow import to proceed.
	 *
	 * @return string|false File path on success, false on failure
	 */
	public function create_backup() {
		try {
			$dir = $this->ensure_directory_exists();

			global $wp_filesystem;
			if ( empty( $wp_filesystem ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
				WP_Filesystem();
			}

			if ( ! $dir || ! $wp_filesystem->is_writable( $dir ) ) {
				$this->log_error( 'SEMFE: Backup directory not writable: ' . $dir );
				return false;
			}

			$filename = $this->generate_filename();
			$filepath = trailingslashit( $dir ) . $filename;

			// Export all current settings
			$export_manager = Plugin::instance()->export_manager;
			$category_manager = Plugin::instance()->category_manager;
			$all_categories = array_keys( $category_manager->get_categories() );

			$export_data = $export_manager->export( $all_categories );
			$export_data['_backup_meta'] = [
				'created_at' => gmdate( 'c' ),
				'type' => 'auto-backup-before-import',
			];

			$json = wp_json_encode( $export_data, JSON_PRETTY_PRINT );

			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
			$result = file_put_contents( $filepath, $json );

			if ( false === $result ) {
				$this->log_error( 'SEMFE: Failed to write backup file: ' . $filepath );
				return false;
			}

			// Cleanup old backups (non-blocking)
			$this->cleanup_old_backups();

			return $filepath;

		} catch ( \Exception $e ) {
			$this->log_error( 'SEMFE: Backup failed - ' . $e->getMessage() );
			return false;
		}
	}

	/**
	 * Get the backup directory path for the current site
	 *
	 * Uses wp_upload_dir() to respect custom upload directories.
	 * On multisite, this automatically returns the correct path per site.
	 *
	 * @return string Backup directory path
	 */
	public function get_backup_directory() {
		$upload_dir = wp_upload_dir();
		return trailingslashit( $upload_dir['basedir'] ) . 'semfe/auto-backup';
	}

	/**
	 * Ensure the backup directory exists with proper protection
	 *
	 * Creates the directory if it doesn't exist and adds an index.php
	 * file to prevent directory listing.
	 *
	 * @return string|false Directory path on success, false on failure
	 */
	public function ensure_directory_exists() {
		$dir = $this->get_backup_directory();

		if ( ! file_exists( $dir ) ) {
			$created = wp_mkdir_p( $dir );
			if ( ! $created ) {
				$this->log_error( 'SEMFE: Could not create backup directory: ' . $dir );
				return false;
			}
		}

		// Add index.php to prevent directory listing
		$index = trailingslashit( $dir ) . 'index.php';
		if ( ! file_exists( $index ) ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_file_put_contents
			file_put_contents( $index, '<?php // Silence is golden.' );
		}

		return $dir;
	}

	/**
	 * Generate a unique backup filename
	 *
	 * Format: backup-YYYYMMDD-HHMMSS-{random16hex}.json
	 * The 16 hex character suffix (64 bits) makes the filename unguessable.
	 *
	 * @return string Filename
	 */
	public function generate_filename() {
		// 16 hex chars = 64 bits of randomness (unguessable)
		$random = bin2hex( random_bytes( 8 ) );
		return 'backup-' . gmdate( 'Ymd-His' ) . '-' . $random . '.json';
	}

	/**
	 * Clean up backups older than the retention period
	 *
	 * Runs after each new backup to keep the directory clean.
	 * Silently ignores errors to avoid blocking operations.
	 */
	public function cleanup_old_backups() {
		$dir = $this->get_backup_directory();

		if ( ! is_dir( $dir ) ) {
			return;
		}

		$files = glob( trailingslashit( $dir ) . 'backup-*.json' );
		if ( empty( $files ) ) {
			return;
		}

		$threshold = time() - ( self::RETENTION_DAYS * DAY_IN_SECONDS );

		foreach ( $files as $file ) {
			if ( filemtime( $file ) < $threshold ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink
				@unlink( $file );
			}
		}
	}
}
