<?php

namespace SEMFE;

if ( !defined( 'ABSPATH' ) ) {
    exit;
}
/**
 * Admin Page
 * Creates root-level admin page for Settings Manager
 */
class Admin_Page {
    /**
     * Plugin instance
     */
    private $plugin;

    /**
     * Constructor
     */
    public function __construct( $plugin ) {
        $this->plugin = $plugin;
        add_action( 'admin_menu', [$this, 'register_admin_menu'], 100 );
        add_action( 'admin_enqueue_scripts', [$this, 'enqueue_admin_scripts'] );
        // Add filter early (before Elementor's current_screen hook at priority 10)
        add_action( 'current_screen', [$this, 'dequeue_elementor_admin_bar'], 9 );
    }

    /**
     * Register admin menu
     */
    public function register_admin_menu() {
        add_menu_page(
            esc_html__( 'Settings Manager for Elementor', 'settings-manager-for-elementor' ),
            esc_html__( 'Settings Manager for Elementor', 'settings-manager-for-elementor' ),
            'manage_options',
            'settings-manager-for-elementor',
            [$this, 'render_admin_page'],
            'dashicons-admin-settings',
            59
        );
    }

    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts( $hook ) {
        // Only load on our admin page
        if ( $hook !== 'toplevel_page_settings-manager-for-elementor' ) {
            return;
        }
        wp_enqueue_script(
            'semfe-admin',
            SEMFE_URL . 'assets/js/admin-page.js',
            ['jquery', 'wp-i18n'],
            SEMFE_VERSION,
            true
        );
        // Set script translations for JavaScript i18n
        wp_set_script_translations( 'semfe-admin', 'settings-manager-for-elementor', SEMFE_PATH . 'languages' );
        wp_enqueue_style(
            'semfe-admin',
            SEMFE_URL . 'assets/css/admin-page.css',
            [],
            SEMFE_VERSION
        );
        wp_localize_script( 'semfe-admin', 'semfeAdminConfig', [
            'restUrl'            => rest_url( 'semfe/v1' ),
            'nonce'              => wp_create_nonce( 'wp_rest' ),
            'categories'         => $this->plugin->category_manager->get_categories(),
            'grouped_categories' => $this->plugin->category_manager->get_grouped_categories(),
            'default_export'     => $this->plugin->category_manager->get_default_export(),
            'default_import'     => $this->plugin->category_manager->get_default_import(),
            'can_use_premium'    => semfe_fs()->can_use_premium_code(),
            'is_multisite'       => is_multisite(),
            'is_network_admin'   => current_user_can( 'manage_network' ),
            'current_blog_id'    => get_current_blog_id(),
        ] );
    }

    /**
     * Disable Elementor admin top bar, notices, and footer on our plugin pages
     */
    public function dequeue_elementor_admin_bar() {
        $screen = get_current_screen();
        // Check if we're on our plugin's pages
        if ( strpos( $screen->id ?? '', 'settings-manager-for-elementor' ) === false ) {
            return;
        }
        // Disable Elementor admin top bar
        add_filter( 'elementor/admin-top-bar/is-active', '__return_false' );
        // Remove Elementor admin notices
        $admin_notices = \Elementor\Plugin::$instance->admin->get_component( 'admin-notices' );
        if ( $admin_notices ) {
            remove_action( 'admin_notices', [$admin_notices, 'admin_notices'], 20 );
        }
        // Remove Elementor footer text
        add_filter( 'admin_footer_text', [$this, 'restore_default_footer_text'], 11 );
    }

    /**
     * Restore default WordPress footer text on our pages
     */
    public function restore_default_footer_text( $footer_text ) {
        return '';
    }

    /**
     * Render admin page
     */
    public function render_admin_page() {
        include SEMFE_PATH . 'templates/admin-page.php';
    }

    /**
     * Render category tree for export section
     * Server-side rendering to avoid flash of unstyled content
     */
    public function render_category_tree() {
        $grouped_categories = $this->plugin->category_manager->get_grouped_categories();
        $can_use_premium = semfe_fs()->can_use_premium_code();
        foreach ( $grouped_categories as $group_id => $group ) {
            ?>
			<div class="semfe-category-group">
				<h4><?php 
            echo esc_html( $group['label'] );
            ?></h4>
				<div class="semfe-category-list">
					<?php 
            foreach ( $group['categories'] as $category ) {
                $is_pro = !empty( $category['is_pro'] ) && !$can_use_premium;
                $is_disabled = ( $is_pro ? ' disabled' : '' );
                $is_checked = ( !$is_pro && $category['default_export'] ? ' checked' : '' );
                $item_class = ( $is_pro ? 'semfe-category-item semfe-pro-category' : 'semfe-category-item' );
                ?>
						<label class="<?php 
                echo esc_attr( $item_class );
                ?>">
							<span><input type="checkbox" value="<?php 
                echo esc_attr( $category['id'] );
                ?>" data-group="<?php 
                echo esc_attr( $group_id );
                ?>"<?php 
                echo esc_attr( $is_checked . $is_disabled );
                ?>>&ZeroWidthSpace;</span>
							<div>
								<strong><?php 
                echo esc_html( $category['label'] );
                ?></strong>
								<span class="description"><?php 
                echo esc_html( $category['description'] );
                ?></span>
							</div>
							<?php 
                if ( $is_pro ) {
                    ?>
								<span class="semfe-pro-badge">PRO</span>
							<?php 
                }
                ?>
						</label>
					<?php 
            }
            ?>
				</div>
			</div>
			<?php 
        }
    }

}
