(function($) {
    'use strict';

    // Import WordPress i18n functions
    const { __, _x, sprintf } = wp.i18n;

    let importData = null;
    let networkSites = []; // Cached network sites

    // Export for PRO-only network JS
    window.semfeAdmin = window.semfeAdmin || {};

    $(document).ready(function() {
        init();
    });

    function init() {
        // Export categories are rendered server-side in PHP
        setupSelectAll('#export-select-all', '#export-category-tree');
        setupExport();
        setupImport();

        // Setup tabs
        setupTabs();
        handleDeepLink();

        // Load saved filename from localStorage
        if ($('#export-filename').length) {
            const savedFilename = localStorage.getItem('semfe_export_filename');
            if (savedFilename) {
                $('#export-filename').val(savedFilename);
            }
        }

        // Setup description character counter
        setupDescriptionCounter();

        // Load network sites on init (if multisite + super admin)
        if (semfeAdminConfig.is_multisite && semfeAdminConfig.is_network_admin) {
            loadNetworkSites();
        }
    }

    function renderCategoryTree(container, groupedCategories, context) {
        let html = '';

        for (const groupId in groupedCategories) {
            const group = groupedCategories[groupId];

            html += '<div class="semfe-category-group">';
            html += '<h4>' + group.label + '</h4>';
            html += '<div class="semfe-category-list">';

            group.categories.forEach(function(category) {
                // Check if category is PRO and user doesn't have premium
                const isPro = category.is_pro && !semfeAdminConfig.can_use_premium;
                const isDisabled = isPro ? ' disabled' : '';
                // For PRO categories in free version: unchecked. Otherwise: use default_export/default_import
                const isChecked = isPro ? '' : ((context === 'export') ? (category.default_export ? ' checked' : '') : (category.default_import ? ' checked' : ''));
                const itemClass = isPro ? 'semfe-category-item semfe-pro-category' : 'semfe-category-item';
                const proBadge = isPro ? '<span class="semfe-pro-badge">PRO</span>' : '';

                html += '<label class="' + itemClass + '">';
                html += '<span><input type="checkbox" value="' + category.id + '" data-group="' + groupId + '"' + isChecked + isDisabled + '>&ZeroWidthSpace;</span>';
				html += '<div>';
                html += '<strong>' + category.label + '</strong>';
                html += '<span class="description">' + category.description + '</span>';
				html += '</div>';
                html += proBadge;
                html += '</label>';
            });

            html += '</div>';
            html += '</div>';
        }

        $(container).html(html);
    }

    function setupSelectAll(checkboxSelector, containerSelector) {
        $(checkboxSelector).on('change', function() {
            const isChecked = $(this).is(':checked');
            // Skip disabled checkboxes (PRO categories in free version)
            $(containerSelector + ' input[type="checkbox"]:not(:disabled)').prop('checked', isChecked);
        });

        // Update select all when individual checkboxes change
        $(containerSelector).on('change', 'input[type="checkbox"]', function() {
            // Only count non-disabled checkboxes
            const total = $(containerSelector + ' input[type="checkbox"]:not(:disabled)').length;
            const checked = $(containerSelector + ' input[type="checkbox"]:not(:disabled):checked').length;
            $(checkboxSelector).prop('checked', total === checked);
        });
    }

    /**
     * Setup character counter for description textarea
     */
    function setupDescriptionCounter() {
        const $textarea = $('#export-description');
        const $counter = $('.semfe-description-option .semfe-char-counter');

        if (!$textarea.length || !$counter.length) return;

        $textarea.on('input', function() {
            const length = $(this).val().length;
            $counter.text(length + ' / 1000 ' + __('characters', 'settings-manager-for-elementor'));
        });
    }

    function setupExport() {
        $('#export-btn').on('click', function() {
            const selectedCategories = getSelectedCategories('#export-category-tree');

            if (selectedCategories.length === 0) {
                showStatus('#export-status', 'error', __('Please select at least one category to export.', 'settings-manager-for-elementor'));
                return;
            }

            exportSettings(selectedCategories);
        });
    }

    function exportSettings(categories) {
        const $btn = $('#export-btn');
        const originalText = $btn.html();
        const description = $('#export-description').val() || '';

        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> ' + __('Exporting...', 'settings-manager-for-elementor'));
        showStatus('#export-status', 'info', __('Exporting settings...', 'settings-manager-for-elementor'));

        $.ajax({
            url: semfeAdminConfig.restUrl + '/export',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', semfeAdminConfig.nonce);
            },
            contentType: 'application/json',
            data: JSON.stringify({
                categories: categories,
                description: description
            }),
            success: function(response) {
                if (response.success) {
                    // Build filename
                    let filename = 'elementor-settings-' + response.data.export_timestamp + '.json';
                    // Use custom name if provided
                    if ($('#export-filename').length) {
                        const customName = $('#export-filename').val().trim();
                        if (customName) {
                            // Sanitize: remove invalid filename characters
                            filename = customName.replace(/[<>:"/\\|?*]/g, '-') + '.json';
                            // Save to localStorage for next time
                            localStorage.setItem('semfe_export_filename', customName);
                        } else {
                            // Clear localStorage if empty
                            localStorage.removeItem('semfe_export_filename');
                        }
                    }
                    downloadJSON(response.data, filename);
                    showStatus('#export-status', 'success', __('Export completed successfully!', 'settings-manager-for-elementor'));
                } else {
                    showStatus('#export-status', 'error', __('Export failed', 'settings-manager-for-elementor'));
                }
            },
            error: function(xhr) {
                const message = xhr.responseJSON && xhr.responseJSON.message ? xhr.responseJSON.message : __('Export failed', 'settings-manager-for-elementor');
                showStatus('#export-status', 'error', message);
            },
            complete: function() {
                $btn.prop('disabled', false).html(originalText);
            }
        });
    }

    function setupImport() {
        $('#choose-file-btn').on('click', function() {
            $('#import-file-input').click();
        });

        $('#import-file-input').on('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                $('#file-name').text(file.name);
                loadImportFile(file);
            }
        });

        // Note: Import select-all handler is set up via setupSelectAll() in renderCategoryTreeForImport()
        // to properly exclude disabled PRO categories

        $('#import-btn').on('click', function() {
            importSettings();
        });

        $('#cancel-import-btn').on('click', function() {
            cancelImport();
        });
    }

    function loadImportFile(file) {
        const reader = new FileReader();

        reader.onload = function(e) {
            try {
                importData = JSON.parse(e.target.result);
                previewImport(importData);
            } catch (error) {
                showStatus('#import-status', 'error', __('Invalid JSON file: ', 'settings-manager-for-elementor') + error.message);
            }
        };

        reader.readAsText(file);
    }

    function previewImport(data) {
        showStatus('#import-status', 'info', __('Loading preview...', 'settings-manager-for-elementor'));

        $.ajax({
            url: semfeAdminConfig.restUrl + '/import-preview',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', semfeAdminConfig.nonce);
            },
            contentType: 'application/json',
            data: JSON.stringify(data),
            success: function(response) {
                if (response.success && response.preview.valid) {
                    renderImportPreview(response.preview);
                    $('#import-preview').show();
                    showStatus('#import-status', 'success', __('Preview loaded', 'settings-manager-for-elementor'));
                } else {
                    showStatus('#import-status', 'error', response.preview.error || __('Invalid import data', 'settings-manager-for-elementor'));
                }
            },
            error: function(xhr) {
                const message = xhr.responseJSON && xhr.responseJSON.message ? xhr.responseJSON.message : __('Preview failed', 'settings-manager-for-elementor');
                showStatus('#import-status', 'error', message);
            }
        });
    }

    function renderImportPreview(preview) {
        let html = '<div class="semfe-preview-info">';
        html += '<p><strong>' + __('Exported:', 'settings-manager-for-elementor') + '</strong> ' + preview.export_date + '</p>';
        html += '<p><strong>' + __('From:', 'settings-manager-for-elementor') + '</strong> ' + preview.site_url + '</p>';
        html += '<p><strong>' + __('Settings Manager for Elementor Version:', 'settings-manager-for-elementor') + '</strong> ' + preview.plugin_version + '</p>';
        html += '<p><strong>' + __('Elementor Version:', 'settings-manager-for-elementor') + '</strong> ' + preview.elementor_version + '</p>';
        html += '<p><strong>' + __('Theme:', 'settings-manager-for-elementor') + '</strong> ' + preview.theme_name + ' ' + preview.theme_version + '</p>';
        if (preview.description) {
            html += '<p><strong>' + __('Description:', 'settings-manager-for-elementor') + '</strong></p>';
            html += '<pre class="semfe-description-preview">' + escapeHtml(preview.description) + '</pre>';
        }
        html += '<p><strong>' + __('Contains:', 'settings-manager-for-elementor') + '</strong></p>';
        html += '<ul>';

        for (const categoryId in preview.counts) {
            const category = semfeAdminConfig.categories[categoryId];
            if (category) {
                html += '<li><strong>' + category.label + ':</strong> ' + sprintf(__('%d items', 'settings-manager-for-elementor'), preview.counts[categoryId]) + '</li>';
            }
        }

        html += '</ul>';
        html += '<p><strong>' + __('Total:', 'settings-manager-for-elementor') + '</strong> ' + sprintf(__('%d items', 'settings-manager-for-elementor'), preview.total_items) + '</p>';
        html += '</div>';

        $('#import-preview-content').html(html);

        // Render category tree for import
        renderCategoryTreeForImport(preview.categories);
    }

    function renderCategoryTreeForImport(availableCategories) {
        const grouped = {};

        availableCategories.forEach(function(categoryId) {
            const category = semfeAdminConfig.categories[categoryId];
            if (category) {
                const groupId = category.group;
                if (!grouped[groupId]) {
                    grouped[groupId] = {
                        label: category.group_label,
                        categories: []
                    };
                }
                grouped[groupId].categories.push(category);
            }
        });

        renderCategoryTree('#import-category-tree', grouped, 'import');
        setupSelectAll('#import-select-all', '#import-category-tree');
    }

    function importSettings() {
        const selectedCategories = getSelectedCategories('#import-category-tree');

        if (selectedCategories.length === 0) {
            showStatus('#import-status', 'error', __('Please select at least one category to import', 'settings-manager-for-elementor'));
            return;
        }

        const strategy = $('input[name="import-strategy"]:checked').val();
        const $btn = $('#import-btn');
        const originalText = $btn.html();

        if (!confirm(__('Are you sure you want to import these settings? This will modify your Elementor configuration.', 'settings-manager-for-elementor'))) {
            return;
        }

        $btn.prop('disabled', true).html('<span class="dashicons dashicons-update spin"></span> ' + __('Importing...', 'settings-manager-for-elementor'));
        showStatus('#import-status', 'info', __('Importing settings...', 'settings-manager-for-elementor'));

        $.ajax({
            url: semfeAdminConfig.restUrl + '/import',
            method: 'POST',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', semfeAdminConfig.nonce);
            },
            contentType: 'application/json',
            data: JSON.stringify({
                data: importData,
                categories: selectedCategories,
                strategy: strategy
            }),
            success: function(response) {
                if (response.success) {
                    let message = __('Import completed!', 'settings-manager-for-elementor');
                    if (response.errors && Object.keys(response.errors).length > 0) {
                        message += __(' Some categories had errors.', 'settings-manager-for-elementor');
                    }
                    showStatus('#import-status', 'success', message);
                } else {
                    showStatus('#import-status', 'error', response.error || __('Import failed', 'settings-manager-for-elementor'));
                }
            },
            error: function(xhr) {
                const message = xhr.responseJSON && xhr.responseJSON.message ? xhr.responseJSON.message : __('Import failed', 'settings-manager-for-elementor');
                showStatus('#import-status', 'error', message);
            },
            complete: function() {
                $btn.prop('disabled', false).html(originalText);
            }
        });
    }

    function cancelImport() {
        importData = null;
        $('#import-preview').hide();
        $('#import-file-input').val('');
        $('#file-name').text('');
        showStatus('#import-status', '', '');
    }

    function getSelectedCategories(container) {
        const selected = [];
        $(container + ' input[type="checkbox"]:checked').each(function() {
            selected.push($(this).val());
        });
        return selected;
    }

    function downloadJSON(data, filename) {
        const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }

    function showStatus(container, type, message) {
        let className = 'notice ';
        if (type === 'success') className += 'notice-success';
        else if (type === 'error') className += 'notice-error';
        else if (type === 'info') className += 'notice-info';

        $(container).html('<div class="' + className + '"><p>' + message + '</p></div>');

        if (message === '') {
            $(container).empty();
        }
    }

    function getTimestamp() {
        const now = new Date();
        return now.getFullYear() +
            String(now.getMonth() + 1).padStart(2, '0') +
            String(now.getDate()).padStart(2, '0') + '-' +
            String(now.getHours()).padStart(2, '0') +
            String(now.getMinutes()).padStart(2, '0');
    }

    // ==========================================
    // Tab Navigation Functions
    // ==========================================

    /**
     * Setup tab navigation
     */
    function setupTabs() {
        $('.semfe-admin-page .nav-tab').on('click', function(e) {
            e.preventDefault();

            const targetTab = $(this).data('tab');
            switchTab(targetTab);
        });
    }

    /**
     * Switch to a specific tab
     * @param {string} tabName - Name of tab to switch to (export, import, help)
     */
    function switchTab(tabName) {
        const $page = $('.semfe-admin-page');

        // Update tab navigation active state
        $page.find('.nav-tab').removeClass('nav-tab-active').attr('aria-selected', 'false');
        $page.find('.nav-tab[data-tab="' + tabName + '"]').addClass('nav-tab-active').attr('aria-selected', 'true');

        // Update tab content visibility
        $page.find('.semfe-tab-content').removeClass('active');
        $page.find('.semfe-tab-content[data-tab="' + tabName + '"]').addClass('active');

        // Update URL hash for deep linking (replaceState to avoid history pollution)
        if (history.replaceState) {
            history.replaceState(null, null, '#' + tabName);
        } else {
            window.location.hash = tabName;
        }
    }

    /**
     * Handle deep linking via URL hash
     * Switches to tab if valid hash present in URL
     */
    function handleDeepLink() {
        const hash = window.location.hash.substring(1).toLowerCase(); // Remove # and normalize
        const validTabs = ['export', 'import', 'help'];

        if (hash && validTabs.includes(hash)) {
            switchTab(hash);
        }

        // Listen for hash changes (browser back/forward)
        $(window).on('hashchange', function() {
            const newHash = window.location.hash.substring(1).toLowerCase();
            if (newHash && validTabs.includes(newHash)) {
                switchTab(newHash);
            }
        });
    }

    // ==========================================
    // Network Sites Functions (for multisite)
    // ==========================================

    /**
     * Load network sites on init (if multisite + super admin)
     * Always loads actual sites for both free and PRO users.
     * Free users see all checkboxes disabled (PRO teaser).
     */
    function loadNetworkSites() {
        const $container = $('#import-sites-list');
        if (!$container.length) return;

        $.ajax({
            url: semfeAdminConfig.restUrl + '/network/sites',
            method: 'GET',
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', semfeAdminConfig.nonce);
            },
            success: function(response) {
                if (response.success && response.sites) {
                    networkSites = response.sites;
                    // Pass isPremium flag to disable all checkboxes for free users
                    renderSitesList($container, response.sites, semfeAdminConfig.can_use_premium);
                } else {
                    $container.html('<div class="semfe-no-sites">' + __('No other network sites found.', 'settings-manager-for-elementor') + '</div>');
                }
            },
            error: function() {
                $container.html('<div class="semfe-no-sites">' + __('Failed to load network sites.', 'settings-manager-for-elementor') + '</div>');
            }
        });
    }

    /**
     * Render site checkboxes
     * @param {jQuery} $container - Container element
     * @param {Array} sites - Array of site objects
     * @param {boolean} isPremium - Whether user has premium license (false = disable all checkboxes)
     */
    function renderSitesList($container, sites, isPremium) {
        if (!sites || sites.length === 0) {
            $container.html('<div class="semfe-no-sites">' + __('No other network sites found.', 'settings-manager-for-elementor') + '</div>');
            return;
        }

        let html = '';

        // Select All (disabled for free users)
        const selectAllDisabled = isPremium ? '' : ' disabled';
        html += '<div class="semfe-sites-select-all">';
        html += '<label><input type="checkbox" id="network-sites-select-all"' + selectAllDisabled + '> ' + __('Select All', 'settings-manager-for-elementor') + '</label>';
        html += '</div>';

        // Site list
        sites.forEach(function(site) {
            // Disabled if: not premium OR Elementor not active
            const isDisabledByElementor = !site.elementor_active;
            const isDisabled = !isPremium || isDisabledByElementor;
            const itemClass = isDisabled ? 'semfe-site-item semfe-site-disabled' : 'semfe-site-item';
            const checkboxDisabled = isDisabled ? ' disabled' : '';

            html += '<div class="' + itemClass + '">';
            html += '<label>';
            html += '<input type="checkbox" value="' + site.id + '" data-name="' + escapeHtml(site.name) + '"' + checkboxDisabled + '>';
            html += '<div class="semfe-site-info">';
            html += '<span class="semfe-site-name">' + escapeHtml(site.name) + '</span>';
            html += '<span class="semfe-site-url">' + escapeHtml(site.url) + '</span>';
            if (isDisabledByElementor) {
                html += '<span class="semfe-site-warning"><span class="dashicons dashicons-warning"></span>' + __('Elementor not active', 'settings-manager-for-elementor') + '</span>';
            }
            html += '</div>';
            html += '</label>';
            html += '</div>';
        });

        $container.html(html);

        // Only setup interactivity for premium users
        if (isPremium) {
            setupNetworkSitesSelectAll();
            $container.on('change', 'input[type="checkbox"]', updateImportButtonText);
        }
    }

    /**
     * Setup select all for network sites
     */
    function setupNetworkSitesSelectAll() {
        $('#network-sites-select-all').on('change', function() {
            const isChecked = $(this).is(':checked');
            $('#import-sites-list input[type="checkbox"]:not(:disabled)').prop('checked', isChecked);
            updateImportButtonText();
        });

        // Update select all when individual checkboxes change
        $('#import-sites-list').on('change', 'input[type="checkbox"]:not(#network-sites-select-all)', function() {
            const total = $('#import-sites-list input[type="checkbox"]:not(:disabled):not(#network-sites-select-all)').length;
            const checked = $('#import-sites-list input[type="checkbox"]:not(:disabled):not(#network-sites-select-all):checked').length;
            $('#network-sites-select-all').prop('checked', total > 0 && total === checked);
        });
    }

    /**
     * Get selected network sites
     */
    function getSelectedSites() {
        const selected = [];
        $('#import-sites-list input[type="checkbox"]:checked:not(#network-sites-select-all)').each(function() {
            selected.push({
                id: parseInt($(this).val(), 10),
                name: $(this).data('name')
            });
        });
        return selected;
    }

    /**
     * Update import button text based on selected network sites
     */
    function updateImportButtonText() {
        const selectedSites = getSelectedSites();
        const $btn = $('#import-btn');

        if (selectedSites.length === 0) {
            $btn.text(__('Import to This Site', 'settings-manager-for-elementor'));
        } else {
            $btn.text(sprintf(
                __('Import to This Site + %d Network Sites', 'settings-manager-for-elementor'),
                selectedSites.length
            ));
        }
    }

    /**
     * Show inline progress in .semfe-status
     */
    function showInlineProgress(sites, statusContainer) {
        const totalSites = sites.length;

        let html = '<div class="semfe-inline-progress">';
        html += '<h4>' + sprintf(__('Importing to %d sites...', 'settings-manager-for-elementor'), totalSites) + ' (<span class="progress-count">0</span>/' + totalSites + ')</h4>';
        html += '<div class="semfe-progress-bar"><div class="semfe-progress-bar-fill" style="width: 0%"></div></div>';
        html += '<div class="semfe-progress-sites">';

        sites.forEach(function(site, index) {
            html += '<div class="semfe-progress-site pending" data-index="' + index + '">';
            html += '<span class="dashicons dashicons-marker"></span>';
            html += '<span class="site-name">' + escapeHtml(site.name) + '</span>';
            html += '</div>';
        });

        html += '</div>';
        html += '<div class="semfe-progress-actions">';
        html += '<button type="button" class="button" id="cancel-network-import">' + __('Cancel', 'settings-manager-for-elementor') + '</button>';
        html += '</div>';
        html += '</div>';

        $(statusContainer).html(html);
    }

    /**
     * Update inline progress (called after each site)
     */
    function updateInlineProgress(index, status, error, statusContainer) {
        const $site = $(statusContainer).find('.semfe-progress-site[data-index="' + index + '"]');
        const totalSites = $(statusContainer).find('.semfe-progress-site').length;

        // Update site status
        $site.removeClass('pending in-progress').addClass(status);

        // Update icon
        let icon = 'marker';
        if (status === 'success') icon = 'yes-alt';
        else if (status === 'error') icon = 'dismiss';
        else if (status === 'skipped') icon = 'minus';
        else if (status === 'in-progress') icon = 'update';

        $site.find('.dashicons').attr('class', 'dashicons dashicons-' + icon);
        if (status === 'in-progress') {
            $site.find('.dashicons').addClass('spin');
        }

        // Add error message if present
        if (error) {
            $site.append('<span class="site-error"> - ' + escapeHtml(error) + '</span>');
        }

        // Update progress bar and count
        const completed = $(statusContainer).find('.semfe-progress-site.success, .semfe-progress-site.error, .semfe-progress-site.skipped').length;
        const percentage = Math.round((completed / totalSites) * 100);
        $(statusContainer).find('.semfe-progress-bar-fill').css('width', percentage + '%');
        $(statusContainer).find('.progress-count').text(completed);

        // Mark next site as in-progress
        if (status !== 'in-progress') {
            const $next = $(statusContainer).find('.semfe-progress-site.pending').first();
            if ($next.length) {
                updateInlineProgress($next.data('index'), 'in-progress', null, statusContainer);
            }
        }
    }

    /**
     * Show final results summary in .semfe-status
     */
    function showInlineResults(results, cancelled, statusContainer) {
        const successCount = results.success.length;
        const errorCount = results.errors.length;
        const totalCount = successCount + errorCount;

        let statusClass = 'semfe-inline-results';
        let icon = 'yes-alt';
        let message = '';

        if (cancelled) {
            statusClass += ' cancelled';
            icon = 'warning';
            message = sprintf(__('Cancelled after %d sites', 'settings-manager-for-elementor'), successCount);
            if (errorCount > 0) {
                message += ', ' + sprintf(__('%d errors', 'settings-manager-for-elementor'), errorCount);
            }
        } else if (errorCount > 0) {
            statusClass += ' has-errors';
            icon = 'warning';
            message = sprintf(__('Imported to %d sites, %d errors', 'settings-manager-for-elementor'), successCount, errorCount);
        } else {
            message = sprintf(__('Successfully imported to %d sites', 'settings-manager-for-elementor'), successCount);
        }

        let html = '<div class="' + statusClass + '">';
        html += '<h4><span class="dashicons dashicons-' + icon + '"></span> ' + message + '</h4>';
        html += '<div class="semfe-progress-sites">';

        // Show successful sites
        results.success.forEach(function(site) {
            html += '<div class="semfe-progress-site success">';
            html += '<span class="dashicons dashicons-yes-alt"></span>';
            html += '<span class="site-name">' + escapeHtml(site.name) + '</span>';
            html += '</div>';
        });

        // Show error sites
        results.errors.forEach(function(site) {
            html += '<div class="semfe-progress-site error">';
            html += '<span class="dashicons dashicons-dismiss"></span>';
            html += '<span class="site-name">' + escapeHtml(site.name) + '</span>';
            html += '<span class="site-error"> - ' + escapeHtml(site.error || __('Import failed', 'settings-manager-for-elementor')) + '</span>';
            html += '</div>';
        });

        // Show skipped sites (if cancelled)
        if (results.skipped) {
            results.skipped.forEach(function(site) {
                html += '<div class="semfe-progress-site skipped">';
                html += '<span class="dashicons dashicons-minus"></span>';
                html += '<span class="site-name">' + escapeHtml(site.name) + ' - ' + __('Skipped', 'settings-manager-for-elementor') + '</span>';
                html += '</div>';
            });
        }

        html += '</div>';
        html += '</div>';

        // Add reload notice
        html += '<div class="notice notice-info"><p>' + __('Reload Elementor editor to see changes.', 'settings-manager-for-elementor') + '</p></div>';

        $(statusContainer).html(html);
    }

    /**
     * Escape HTML entities
     */
    function escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // Export functions for PRO-only network JS
    window.semfeAdmin = {
        getImportData: function() { return importData; },
        getSelectedCategories: function() { return getSelectedCategories('#import-category-tree'); },
        getSelectedSites: getSelectedSites,
        showStatus: showStatus,
        showInlineProgress: showInlineProgress,
        updateInlineProgress: updateInlineProgress,
        showInlineResults: showInlineResults,
        updateImportButtonText: updateImportButtonText,
        networkSites: function() { return networkSites; }
    };

})(jQuery);
