<?php
/**
 * Plugin Name:       Session Shredder for WooCommerce
 * Plugin URI:        https://mambaspeed.com/session-shredder/
 * Description:       Smart, rule-based session pruner for WooCommerce 10.3+ that builds on the experimental guest sessions engine to safely clear out old, low-value sessions.
 * Version:           1.1.0
 * Author:            Rynaldo Stoltz
 * Author URI:        https://mambaspeed.com
 * Requires Plugins:  woocommerce
 * Requires at least: 6.0
 * Requires PHP:      8.0
 * WC requires at least: 10.3
 * WC tested up to:   10.3.5
 * Text Domain:       session-shredder-for-woocommerce
 * Domain Path:       /languages
 * License:           GPL-2.0+
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 *
 * @package Session_Shredder
 */

defined( 'ABSPATH' ) || exit;

define( 'SESSION_SHREDDER_VERSION', '1.1.0' );
define( 'SESSION_SHREDDER_PATH', plugin_dir_path( __FILE__ ) );
define( 'SESSION_SHREDDER_URL', plugin_dir_url( __FILE__ ) );
define( 'SESSION_SHREDDER_PLUGIN_FILE', __FILE__ );

// Declare compatibility with WooCommerce HPOS (custom order tables).
add_action(
	'before_woocommerce_init',
	static function () {
		if ( class_exists( '\\Automattic\\WooCommerce\\Utilities\\FeaturesUtil' ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', SESSION_SHREDDER_PLUGIN_FILE, true );
		}
	}
);

// Load core classes.
require_once SESSION_SHREDDER_PATH . 'includes/class-session-shredder.php';

if ( is_admin() ) {
	require_once SESSION_SHREDDER_PATH . 'includes/admin/class-admin-settings.php';
}

if ( defined( 'WP_CLI' ) && WP_CLI ) {
	require_once SESSION_SHREDDER_PATH . 'includes/class-session-shredder-cli.php';
}

/**
 * Bootstrap the plugin once WooCommerce is loaded.
 *
 * @return void
 */
function session_shredder_init() {
	if ( ! class_exists( 'WooCommerce' ) ) {
		return;
	}

	if ( defined( 'WC_VERSION' ) ) {
		$wc_version = WC_VERSION;
	} elseif ( function_exists( 'WC' ) && WC() instanceof WooCommerce ) {
		$wc_version = WC()->version;
	} else {
		$wc_version = null;
	}

	if ( empty( $wc_version ) || version_compare( $wc_version, '10.3', '<' ) ) {
		add_action(
			'admin_notices',
			static function () {
				echo '<div class="notice notice-error"><p>' . esc_html__( 'Session Shredder requires WooCommerce 10.3 or newer. Please update WooCommerce.', 'session-shredder-for-woocommerce' ) . '</p></div>';
			}
		);
		return;
	}

	if ( is_admin() && class_exists( 'Session_Shredder_Admin_Settings' ) ) {
		new Session_Shredder_Admin_Settings();
	}

	Session_Shredder::get_instance();
}
add_action( 'plugins_loaded', 'session_shredder_init', 20 );

/**
 * Activation hook.
 *
 * @return void
 */
function session_shredder_activate() {
	// Opt-in to WooCommerce experimental guest sessions if present.
	if ( 'yes' !== get_option( 'woocommerce_experimental_guest_sessions' ) ) {
		update_option( 'woocommerce_experimental_guest_sessions', 'yes' );
	}

	if ( ! wp_next_scheduled( 'session_shredder_cron' ) ) {
		wp_schedule_event( time(), 'hourly', 'session_shredder_cron' );
	}
}
register_activation_hook( __FILE__, 'session_shredder_activate' );

/**
 * Deactivation hook.
 *
 * @return void
 */
function session_shredder_deactivate() {
	wp_clear_scheduled_hook( 'session_shredder_cron' );

	if ( function_exists( 'wp_cache_flush' ) ) {
		wp_cache_flush();
	}
}
register_deactivation_hook( __FILE__, 'session_shredder_deactivate' );
