<?php
/**
 * Core Session Shredder class.
 *
 * @package Session_Shredder
 */

defined( 'ABSPATH' ) || exit;

class Session_Shredder {

	/**
	 * Singleton instance.
	 *
	 * @var Session_Shredder|null
	 */
	private static $instance = null;

	/**
	 * Sessions table name.
	 *
	 * @var string
	 */
	private $sessions_table = '';

	/**
	 * Session ID column name.
	 *
	 * @var string
	 */
	private $session_id_column = '';

	/**
	 * Session value column name.
	 *
	 * @var string
	 */
	private $session_value_column = '';

	/**
	 * Session expiry column name.
	 *
	 * @var string
	 */
	private $session_expiry_column = '';

	private $last_prune_reason = '';

	/**
	 * Get singleton instance.
	 *
	 * @return Session_Shredder
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Constructor.
	 */
	private function __construct() {
		// Track user activity on each page load.
		add_action( 'wp', array( $this, 'track_session_activity' ) );

		// Also track when Woo updates a session.
		add_action( 'woocommerce_session_updated', array( $this, 'track_session_activity' ) );

		// Rule-based pruning cron.
		add_action( 'session_shredder_cron', array( $this, 'ai_prune_sessions' ) );

		// Cache invalidation after pruning.
		add_action( 'session_shredder_pruned', array( $this, 'invalidate_caches' ), 10, 1 );

		// Ensure cron is scheduled.
		if ( ! wp_next_scheduled( 'session_shredder_cron' ) ) {
			wp_schedule_event( time(), 'hourly', 'session_shredder_cron' );
		}

		// Admin UI.
		if ( is_admin() ) {
			require_once SESSION_SHREDDER_PATH . 'includes/admin/class-admin-dashboard.php';
			new Session_Shredder_Admin_Dashboard( $this );
		}
	}

	/**
	 * Track per-session activity used as pruning features.
	 *
	 * @return void
	 */
	public function track_session_activity() {
		if ( ! function_exists( 'WC' ) || ! WC()->session ) {
			return;
		}

		$session_id = $this->get_current_session_id();

		if ( ! $session_id ) {
			return;
		}

		$key      = $this->get_features_option_key( $session_id );
		$current  = get_option( $key, array() );
		$features = $this->collect_features( $session_id, $current );

		// Store as non-autoloading options to avoid polluting autoload cache.
		update_option( $key, $features, false );
	}

	/**
	 * Collect pruning features for a given session.
	 *
	 * @param string $session_id Session identifier.
	 * @param array  $existing   Existing feature array.
	 *
	 * @return array
	 */
	private function collect_features( $session_id, array $existing = array() ) {
		$session_hash = md5( (string) $session_id );
		unset( $session_id ); // Privacy: we never store raw session IDs in the payload.

		$now      = time();
		$features = wp_parse_args(
			$existing,
			array(
				'first_seen'    => $now,
				'age_hours'     => 0,
				'pageviews'     => 0,
				'cart_value'    => 0,
				'added_to_cart' => 0,
				'geo_hash'      => '',
				'bounce'        => 0,
			)
		);

		$features['pageviews'] = (int) $features['pageviews'] + 1;

		$cart_value    = 0;
		$added_to_cart = 0;

		if ( function_exists( 'WC' ) && WC()->cart ) {
			// Numeric cart total (no formatting).
			$cart_value    = (float) WC()->cart->get_cart_contents_total();
			$added_to_cart = WC()->cart->is_empty() ? 0 : 1;
		}

		$features['cart_value']    = $cart_value;
		$features['added_to_cart'] = $added_to_cart;

		$country = '';
		if ( function_exists( 'WC' ) && WC()->customer ) {
			$country = WC()->customer->get_billing_country();
			if ( ! $country ) {
				$country = WC()->customer->get_shipping_country();
			}
		}

		if ( ! $country ) {
			$country = 'unknown';
		}

		// Hash country to avoid storing PII.
		$features['geo_hash'] = wp_hash( strtolower( (string) $country ) );

		// Age in hours since we first saw this session.
		$features['age_hours'] = max( ( $now - (int) $features['first_seen'] ) / 3600, 0 );

		// Simple bounce heuristic: first pageview and no cart value.
		$features['bounce'] = ( $features['pageviews'] <= 1 && $cart_value <= 0 ) ? 1 : 0;

		$features = apply_filters( 'session_shredder_features', $features, $session_hash );

		return $features;
	}

	/**
	 * Get the current session identifier.
	 *
	 * @return string
	 */
	private function get_current_session_id() {
		if ( ! function_exists( 'WC' ) || ! WC()->session ) {
			return '';
		}

		// Preferred: WooCommerce session cookie.
		if ( method_exists( WC()->session, 'get_session_cookie' ) ) {
			$cookie = WC()->session->get_session_cookie();
			if ( is_array( $cookie ) && ! empty( $cookie[0] ) ) {
				return (string) $cookie[0];
			}
		}

		// Fallback: customer ID (less ideal but stable).
		if ( method_exists( WC()->session, 'get_customer_id' ) ) {
			return (string) WC()->session->get_customer_id();
		}

		return '';
	}

	/**
	 * Build the option key for feature storage.
	 *
	 * @param string $session_id Session identifier.
	 *
	 * @return string
	 */
	private function get_features_option_key( $session_id ) {
		return 'session_shredder_features_' . md5( (string) $session_id );
	}

	private function enforce_feature_options_cap() {
		global $wpdb;

		$soft_limit = (int) apply_filters( 'session_shredder_feature_options_soft_limit', 100000 );
		if ( $soft_limit <= 0 ) {
			return;
		}

		$batch_size = (int) apply_filters( 'session_shredder_feature_options_gc_batch', 1000 );
		if ( $batch_size <= 0 ) {
			$batch_size = 1000;
		}

		$like_pattern = $wpdb->esc_like( 'session_shredder_features_' ) . '%';

		$total = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE %s",
				$like_pattern
			)
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		$over_limit = $total - $soft_limit;
		if ( $over_limit <= 0 ) {
			return;
		}

		$limit = min( $over_limit, $batch_size );

		$option_ids = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT option_id FROM {$wpdb->options} WHERE option_name LIKE %s ORDER BY option_id ASC LIMIT %d",
				$like_pattern,
				$limit
			)
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		if ( empty( $option_ids ) ) {
			return;
		}

		$ids = array();
		foreach ( $option_ids as $id ) {
			$ids[] = (int) $id;
		}

		if ( empty( $ids ) ) {
			return;
		}

		// Build placeholders for wpdb::prepare() - one %d for each ID.
		$placeholders = implode( ',', array_fill( 0, count( $ids ), '%d' ) );

		// Use spread operator to pass array elements as individual arguments to wpdb::prepare().
		$query = $wpdb->prepare(
			"DELETE FROM {$wpdb->options} WHERE option_id IN ($placeholders)",
			...$ids
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		$wpdb->query( $query ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
	}

	/**
	 * Run rule-based pruning for stale sessions.
	 *
	 * @param bool $dry_run Whether to only count candidates without deleting.
	 *
	 * @return int Number of pruned (or would-be-pruned) sessions.
	 */
	public function ai_prune_sessions( $dry_run = false ) {
		global $wpdb;

		$this->enforce_feature_options_cap();
		$this->init_sessions_table_meta();

		if ( ! $this->sessions_table || ! $this->session_expiry_column || ! $this->session_id_column ) {
			return 0;
		}

		$base_age_hours = (float) get_option( 'session_shredder_base_age_hours', 48 ); // Base age window for candidate sessions.
		$base_age_hours = (float) apply_filters( 'session_shredder_base_age_hours', $base_age_hours );
		if ( $base_age_hours <= 0 ) {
			$base_age_hours = 1;
		}
		$threshold_ts   = time() - ( $base_age_hours * HOUR_IN_SECONDS );

		do_action( 'session_shredder_before_run', $base_age_hours, $dry_run );

		// Sanitize identifiers and fetch candidate sessions based on age (prepared query).
		$id_column     = esc_sql( $this->session_id_column );
		$expiry_column = esc_sql( $this->session_expiry_column );
		$table         = esc_sql( $this->sessions_table );

		$sessions = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT {$id_column} AS session_id FROM {$table} WHERE {$expiry_column} < %d",
				$threshold_ts
			)
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		if ( empty( $sessions ) ) {
			// No candidates found. For real runs, still log a datapoint so the chart reflects the current active sessions.
			if ( ! $dry_run ) {
				$current_total = $this->get_sessions_count();
				$this->update_stats( 0, $current_total, $current_total );
			}

			do_action( 'session_shredder_after_run', 0, $dry_run );

			return 0;
		}

		$pruned_count  = 0;
		$reason_counts = array();

		foreach ( $sessions as $row ) {
			$sid = (string) $row->session_id;

			$feature_key = $this->get_features_option_key( $sid );
			$raw_features    = get_option( $feature_key, array() );
			$has_tracked_features = ! empty( $raw_features );

			$features = wp_parse_args(
				$raw_features,
				array(
					'age_hours'     => $base_age_hours,
					'pageviews'     => 0,
					'cart_value'    => 0,
					'added_to_cart' => 0,
					'geo_hash'      => wp_hash( 'unknown' ),
					'bounce'        => 0,
				)
			);

			$features['has_tracked_features'] = $has_tracked_features ? 1 : 0;

			$should_prune = $this->should_prune_by_rule( $sid, $features );

			$reason = $this->last_prune_reason;
			if ( $should_prune ) {
				if ( empty( $reason ) ) {
					$reason = 'other';
				}
				if ( ! isset( $reason_counts[ $reason ] ) ) {
					$reason_counts[ $reason ] = 0;
				}
				$reason_counts[ $reason ]++;
			}

			if ( $should_prune ) {
				if ( ! $dry_run ) {
					$this->delete_session( $sid );
				}
				$pruned_count++;
			}

			// Clean up feature storage only on real runs.
			if ( ! $dry_run ) {
				delete_option( $feature_key );
			}
		}

		if ( ! $dry_run ) {
			// Capture total sessions after pruning and infer total before for charting.
			$remaining_sessions = $this->get_sessions_count();
			$total_before       = $remaining_sessions + (int) $pruned_count;

			$this->update_stats( $pruned_count, $total_before, $remaining_sessions, $reason_counts );

			/**
			 * Fires after sessions were pruned by Session Shredder.
			 *
			 * @param int $pruned_count Number of sessions pruned.
			 */
			do_action( 'session_shredder_pruned', $pruned_count );
		}

		do_action( 'session_shredder_after_run', $pruned_count, $dry_run );

		return $pruned_count;
	}

	/**
	 * Rule-based pruning decision for a given session.
	 *
	 * @param string $session_id Session ID.
	 * @param array  $features   Feature array.
	 *
	 * @return bool
	 */
	private function should_prune_by_rule( $session_id, array $features ) {
		unset( $session_id );

		$age_hours  = isset( $features['age_hours'] ) ? (float) $features['age_hours'] : 999;
		$pageviews  = isset( $features['pageviews'] ) ? (int) $features['pageviews'] : 0;
		$cart_value = isset( $features['cart_value'] ) ? (float) $features['cart_value'] : 0.0;
		$has_tracked_features = ! empty( $features['has_tracked_features'] );

		// Reset reason before evaluating rules; it will be set only when we decide to prune.
		$this->last_prune_reason = '';

		$hard_timeout_hours        = (float) get_option( 'session_shredder_hard_timeout_hours', 72 );
		$bounce_max_pageviews      = (int) get_option( 'session_shredder_bounce_max_pageviews', 1 );
		$bounce_max_cart_value     = (float) get_option( 'session_shredder_bounce_max_cart_value', 0 );
		$protect_min_cart_value    = (float) get_option( 'session_shredder_protect_min_cart_value', 20 );
		$hard_timeout_hours        = $hard_timeout_hours > 0 ? $hard_timeout_hours : 1;
		$bounce_max_pageviews      = $bounce_max_pageviews >= 0 ? $bounce_max_pageviews : 0;
		$bounce_max_cart_value     = $bounce_max_cart_value >= 0 ? $bounce_max_cart_value : 0;
		$protect_min_cart_value    = $protect_min_cart_value >= 0 ? $protect_min_cart_value : 0;

		$should_prune = false;

		// Protection: valuable carts are not pruned before the hard timeout.
		if ( $cart_value >= $protect_min_cart_value && $age_hours < $hard_timeout_hours ) {
			$should_prune       = false;
			$this->last_prune_reason = '';
		} elseif ( $age_hours >= $hard_timeout_hours ) {
			// Rule 1: hard timeout.
			$should_prune            = true;
			$this->last_prune_reason = 'hard_timeout';
		} elseif ( $has_tracked_features && $pageviews <= $bounce_max_pageviews && $cart_value <= $bounce_max_cart_value ) {
			// Rule 2: bounce with no meaningful cart activity.
			$should_prune            = true;
			$this->last_prune_reason = 'bounce';
		}

		unset( $features['has_tracked_features'] );

		$filtered_should_prune = (bool) apply_filters( 'session_shredder_should_prune', $should_prune, $features );
		if ( $filtered_should_prune && ! $should_prune && '' === $this->last_prune_reason ) {
			// Pruned only because a filter said so.
			$this->last_prune_reason = 'filtered';
		}

		return $filtered_should_prune;
	}

	/**
	 * Initialize session table metadata to support both legacy and 10.3+ tables.
	 *
	 * @return void
	 */
	private function init_sessions_table_meta() {
		global $wpdb;

		if ( $this->sessions_table ) {
			return;
		}

		$candidates = array(
			$wpdb->prefix . 'wc_sessions',
			$wpdb->prefix . 'woocommerce_sessions',
		);

		foreach ( $candidates as $table ) {
			$exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery
			if ( $exists === $table ) {
				$this->sessions_table = $table;
				break;
			}
		}

		if ( ! $this->sessions_table ) {
			return;
		}

		$columns = $wpdb->get_col( 'DESC ' . $this->sessions_table, 0 ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		if ( empty( $columns ) ) {
			$this->sessions_table = '';
			return;
		}

		$this->session_id_column     = in_array( 'session_id', $columns, true ) ? 'session_id' : ( in_array( 'session_key', $columns, true ) ? 'session_key' : '' );
		$this->session_value_column  = in_array( 'session_value', $columns, true ) ? 'session_value' : ( in_array( 'session_data', $columns, true ) ? 'session_data' : '' );
		$this->session_expiry_column = in_array( 'session_expiry', $columns, true ) ? 'session_expiry' : ( in_array( 'session_expiration', $columns, true ) ? 'session_expiration' : '' );
	}

	/**
	 * Delete a session row safely.
	 *
	 * @param string $sid Session ID.
	 *
	 * @return void
	 */
	private function delete_session( $sid ) {
		global $wpdb;

		if ( ! $this->sessions_table || ! $this->session_id_column ) {
			return;
		}

		$result = $wpdb->delete(
			$this->sessions_table,
			array( $this->session_id_column => $sid ),
			array( '%s' )
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		if ( false === $result ) {
			if ( function_exists( 'wc_get_logger' ) ) {
				wc_get_logger()->error( 'Failed to delete session ' . $sid . ' - ' . $wpdb->last_error, array( 'source' => 'session-shredder' ) );
			} else {
				error_log( 'Session Shredder failed deleting session ' . $sid . ': ' . $wpdb->last_error );
			}
		}
	}

	/**
	 * Update statistics and log summary.
	 *
	 * @param int $pruned       Number of pruned sessions.
	 * @param int $total_before Total sessions before the run.
	 * @param int $total_after  Total sessions after the run.
	 *
	 * @return void
	 */
	private function update_stats( $pruned, $total_before, $total_after, $reasons = array() ) {
		$pruned       = (int) $pruned;
		$total_before = (int) $total_before;
		$total_after  = (int) $total_after;
		$reasons      = is_array( $reasons ) ? $reasons : array();
		$stats = get_option(
			'session_shredder_stats',
			array(
				'pruned_total'      => 0,
				'pruned_today'      => 0,
				'pruned_today_date' => gmdate( 'Y-m-d' ),
				'last_prune'        => 0,
				'last_runs'         => array(),
			)
		);

		$today = gmdate( 'Y-m-d' );

		if ( empty( $stats['pruned_today_date'] ) || $stats['pruned_today_date'] !== $today ) {
			$stats['pruned_today_date'] = $today;
			$stats['pruned_today']      = 0;
		}

		$stats['pruned_total'] += $pruned;
		$stats['pruned_today'] += $pruned;
		$stats['last_prune']    = time();

		if ( ! isset( $stats['last_runs'] ) || ! is_array( $stats['last_runs'] ) ) {
			$stats['last_runs'] = array();
		}

		$stats['last_runs'][] = array(
			'pruned'       => $pruned,
			'total_before' => $total_before,
			'total_after'  => $total_after,
			'timestamp'    => time(),
			'reasons'      => $reasons,
		);

		// Keep at most 50 recent runs for the chart.
		if ( count( $stats['last_runs'] ) > 50 ) {
			$stats['last_runs'] = array_slice( $stats['last_runs'], -50 );
		}

		update_option( 'session_shredder_stats', $stats, false );

		if ( function_exists( 'wc_get_logger' ) ) {
			$logger = wc_get_logger();

			$reason_parts = array();
			$hard_count   = isset( $reasons['hard_timeout'] ) ? (int) $reasons['hard_timeout'] : 0;
			$bounce_count = isset( $reasons['bounce'] ) ? (int) $reasons['bounce'] : 0;
			$filtered_cnt = isset( $reasons['filtered'] ) ? (int) $reasons['filtered'] : 0;
			$other_count  = isset( $reasons['other'] ) ? (int) $reasons['other'] : 0;

			if ( $hard_count > 0 ) {
				$reason_parts[] = sprintf( '%d hard-timeout', $hard_count );
			}
			if ( $bounce_count > 0 ) {
				$reason_parts[] = sprintf( '%d bounce', $bounce_count );
			}
			if ( $filtered_cnt > 0 ) {
				$reason_parts[] = sprintf( '%d via filter', $filtered_cnt );
			}
			if ( $other_count > 0 ) {
				$reason_parts[] = sprintf( '%d other', $other_count );
			}

			$reason_text = '';
			if ( ! empty( $reason_parts ) ) {
				$reason_text = ' Reasons: ' . implode( ', ', $reason_parts ) . '.';
			}

			$logger->info(
				sprintf(
					'Pruned %d sessions (rule-based). Before: %d, after: %d.%s',
					(int) $pruned,
					(int) $total_before,
					(int) $total_after,
					$reason_text
				),
				array( 'source' => 'session-shredder' )
			);
		}
	}

	/**
	 * Get stored statistics for admin dashboard.
	 *
	 * @return array
	 */
	public function get_stats() {
		$stats = get_option( 'session_shredder_stats', array() );
		if ( ! is_array( $stats ) ) {
			$stats = array();
		}

		return $stats;
	}

	/**
	 * Get total number of sessions in the table.
	 *
	 * @return int
	 */
	public function get_sessions_count() {
		global $wpdb;

		$this->init_sessions_table_meta();

		if ( ! $this->sessions_table ) {
			return 0;
		}

		$table = esc_sql( $this->sessions_table );

		$count = (int) $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM {$table} WHERE 1 = %d",
				1
			)
		); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.DirectQuery

		return max( 0, $count );
	}

	/**
	 * Invalidate caches after pruning.
	 *
	 * @param int $count Number of pruned sessions.
	 *
	 * @return void
	 */
	public function invalidate_caches( $count ) {
		unset( $count );

		if ( function_exists( 'wp_cache_flush_group' ) ) {
			// Some cache plugins expose this.
			wp_cache_flush_group( 'wc' );
		} elseif ( function_exists( 'wp_cache_flush' ) ) {
			wp_cache_flush();
		}
	}
}
