<?php
/**
 * WP-CLI commands for Session Shredder.
 *
 * @package Session_Shredder
 */

defined( 'ABSPATH' ) || exit;

if ( defined( 'WP_CLI' ) && WP_CLI ) {

	/**
	 * Session Shredder WP-CLI commands.
	 */
	class Session_Shredder_CLI {

		/**
		 * Analyze sessions that would be pruned (dry run).
		 *
		 * ## EXAMPLES
		 *
		 *     wp session-shredder analyze
		 *
		 * @param array $args       Positional arguments (unused).
		 * @param array $assoc_args Associative arguments (unused).
		 *
		 * @return void
		 */
		public function analyze( $args, $assoc_args ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
			$this->ensure_environment();

			$core = Session_Shredder::get_instance();

			$start        = microtime( true );
			$candidate_count = $core->ai_prune_sessions( true );
			$duration     = microtime( true ) - $start;

			$total_sessions = $core->get_sessions_count();

			\WP_CLI::log( sprintf( 'Total sessions: %d', $total_sessions ) );
			\WP_CLI::log( sprintf( 'Dry run: %d session(s) would be pruned.', $candidate_count ) );
			\WP_CLI::success( sprintf( 'Analysis complete in %.3f seconds.', $duration ) );
		}

		/**
		 * Prune sessions using current rules.
		 *
		 * ## EXAMPLES
		 *
		 *     wp session-shredder prune
		 *
		 * @param array $args       Positional arguments (unused).
		 * @param array $assoc_args Associative arguments (unused).
		 *
		 * @return void
		 */
		public function prune( $args, $assoc_args ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
			$this->ensure_environment();

			$core = Session_Shredder::get_instance();

			$start    = microtime( true );
			$pruned   = $core->ai_prune_sessions( false );
			$duration = microtime( true ) - $start;

			$total_sessions = $core->get_sessions_count();

			\WP_CLI::log( sprintf( 'Total sessions after prune: %d', $total_sessions ) );
			\WP_CLI::success( sprintf( 'Pruned %d session(s) in %.3f seconds.', $pruned, $duration ) );
		}

		/**
		 * Ensure WooCommerce and Session Shredder environment is ready.
		 *
		 * @return void
		 */
		private function ensure_environment() {
			if ( ! class_exists( 'WooCommerce' ) ) {
				\WP_CLI::error( 'WooCommerce is not active.' );
			}

			if ( defined( 'WC_VERSION' ) ) {
				$wc_version = WC_VERSION;
			} elseif ( function_exists( 'WC' ) && WC() instanceof WooCommerce ) {
				$wc_version = WC()->version;
			} else {
				$wc_version = null;
			}

			if ( empty( $wc_version ) || version_compare( $wc_version, '10.3', '<' ) ) {
				\WP_CLI::error( 'Session Shredder requires WooCommerce 10.3 or newer.' );
			}
		}
	}

	\WP_CLI::add_command( 'session-shredder', 'Session_Shredder_CLI' );
}
