<?php
/**
 * WooCommerce settings integration for Session Shredder.
 *
 * @package Session_Shredder
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Session_Shredder_Admin_Settings' ) ) {

	class Session_Shredder_Admin_Settings {

		/**
		 * Tab ID.
		 *
		 * @var string
		 */
		const TAB_ID = 'session_shredder';

		/**
		 * Constructor.
		 */
		public function __construct() {
			add_filter( 'woocommerce_settings_tabs_array', array( $this, 'add_settings_tab' ), 70 );
			add_action( 'woocommerce_settings_tabs_' . self::TAB_ID, array( $this, 'render_settings' ) );
			add_action( 'woocommerce_update_options_' . self::TAB_ID, array( $this, 'save_settings' ) );
		}

		/**
		 * Add custom tab to WooCommerce settings.
		 *
		 * @param array $tabs Existing tabs.
		 *
		 * @return array
		 */
		public function add_settings_tab( $tabs ) {
			$tabs[ self::TAB_ID ] = __( 'Session Shredder', 'session-shredder-for-woocommerce' );
			return $tabs;
		}

		/**
		 * Render settings fields.
		 *
		 * @return void
		 */
		public function render_settings() {
			$settings = $this->get_settings();

			if ( class_exists( 'WC_Admin_Settings' ) ) {
				WC_Admin_Settings::output_fields( $settings );
			}
		}

		/**
		 * Save settings.
		 *
		 * @return void
		 */
		public function save_settings() {
			$settings = $this->get_settings();

			if ( class_exists( 'WC_Admin_Settings' ) ) {
				WC_Admin_Settings::save_fields( $settings );
			}
		}

		/**
		 * Define settings fields.
		 *
		 * @return array
		 */
		private function get_settings() {
			return array(
				array(
					'title' => __( 'Session Shredder Rules', 'session-shredder-for-woocommerce' ),
					'type'  => 'title',
					'id'    => 'session_shredder_settings_section',
					'desc'  => __( 'Configure how Session Shredder evaluates and prunes old sessions. Defaults are conservative to avoid deleting valuable carts.', 'session-shredder-for-woocommerce' ),
				),
				array(
					'title'             => __( 'Base age before pruning (hours)', 'session-shredder-for-woocommerce' ),
					'desc'              => __( 'Minimum age before any session is even considered as a prune candidate. Younger sessions are always kept, even if they look like bounces.', 'session-shredder-for-woocommerce' ),
					'id'                => 'session_shredder_base_age_hours',
					'default'           => 48,
					'type'              => 'number',
					'custom_attributes' => array(
						'min'  => '1',
						'step' => '1',
					),
					'css'               => 'width: 80px;',
					'desc_tip'          => __( 'Lower values prune more aggressively; higher values keep sessions around for longer.', 'session-shredder-for-woocommerce' ),
				),
				array(
					'title'             => __( 'Hard timeout age (hours)', 'session-shredder-for-woocommerce' ),
					'desc'              => __( 'Absolute maximum age for any session. Once a session reaches this age it becomes eligible for pruning even if it had some activity.', 'session-shredder-for-woocommerce' ),
					'id'                => 'session_shredder_hard_timeout_hours',
					'default'           => 72,
					'type'              => 'number',
					'custom_attributes' => array(
						'min'  => '1',
						'step' => '1',
					),
					'css'               => 'width: 80px;',
					'desc_tip'          => __( 'Use this as a safety net so very old sessions are eventually cleared, even if they carried some value.', 'session-shredder-for-woocommerce' ),
				),
				array(
					'title'             => __( 'Bounce max pageviews', 'session-shredder-for-woocommerce' ),
					'desc'              => __( 'If a session has pageviews at or below this number and low cart value, it is treated as a bounce and can be pruned once it is old enough.', 'session-shredder-for-woocommerce' ),
					'id'                => 'session_shredder_bounce_max_pageviews',
					'default'           => 1,
					'type'              => 'number',
					'custom_attributes' => array(
						'min'  => '0',
						'step' => '1',
					),
					'css'               => 'width: 80px;',
					'desc_tip'          => __( 'Lower values mean only very shallow visits are pruned as bounces.', 'session-shredder-for-woocommerce' ),
				),
				array(
					'title'             => __( 'Bounce max cart value', 'session-shredder-for-woocommerce' ),
					'desc'              => __( 'Maximum cart value for a session to still be treated as a bounce (in store currency units). Higher-value carts will no longer be treated as simple bounces.', 'session-shredder-for-woocommerce' ),
					'id'                => 'session_shredder_bounce_max_cart_value',
					'default'           => 0,
					'type'              => 'number',
					'custom_attributes' => array(
						'min'  => '0',
						'step' => '1',
					),
					'css'               => 'width: 100px;',
					'desc_tip'          => __( 'Keep this low to avoid pruning sessions that had non-trivial cart activity.', 'session-shredder-for-woocommerce' ),
				),
				array(
					'title'             => __( 'Protect sessions with cart value above', 'session-shredder-for-woocommerce' ),
					'desc'              => __( 'Sessions with cart value equal or higher than this amount are protected from pruning until they reach the hard timeout age.', 'session-shredder-for-woocommerce' ),
					'id'                => 'session_shredder_protect_min_cart_value',
					'default'           => 20,
					'type'              => 'number',
					'custom_attributes' => array(
						'min'  => '0',
						'step' => '1',
					),
					'css'               => 'width: 100px;',
					'desc_tip'          => __( 'Raise this if you want to be extra cautious with medium-to-high value carts.', 'session-shredder-for-woocommerce' ),
				),
				array(
					'type' => 'sectionend',
					'id'   => 'session_shredder_settings_section',
				),
			);
		}
	}
}
