<?php
/**
 * Admin dashboard for Session Shredder.
 *
 * @package Session_Shredder
 */

defined( 'ABSPATH' ) || exit;

class Session_Shredder_Admin_Dashboard {

	/**
	 * Core plugin instance.
	 *
	 * @var Session_Shredder
	 */
	private $core;

	/**
	 * Constructor.
	 *
	 * @param Session_Shredder $core Core plugin instance.
	 */
	public function __construct( Session_Shredder $core ) {
		$this->core = $core;

		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'admin_init', array( $this, 'handle_manual_prune' ) );
		add_action( 'wp_ajax_session_shredder_get_stats', array( $this, 'ajax_get_stats' ) );
	}

	/**
	 * Add the submenu under WooCommerce.
	 *
	 * @return void
	 */
	public function add_admin_menu() {
		add_submenu_page(
			'woocommerce',
			__( 'Session Shredder', 'session-shredder-for-woocommerce' ),
			__( 'Session Shredder', 'session-shredder-for-woocommerce' ),
			'manage_woocommerce',
			'session-shredder',
			array( $this, 'render' )
		);
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current admin page hook.
	 *
	 * @return void
	 */
	public function enqueue_assets( $hook ) {
		if ( 'woocommerce_page_session-shredder' !== $hook ) {
			return;
		}

		wp_enqueue_style(
			'session-shredder-admin',
			SESSION_SHREDDER_URL . 'assets/admin.css',
			array(),
			SESSION_SHREDDER_VERSION
		);

		wp_enqueue_script(
			'session-shredder-chartjs',
			SESSION_SHREDDER_URL . 'assets/chart.umd.min.js',
			array(),
			'4.4.0',
			true
		);

		wp_enqueue_script(
			'session-shredder-admin',
			SESSION_SHREDDER_URL . 'assets/admin.js',
			array( 'session-shredder-chartjs', 'jquery' ),
			SESSION_SHREDDER_VERSION,
			true
		);

		wp_localize_script(
			'session-shredder-admin',
			'session_shredder_ajax',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'session-shredder-stats' ),
			)
		);
	}

	/**
	 * Handle manual prune form submission.
	 *
	 * @return void
	 */
	public function handle_manual_prune() {
		if ( ! isset( $_POST['session_shredder_manual_prune'] ) && ! isset( $_POST['session_shredder_dry_run'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			return;
		}

		if ( ! isset( $_POST['session_shredder_manual_prune_nonce'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			return;
		}

		$nonce = sanitize_text_field( wp_unslash( $_POST['session_shredder_manual_prune_nonce'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing

		if ( ! wp_verify_nonce( $nonce, 'session_shredder_manual_prune_action' ) ) {
			return;
		}

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return;
		}

		$is_dry_run = isset( $_POST['session_shredder_dry_run'] ); // phpcs:ignore WordPress.Security.NonceVerification.Missing
		$count     = $this->core->ai_prune_sessions( $is_dry_run );

		if ( $is_dry_run ) {
			$message = sprintf(
				/* translators: %d: number of sessions that would be pruned */
				_n( 'Dry run: %d session would be pruned.', 'Dry run: %d sessions would be pruned.', $count, 'session-shredder-for-woocommerce' ),
				$count
			);
		} else {
			$message = sprintf(
				/* translators: %d: number of pruned sessions */
				_n( 'Pruned %d session.', 'Pruned %d sessions.', $count, 'session-shredder-for-woocommerce' ),
				$count
			);
		}

		add_settings_error(
			'session-shredder',
			'session-shredder-manual-prune',
			$message,
			'updated'
		);
	}

	/**
	 * Render the dashboard page.
	 *
	 * @return void
	 */
	public function render() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'You do not have permission to access this page.', 'session-shredder-for-woocommerce' ) );
		}

		$total_sessions = $this->core->get_sessions_count();
		$stats          = $this->core->get_stats();

		include SESSION_SHREDDER_PATH . 'includes/admin/partials/dashboard-template.php';
	}

	/**
	 * AJAX endpoint for fetching stats used by the admin chart.
	 *
	 * @return void
	 */
	public function ajax_get_stats() {
		check_ajax_referer( 'session-shredder-stats', 'nonce' );

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'session-shredder-for-woocommerce' ) ), 403 );
		}

		$stats = $this->core->get_stats();
		
		$values_pruned = array();
		$values_total  = array();
		$labels        = array();
		
		if ( ! empty( $stats['last_runs'] ) && is_array( $stats['last_runs'] ) ) {
			$i = 1;
			foreach ( $stats['last_runs'] as $row ) {
				$values_pruned[] = isset( $row['pruned'] ) ? (int) $row['pruned'] : 0;
				$values_total[]  = isset( $row['total_after'] ) ? (int) $row['total_after'] : 0;
				$labels[]        = isset( $row['timestamp'] ) ? gmdate( 'H:i', (int) $row['timestamp'] ) : '#' . $i;
				$i++;
			}
		}
		
		wp_send_json_success(
			array(
				'values'       => $values_pruned,
				'values_total' => $values_total,
				'labels'       => $labels,
			)
		);
	}
}
