<?php
/**
 * Plugin Name: Server IP Dashboard
 * Description: Displays the server’s Internal IP and the site’s External/Public IP on the WordPress admin dashboard (admin-only by default). Includes AJAX refresh and a settings page.
 * Version: 1.0.1
 * Author: Md Fahad
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: server-ip-dashboard
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Prevent direct access.
}

/*-----------------------------------------------------------------------------
 * Constants
 *---------------------------------------------------------------------------*/
define( 'SRVIP_VERSION', '1.0.1' );
define( 'SRVIP_OPTION',  'srvip_settings' );

/*-----------------------------------------------------------------------------
 * Defaults
 *---------------------------------------------------------------------------*/
function srvip_default_settings() {
	return array(
		'refresh_interval' => 0, // seconds; 0 = disabled
		'allow_editors'    => 0, // 0/1
		'api_url'          => 'https://api.ipify.org?format=json',
	);
}

/*-----------------------------------------------------------------------------
 * Activation: seed defaults
 *---------------------------------------------------------------------------*/
register_activation_hook( __FILE__, function () {
	$opts = get_option( SRVIP_OPTION );
	if ( ! is_array( $opts ) ) {
		update_option( SRVIP_OPTION, srvip_default_settings(), false );
	} else {
		update_option( SRVIP_OPTION, wp_parse_args( $opts, srvip_default_settings() ), false );
	}
} );

/*-----------------------------------------------------------------------------
 * Uninstall cleanup (handled by uninstall.php)
 *---------------------------------------------------------------------------*/
/**
 * If you have uninstall.php (recommended), DO NOT also register an uninstall hook here.
 * WordPress will call uninstall.php automatically.
 * If you do NOT ship uninstall.php, you can uncomment the lines below.
 *
 * function srvip_uninstall() {
 * 	delete_option( SRVIP_OPTION );
 * 	delete_transient( 'srvip_external_ip' );
 * }
 * register_uninstall_hook( __FILE__, 'srvip_uninstall' );
 */

/*-----------------------------------------------------------------------------
 * Capability helper
 *---------------------------------------------------------------------------*/
function srvip_user_can_view() {
	if ( current_user_can( 'manage_options' ) ) {
		return true;
	}
	$opts = get_option( SRVIP_OPTION, srvip_default_settings() );
	return ( ! empty( $opts['allow_editors'] ) && current_user_can( 'edit_pages' ) );
}

/*-----------------------------------------------------------------------------
 * Dashboard widget
 *---------------------------------------------------------------------------*/
add_action( 'wp_dashboard_setup', function () {
	if ( ! is_admin() || ! srvip_user_can_view() ) {
		return;
	}
	wp_add_dashboard_widget(
		'srvip_widget',
		esc_html__( 'Server IP Information', 'server-ip-dashboard' ),
		'srvip_render_dashboard_widget'
	);
} );

function srvip_render_dashboard_widget() {
	$nonce = wp_create_nonce( 'srvip_ip_nonce' );
	echo '<div id="srvip-widget" class="srvip-widget">';
	echo '<p><strong>' . esc_html__( 'Server Internal IP:', 'server-ip-dashboard' ) . '</strong> <span id="srvip-internal">' . esc_html__( 'Fetching…', 'server-ip-dashboard' ) . '</span></p>';
	echo '<p><strong>' . esc_html__( 'External/Public IP:', 'server-ip-dashboard' ) . '</strong> <span id="srvip-external">' . esc_html__( 'Fetching…', 'server-ip-dashboard' ) . '</span></p>';
	echo '<p><button id="srvip-refresh" class="button">' . esc_html__( 'Refresh IP Info', 'server-ip-dashboard' ) . '</button></p>';
	echo '<input type="hidden" id="srvip-nonce" value="' . esc_attr( $nonce ) . '">';
	echo '<noscript><em>' . esc_html__( 'Enable JavaScript to fetch IP information.', 'server-ip-dashboard' ) . '</em></noscript>';
	echo '</div>';
}

/*-----------------------------------------------------------------------------
 * Admin assets – load only on Dashboard
 *---------------------------------------------------------------------------*/
add_action( 'admin_enqueue_scripts', function ( $hook ) {
	if ( 'index.php' !== $hook || ! srvip_user_can_view() ) {
		return;
	}

	// JS
	wp_enqueue_script(
		'srvip-admin',
		plugins_url( 'admin/admin.js', __FILE__ ),
		array( 'jquery' ),
		SRVIP_VERSION,
		true
	);

	$opts     = get_option( SRVIP_OPTION, srvip_default_settings() );
	$interval = isset( $opts['refresh_interval'] ) ? absint( $opts['refresh_interval'] ) : 0;

	wp_localize_script(
		'srvip-admin',
		'SrvipData',
		array(
			'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
			'nonce'           => wp_create_nonce( 'srvip_ip_nonce' ),
			'refreshInterval' => $interval,
			'updatedLabel'    => __( 'Last updated:', 'server-ip-dashboard' ),
			'errorLabel'      => __( 'Error fetching IP', 'server-ip-dashboard' ),
		)
	);

	// Optional CSS
	wp_enqueue_style(
		'srvip-admin',
		plugins_url( 'admin/admin.css', __FILE__ ),
		array(),
		SRVIP_VERSION
	);
} );

/*-----------------------------------------------------------------------------
 * AJAX: fetch internal + external IPs
 *---------------------------------------------------------------------------*/
add_action( 'wp_ajax_srvip_get_ips', 'srvip_handle_get_ips' );
function srvip_handle_get_ips() {
	check_ajax_referer( 'srvip_ip_nonce', 'security' );
	if ( ! srvip_user_can_view() ) {
		wp_send_json_error( array( 'message' => 'Unauthorized' ), 403 );
	}

	// Internal IP with fallbacks (no superglobal warning).
	$internal_ip = '';

	// Preferred: avoid direct $_SERVER; validate via filter_input.
	$fi_ip = filter_input( INPUT_SERVER, 'SERVER_ADDR', FILTER_VALIDATE_IP );
	if ( $fi_ip ) {
		$internal_ip = $fi_ip;
	}

	// Fallback: $_SERVER with unslash + validate (line-scoped ignore only on read).
	if ( '' === $internal_ip && isset( $_SERVER['SERVER_ADDR'] ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$raw = wp_unslash( $_SERVER['SERVER_ADDR'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$val = filter_var( $raw, FILTER_VALIDATE_IP );
		if ( $val ) {
			$internal_ip = $val;
		}
	}

	// Hostname resolution fallback.
	if ( '' === $internal_ip ) {
		$host = function_exists( 'gethostname' ) ? gethostname() : php_uname( 'n' );
		if ( $host ) {
			$resolved = gethostbyname( $host );
			if ( $resolved && $resolved !== $host && filter_var( $resolved, FILTER_VALIDATE_IP ) ) {
				$internal_ip = $resolved;
			}
		}
	}

	if ( '' === $internal_ip ) {
		$internal_ip = __( 'Unavailable', 'server-ip-dashboard' );
	}

	// External IP with transient cache.
	$opts          = get_option( SRVIP_OPTION, srvip_default_settings() );
	$api_url       = ! empty( $opts['api_url'] ) ? $opts['api_url'] : 'https://api.ipify.org?format=json';
	$cache_seconds = max( 30, absint( isset( $opts['refresh_interval'] ) ? $opts['refresh_interval'] : 300 ) ); // at least 30s

	$external_ip = get_transient( 'srvip_external_ip' );
	if ( false === $external_ip ) {
		$response = wp_remote_get(
			$api_url,
			array(
				'timeout'   => 8,
				'sslverify' => true,
				'headers'   => array( 'Accept' => 'application/json, text/plain, */*' ),
			)
		);

		if ( is_wp_error( $response ) ) {
			$external_ip = __( 'Unavailable', 'server-ip-dashboard' );
		} else {
			$body = wp_remote_retrieve_body( $response );
			$body = is_string( $body ) ? trim( $body ) : '';

			$ip = '';
			if ( '' !== $body ) {
				// Try JSON first ({"ip":"x.x.x.x"}), else plain text IP.
				$data = json_decode( $body, true );
				if ( is_array( $data ) && ! empty( $data['ip'] ) && filter_var( $data['ip'], FILTER_VALIDATE_IP ) ) {
					$ip = $data['ip'];
				} elseif ( filter_var( $body, FILTER_VALIDATE_IP ) ) {
					$ip = $body;
				}
			}
			$external_ip = $ip ? $ip : __( 'Unavailable', 'server-ip-dashboard' );
		}

		set_transient( 'srvip_external_ip', $external_ip, $cache_seconds );
	}

	wp_send_json_success(
		array(
			'internal_ip' => esc_html( $internal_ip ),
			'external_ip' => esc_html( $external_ip ),
		)
	);
}

/*-----------------------------------------------------------------------------
 * Settings page (Settings → Server IP Dashboard)
 *---------------------------------------------------------------------------*/
add_action( 'admin_menu', function () {
	// WordPress will also restrict menu visibility by capability,
	// but we guard here as well to avoid rendering for non-admins.
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	add_options_page(
		__( 'Server IP Dashboard Settings', 'server-ip-dashboard' ),
		__( 'Server IP Dashboard', 'server-ip-dashboard' ),
		'manage_options',
		'srvip-settings',
		'srvip_render_settings_page'
	);
} );

/*-----------------------------------------------------------------------------
 * Settings API registration
 *---------------------------------------------------------------------------*/
add_action( 'admin_init', function () {
	register_setting(
		'srvip_option_group',
		SRVIP_OPTION,
		array(
			'type'              => 'array',
			'sanitize_callback' => 'srvip_sanitize_settings',
			'default'           => srvip_default_settings(),
		)
	);

	add_settings_section(
		'srvip_section_general',
		__( 'General', 'server-ip-dashboard' ),
		function () {
			echo '<p>' . esc_html__( 'Configure how the dashboard widget behaves.', 'server-ip-dashboard' ) . '</p>';
		},
		'srvip-settings'
	);

	add_settings_field(
		'refresh_interval',
		__( 'Auto-Refresh Interval (seconds)', 'server-ip-dashboard' ),
		'srvip_field_refresh_interval',
		'srvip-settings',
		'srvip_section_general'
	);

	add_settings_field(
		'allow_editors',
		__( 'Allow Editors to View', 'server-ip-dashboard' ),
		'srvip_field_allow_editors',
		'srvip-settings',
		'srvip_section_general'
	);

	add_settings_field(
		'api_url',
		__( 'External IP API URL', 'server-ip-dashboard' ),
		'srvip_field_api_url',
		'srvip-settings',
		'srvip_section_general'
	);
} );

function srvip_sanitize_settings( $input ) {
	$defaults = srvip_default_settings();
	$out      = is_array( $input ) ? $input : array();

	$out['refresh_interval'] = isset( $out['refresh_interval'] ) ? absint( $out['refresh_interval'] ) : $defaults['refresh_interval'];
	if ( $out['refresh_interval'] > 86400 ) { // clamp to 24h max
		$out['refresh_interval'] = 86400;
	}

	$out['allow_editors'] = ! empty( $out['allow_editors'] ) ? 1 : 0;

	$api_url = isset( $out['api_url'] ) ? trim( $out['api_url'] ) : '';
	if ( '' === $api_url ) {
		$out['api_url'] = $defaults['api_url'];
	} else {
		$san = esc_url_raw( $api_url );
		$out['api_url'] = $san ? $san : $defaults['api_url'];
	}

	return wp_parse_args( $out, $defaults );
}

function srvip_field_refresh_interval() {
	$opts = get_option( SRVIP_OPTION, srvip_default_settings() );
	$val  = isset( $opts['refresh_interval'] ) ? absint( $opts['refresh_interval'] ) : 0;
	echo '<input type="number" min="0" step="1" name="' . esc_attr( SRVIP_OPTION ) . '[refresh_interval]" value="' . esc_attr( $val ) . '" />';
	echo '<p class="description">' . esc_html__( '0 disables auto-refresh. Recommended minimum: 60 seconds.', 'server-ip-dashboard' ) . '</p>';
}

function srvip_field_allow_editors() {
	$opts  = get_option( SRVIP_OPTION, srvip_default_settings() );
	$allow = ! empty( $opts['allow_editors'] );
	echo '<label><input type="checkbox" name="' . esc_attr( SRVIP_OPTION ) . '[allow_editors]" value="1" ' . checked( $allow, true, false ) . ' /> ';
	echo esc_html__( 'Also allow users with the Editor role (capability: edit_pages) to view the widget.', 'server-ip-dashboard' ) . '</label>';
}

function srvip_field_api_url() {
	$opts = get_option( SRVIP_OPTION, srvip_default_settings() );
	$val  = ! empty( $opts['api_url'] ) ? $opts['api_url'] : 'https://api.ipify.org?format=json';
	echo '<input type="url" class="regular-text" name="' . esc_attr( SRVIP_OPTION ) . '[api_url]" value="' . esc_attr( $val ) . '" />';
	echo '<p class="description">' . esc_html__( 'Default expects JSON with an "ip" field. For plain text IP endpoints, the plugin will auto-detect.', 'server-ip-dashboard' ) . '</p>';
}

/*-----------------------------------------------------------------------------
 * Render settings page
 *---------------------------------------------------------------------------*/
function srvip_render_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}
	echo '<div class="wrap">';
	echo '<h1>' . esc_html__( 'Server IP Dashboard Settings', 'server-ip-dashboard' ) . '</h1>';
	echo '<form action="options.php" method="post">';
	settings_fields( 'srvip_option_group' );
	do_settings_sections( 'srvip-settings' );
	submit_button();
	echo '</form>';
	echo '</div>';
}
