<?php
/**
 * IP Fetcher helper functions for Server IP Dashboard plugin.
 *
 * File: /includes/ip-fetcher.php
 *
 * Provides reusable functions to get internal and external IP addresses.
 *
 * @package Server_IP_Dashboard\Includes
 */

defined( 'ABSPATH' ) || exit; // Prevent direct access.

/** Constants (in case not defined elsewhere). */
if ( ! defined( 'SRVIP_EXTERNAL_IP_TRANSIENT' ) ) {
	define( 'SRVIP_EXTERNAL_IP_TRANSIENT', 'srvip_external_ip' );
}
if ( ! defined( 'SRVIP_DEFAULT_EXT_API' ) ) {
	define( 'SRVIP_DEFAULT_EXT_API', 'https://api.ipify.org?format=json' );
}
if ( ! defined( 'SRVIP_OPTION_KEY' ) ) {
	define( 'SRVIP_OPTION_KEY', 'srvip_settings' );
}

/**
 * Get plugin settings with defaults.
 *
 * @return array
 */
function srvip_get_settings() {
	$defaults = array(
		'refresh_interval' => 0,
		'visible_roles'    => array(),
		'external_api_url' => SRVIP_DEFAULT_EXT_API,
	);

	$opts = get_option( SRVIP_OPTION_KEY, array() );
	if ( ! is_array( $opts ) ) {
		$opts = array();
	}

	$opts = wp_parse_args( $opts, $defaults );

	// Normalize.
	$opts['refresh_interval'] = max( 0, absint( $opts['refresh_interval'] ) );

	if ( ! is_array( $opts['visible_roles'] ) ) {
		$opts['visible_roles'] = array();
	} else {
		$opts['visible_roles'] = array_values( array_filter( array_map( 'sanitize_key', $opts['visible_roles'] ) ) );
	}

	$api = isset( $opts['external_api_url'] ) ? trim( (string) $opts['external_api_url'] ) : '';
	$opts['external_api_url'] = $api ? $api : SRVIP_DEFAULT_EXT_API;

	return $opts;
}

/**
 * Resolve the server's internal IP address.
 *
 * @return string|null
 */
function srvip_get_internal_ip() {
	// Preferred: avoid direct superglobal; validate via filter_input.
	$ip = filter_input( INPUT_SERVER, 'SERVER_ADDR', FILTER_VALIDATE_IP );
	if ( $ip ) {
		return (string) $ip; // (Escaped on output by caller.)
	}

	// Fallback: use $_SERVER with explicit unslash + validate (line-scoped ignore for the read).
	if ( isset( $_SERVER['SERVER_ADDR'] ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$raw = wp_unslash( $_SERVER['SERVER_ADDR'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$ip  = filter_var( $raw, FILTER_VALIDATE_IP );
		if ( $ip ) {
			return (string) $ip;
		}
	}

	// Fallback: hostname resolution.
	$host = function_exists( 'gethostname' ) ? gethostname() : php_uname( 'n' );
	if ( $host ) {
		$resolved = gethostbyname( $host );
		if ( $resolved && $resolved !== $host && filter_var( $resolved, FILTER_VALIDATE_IP ) ) {
			return (string) $resolved;
		}
	}

	// Fallback: UDP socket probe.
	$socket = @stream_socket_client( 'udp://8.8.8.8:53', $errno, $errstr, 1 );
	if ( $socket ) {
		$address = stream_socket_get_name( $socket, false ); // "192.168.1.10:54321"
		if ( $address && strpos( $address, ':' ) !== false ) {
			$parts = explode( ':', $address );
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- closing a UDP socket (not file I/O), WP_Filesystem is not applicable.
			@fclose( $socket );
			return $parts[0];
		}
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- closing a UDP socket (not file I/O), WP_Filesystem is not applicable.
		@fclose( $socket );
	}

	return null;
}

/**
 * Resolve the server's external/public IP.
 *
 * Uses a transient cache (5 minutes by default).
 *
 * @return string|null
 */
function srvip_get_external_ip() {
	// Use cached value if available.
	$cached = get_transient( SRVIP_EXTERNAL_IP_TRANSIENT );
	if ( is_string( $cached ) && $cached ) {
		return $cached;
	}

	$settings = srvip_get_settings();
	$url      = esc_url_raw( $settings['external_api_url'] );

	if ( empty( $url ) ) {
		$url = SRVIP_DEFAULT_EXT_API;
	}

	$args = array(
		'timeout' => 5,
		'headers' => array(
			'Accept' => 'application/json, text/plain, */*',
		),
	);

	$response = wp_remote_get( $url, $args );

	if ( is_wp_error( $response ) ) {
		return null;
	}

	$code = (int) wp_remote_retrieve_response_code( $response );
	if ( $code < 200 || $code >= 300 ) {
		return null;
	}

	$body = wp_remote_retrieve_body( $response );
	if ( ! $body ) {
		return null;
	}

	$ip = null;

	// Try JSON first.
	$data = json_decode( $body, true );
	if ( is_array( $data ) && ! empty( $data['ip'] ) && is_string( $data['ip'] ) && filter_var( $data['ip'], FILTER_VALIDATE_IP ) ) {
		$ip = $data['ip'];
	} else {
		// Fallback: plain text IP.
		$maybe = trim( wp_strip_all_tags( $body ) );
		if ( $maybe && filter_var( $maybe, FILTER_VALIDATE_IP ) ) {
			$ip = $maybe;
		}
	}

	// Cache for 5 minutes.
	if ( $ip ) {
		set_transient( SRVIP_EXTERNAL_IP_TRANSIENT, $ip, 5 * MINUTE_IN_SECONDS );
	}

	return $ip ? $ip : null;
}
