<?php
/**
 * Settings Page for "Server IP Dashboard".
 *
 * File: /admin/settings-page.php
 *
 * Responsibilities:
 * - Register Settings > Server IP Dashboard page (admin only).
 * - Register settings, sections, and fields using the Settings API.
 * - Sanitize and validate inputs.
 *
 * @package Server_IP_Dashboard\Admin
 */

defined( 'ABSPATH' ) || exit; // Prevent direct access.

/**
 * Constants shared across plugin files (define if missing).
 */
if ( ! defined( 'SRVIP_OPTION_KEY' ) ) {
	define( 'SRVIP_OPTION_KEY', 'srvip_settings' );
}
if ( ! defined( 'SRVIP_DEFAULT_EXT_API' ) ) {
	define( 'SRVIP_DEFAULT_EXT_API', 'https://api.ipify.org?format=json' );
}

/**
 * Safe getter for settings with sane defaults.
 * Falls back if srvip_get_settings() (from widget file) isn't loaded yet.
 *
 * @return array
 */
if ( ! function_exists( 'srvip_get_settings' ) ) {
	function srvip_get_settings() {
		$defaults = array(
			'refresh_interval' => 0,
			'visible_roles'    => array(), // admins always allowed implicitly
			'external_api_url' => SRVIP_DEFAULT_EXT_API,
		);

		$opts = get_option( SRVIP_OPTION_KEY, array() );
		if ( ! is_array( $opts ) ) {
			$opts = array();
		}
		$opts = wp_parse_args( $opts, $defaults );

		// Normalize.
		$opts['refresh_interval'] = max( 0, absint( $opts['refresh_interval'] ) );
		if ( ! is_array( $opts['visible_roles'] ) ) {
			$opts['visible_roles'] = array();
		} else {
			$opts['visible_roles'] = array_values( array_filter( array_map( 'sanitize_key', $opts['visible_roles'] ) ) );
		}
		$api = isset( $opts['external_api_url'] ) ? trim( (string) $opts['external_api_url'] ) : '';
		$opts['external_api_url'] = $api ? $api : SRVIP_DEFAULT_EXT_API;

		return $opts;
	}
}

/**
 * Register the Settings page (menu).
 */
add_action( 'admin_menu', 'srvip_register_settings_page' );
function srvip_register_settings_page() {
	add_options_page(
		__( 'Server IP Dashboard Settings', 'server-ip-dashboard' ),
		__( 'Server IP Dashboard', 'server-ip-dashboard' ),
		'manage_options',                       // Admins only
		'srvip-settings',
		'srvip_render_settings_page'
	);
}

/**
 * Register settings, section, and fields.
 */
add_action( 'admin_init', 'srvip_register_settings' );
function srvip_register_settings() {

	// Register main option (array) with sanitize callback.
	register_setting(
		'srvip_option_group',
		SRVIP_OPTION_KEY,
		array(
			'type'              => 'array',
			'sanitize_callback' => 'srvip_sanitize_settings',
			'show_in_rest'      => false,
		)
	);

	// Section.
	add_settings_section(
		'srvip_section_general',
		__( 'General Settings', 'server-ip-dashboard' ),
		function () {
			echo '<p>' . esc_html__( 'Control who can see the dashboard widget and how often it refreshes, and set the external IP API endpoint.', 'server-ip-dashboard' ) . '</p>';
		},
		'srvip-settings'
	);

	// Field: Refresh Interval.
	add_settings_field(
		'srvip_refresh_interval',
		__( 'Auto-refresh interval (seconds)', 'server-ip-dashboard' ),
		'srvip_field_refresh_interval_cb',
		'srvip-settings',
		'srvip_section_general'
	);

	// Field: Visible Roles.
	add_settings_field(
		'srvip_visible_roles',
		__( 'Visible to additional roles', 'server-ip-dashboard' ),
		'srvip_field_visible_roles_cb',
		'srvip-settings',
		'srvip_section_general'
	);

	// Field: External API URL.
	add_settings_field(
		'srvip_external_api_url',
		__( 'External IP API URL', 'server-ip-dashboard' ),
		'srvip_field_external_api_url_cb',
		'srvip-settings',
		'srvip_section_general'
	);
}

/**
 * Sanitize callback for all settings.
 *
 * @param array $input Raw input from POST.
 * @return array Sanitized settings.
 */
function srvip_sanitize_settings( $input ) {
	$clean  = array();
	$errors = array();

	// Refresh interval (0 disables auto-refresh). Enforce bounds (0..86400).
	$interval = isset( $input['refresh_interval'] ) ? absint( $input['refresh_interval'] ) : 0;
	if ( $interval < 0 ) {
		$interval = 0;
	}
	if ( $interval > DAY_IN_SECONDS ) {
		$interval = DAY_IN_SECONDS; // cap at 24h
		add_settings_error(
			SRVIP_OPTION_KEY,
			'srvip_refresh_interval_capped',
			sprintf(
				/* translators: %d: max seconds */
				esc_html__( 'Auto-refresh interval capped at %d seconds (24 hours) for safety.', 'server-ip-dashboard' ),
				DAY_IN_SECONDS
			),
			'warning'
		);
	}
	$clean['refresh_interval'] = $interval;

	// Visible roles (checkbox array). Admins always allowed implicitly; we don't need to store 'administrator'.
	$clean['visible_roles'] = array();
	if ( isset( $input['visible_roles'] ) && is_array( $input['visible_roles'] ) ) {
		$all_roles = array_keys( (array) wp_roles()->roles );
		$allowed   = array_map( 'sanitize_key', (array) $input['visible_roles'] );
		foreach ( $allowed as $role_slug ) {
			if ( in_array( $role_slug, $all_roles, true ) && 'administrator' !== $role_slug ) {
				$clean['visible_roles'][] = $role_slug;
			}
		}
		$clean['visible_roles'] = array_values( array_unique( $clean['visible_roles'] ) );
	}

	// External API URL.
	$url_in = isset( $input['external_api_url'] ) ? trim( (string) $input['external_api_url'] ) : '';
	if ( '' === $url_in ) {
		$clean['external_api_url'] = SRVIP_DEFAULT_EXT_API;
	} else {
		$url_sanitized = esc_url_raw( $url_in );
		// Basic scheme check (http/https). Prefer https; allow http with a warning.
		$scheme = wp_parse_url( $url_sanitized, PHP_URL_SCHEME );

		if ( ! $url_sanitized || ! in_array( $scheme, array( 'http', 'https' ), true ) ) {
			$clean['external_api_url'] = SRVIP_DEFAULT_EXT_API;
			add_settings_error(
				SRVIP_OPTION_KEY,
				'srvip_api_url_invalid',
				esc_html__( 'External IP API URL was invalid. Reverted to default.', 'server-ip-dashboard' ),
				'error'
			);
		} else {
			$clean['external_api_url'] = $url_sanitized;
			if ( 'http' === $scheme ) {
				add_settings_error(
					SRVIP_OPTION_KEY,
					'srvip_api_url_insecure',
					esc_html__( 'Warning: Using a non-HTTPS API URL. Consider switching to HTTPS for security.', 'server-ip-dashboard' ),
					'warning'
				);
			}
		}
	}

	return $clean;
}

/**
 * Render the settings page.
 */
function srvip_render_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'server-ip-dashboard' ) );
	}

	// Print any validation messages from add_settings_error().
	echo '<div class="wrap">';
	echo '<h1>' . esc_html__( 'Server IP Dashboard Settings', 'server-ip-dashboard' ) . '</h1>';

	// Output settings errors (if any).
	settings_errors( SRVIP_OPTION_KEY );

	echo '<form method="post" action="options.php">';
		// Prints nonce, action, and option_page fields for the settings page.
		settings_fields( 'srvip_option_group' );
		// Output sections and fields.
		do_settings_sections( 'srvip-settings' );
		submit_button( __( 'Save Changes', 'server-ip-dashboard' ) );
	echo '</form>';

	echo '</div>';
}

/**
 * Field callback: Refresh interval input.
 */
function srvip_field_refresh_interval_cb() {
	$settings = srvip_get_settings();
	$val      = isset( $settings['refresh_interval'] ) ? (int) $settings['refresh_interval'] : 0;

	echo '<input type="number" min="0" max="' . esc_attr( DAY_IN_SECONDS ) . '" step="1" ';
	echo 'id="srvip_refresh_interval" name="' . esc_attr( SRVIP_OPTION_KEY ) . '[refresh_interval]" ';
	echo 'value="' . esc_attr( $val ) . '" class="small-text" />';
	echo ' <span class="description">' . esc_html__( 'Set 0 to disable auto-refresh. Value is in seconds.', 'server-ip-dashboard' ) . '</span>';
}

/**
 * Field callback: Visible roles checkboxes.
 * (Uses checked() to safely render the attribute; no raw $checked echo.)
 */
function srvip_field_visible_roles_cb() {
	$settings       = srvip_get_settings();
	$selected       = isset( $settings['visible_roles'] ) && is_array( $settings['visible_roles'] ) ? $settings['visible_roles'] : array();
	$editable_roles = function_exists( 'get_editable_roles' ) ? get_editable_roles() : wp_roles()->roles;

	if ( empty( $editable_roles ) ) {
		echo '<em>' . esc_html__( 'No roles found.', 'server-ip-dashboard' ) . '</em>';
		return;
	}

	echo '<fieldset>';
	echo '<legend class="screen-reader-text">' . esc_html__( 'Additional roles that can view the dashboard widget', 'server-ip-dashboard' ) . '</legend>';
	echo '<p class="description" style="margin-top:0;">' . esc_html__( 'Administrators are always allowed. Select additional roles to grant view access to the dashboard widget.', 'server-ip-dashboard' ) . '</p>';

	foreach ( $editable_roles as $role_slug => $role_info ) {
		$role_slug = sanitize_key( $role_slug );
		if ( 'administrator' === $role_slug ) {
			// Skip; always allowed, no need to toggle.
			continue;
		}
		echo '<label>';
		echo '<input type="checkbox" name="' . esc_attr( SRVIP_OPTION_KEY ) . '[visible_roles][]" value="' . esc_attr( $role_slug ) . '" ' . checked( in_array( $role_slug, $selected, true ), true, false ) . ' />';
		echo ' ' . esc_html( translate_user_role( $role_info['name'] ) );
		echo '</label><br>';
	}

	echo '</fieldset>';
}

/**
 * Field callback: External API URL input.
 */
function srvip_field_external_api_url_cb() {
	$settings = srvip_get_settings();
	$val      = isset( $settings['external_api_url'] ) ? (string) $settings['external_api_url'] : SRVIP_DEFAULT_EXT_API;

	echo '<input type="url" id="srvip_external_api_url" class="regular-text code" ';
	echo 'name="' . esc_attr( SRVIP_OPTION_KEY ) . '[external_api_url]" ';
	echo 'value="' . esc_attr( $val ) . '" />';
	echo '<p class="description">';
	echo esc_html__( 'Default:', 'server-ip-dashboard' ) . ' ';
	echo '<code>' . esc_html( SRVIP_DEFAULT_EXT_API ) . '</code>. ';
	echo esc_html__( 'Must return either JSON like {"ip":"x.x.x.x"} or plain text IP.', 'server-ip-dashboard' );
	echo '</p>';
}
