<?php
/**
 * Admin Dashboard Widget for "Server IP Dashboard".
 *
 * File: /admin/ip-dashboard-widget.php
 *
 * Responsibilities:
 * - Add Dashboard widget.
 * - Enqueue admin assets (JS/CSS) on Dashboard only.
 * - Provide AJAX endpoint to fetch Internal & External IPs.
 *
 * @package Server_IP_Dashboard\Admin
 */

defined( 'ABSPATH' ) || exit; // Prevent direct access.

// -----------------------------
// Constants / Defaults
// -----------------------------

if ( ! defined( 'SRVIP_OPTION_KEY' ) ) {
	define( 'SRVIP_OPTION_KEY', 'srvip_settings' );
}

if ( ! defined( 'SRVIP_EXTERNAL_IP_TRANSIENT' ) ) {
	define( 'SRVIP_EXTERNAL_IP_TRANSIENT', 'srvip_external_ip' );
}

if ( ! defined( 'SRVIP_DEFAULT_EXT_API' ) ) {
	define( 'SRVIP_DEFAULT_EXT_API', 'https://api.ipify.org?format=json' );
}

if ( ! defined( 'SRVIP_NONCE_ACTION' ) ) {
	define( 'SRVIP_NONCE_ACTION', 'srvip_ip_nonce' );
}

if ( ! defined( 'SRVIP_WIDGET_ID' ) ) {
	define( 'SRVIP_WIDGET_ID', 'srvip_dashboard_widget' );
}

// -----------------------------
// Capability & Settings Helpers
// -----------------------------

/**
 * Get plugin settings with sane defaults.
 *
 * @return array
 */
function srvip_get_settings() {
	$defaults = array(
		'refresh_interval' => 0,
		'visible_roles'    => array(),
		'external_api_url' => SRVIP_DEFAULT_EXT_API,
	);

	$opts = get_option( SRVIP_OPTION_KEY, array() );
	if ( ! is_array( $opts ) ) {
		$opts = array();
	}

	$opts = wp_parse_args( $opts, $defaults );

	$opts['refresh_interval'] = max( 0, absint( $opts['refresh_interval'] ) );

	if ( ! is_array( $opts['visible_roles'] ) ) {
		$opts['visible_roles'] = array();
	} else {
		$opts['visible_roles'] = array_values( array_filter( array_map( 'sanitize_key', $opts['visible_roles'] ) ) );
	}

	$api = isset( $opts['external_api_url'] ) ? trim( (string) $opts['external_api_url'] ) : '';
	$opts['external_api_url'] = $api ? $api : SRVIP_DEFAULT_EXT_API;

	return $opts;
}

/**
 * Check if current user is allowed to view the widget.
 *
 * @return bool
 */
function srvip_current_user_can_view() {
	if ( current_user_can( 'manage_options' ) ) {
		return true;
	}

	$settings = srvip_get_settings();
	$extra    = $settings['visible_roles'];

	if ( empty( $extra ) ) {
		return false;
	}

	$user = wp_get_current_user();
	if ( ! $user || empty( $user->roles ) ) {
		return false;
	}

	foreach ( $user->roles as $role_slug ) {
		if ( in_array( $role_slug, $extra, true ) ) {
			return true;
		}
	}

	return false;
}

// -----------------------------
// Dashboard Widget
// -----------------------------

add_action( 'wp_dashboard_setup', 'srvip_add_dashboard_widget' );
function srvip_add_dashboard_widget() {
	if ( ! srvip_current_user_can_view() ) {
		return;
	}

	wp_add_dashboard_widget(
		SRVIP_WIDGET_ID,
		__( 'Server IP Information', 'server-ip-dashboard' ),
		'srvip_render_dashboard_widget'
	);
}

function srvip_render_dashboard_widget() {
	$nonce = wp_create_nonce( SRVIP_NONCE_ACTION );

	echo '<div id="srvip-widget" class="srvip-widget-wrap">';
	echo '<p><strong>' . esc_html__( 'Server Internal IP:', 'server-ip-dashboard' ) . '</strong> ';
	echo '<span id="srvip-internal">' . esc_html__( 'Fetching…', 'server-ip-dashboard' ) . '</span></p>';

	echo '<p><strong>' . esc_html__( 'External/Public IP:', 'server-ip-dashboard' ) . '</strong> ';
	echo '<span id="srvip-external">' . esc_html__( 'Fetching…', 'server-ip-dashboard' ) . '</span></p>';

	echo '<p class="srvip-actions">';
	echo '<button id="srvip-refresh" class="button">' . esc_html__( 'Refresh IP Info', 'server-ip-dashboard' ) . '</button> ';
	echo '<small id="srvip-updated" style="opacity:.75;margin-left:6px;"></small>';
	echo '</p>';

	echo '<input type="hidden" id="srvip-nonce" value="' . esc_attr( $nonce ) . '">';
	echo '</div>';

	echo '<noscript><p>' . esc_html__( 'Enable JavaScript to fetch IP information.', 'server-ip-dashboard' ) . '</p></noscript>';
}

// -----------------------------
// Admin Assets
// -----------------------------

add_action( 'admin_enqueue_scripts', 'srvip_enqueue_admin_assets' );
function srvip_enqueue_admin_assets( $hook ) {
	if ( 'index.php' !== $hook || ! srvip_current_user_can_view() ) {
		return;
	}

	$settings = srvip_get_settings();

	wp_enqueue_script(
		'srvip-admin-js',
		plugins_url( 'admin.js', __FILE__ ),
		array( 'jquery' ),
		defined( 'SRVIP_VERSION' ) ? SRVIP_VERSION : '1.0.0',
		true
	);

	wp_localize_script(
		'srvip-admin-js',
		'SrvipData',
		array(
			'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
			'nonce'           => wp_create_nonce( SRVIP_NONCE_ACTION ),
			'refreshInterval' => (int) $settings['refresh_interval'],
			'updatedLabel'    => esc_html__( 'Last updated:', 'server-ip-dashboard' ),
			'errorLabel'      => esc_html__( 'Error fetching IP', 'server-ip-dashboard' ),
		)
	);

	// Register a dummy style handle WITH VERSION to satisfy PHPCS and cache-bust inline CSS.
	wp_register_style(
		'srvip-admin-inline',
		false,
		array(),
		defined( 'SRVIP_VERSION' ) ? SRVIP_VERSION : '1.0.0'
	);
	wp_enqueue_style( 'srvip-admin-inline' );
	wp_add_inline_style( 'srvip-admin-inline', '.srvip-widget-wrap p{margin:.5em 0}.srvip-actions{margin-top:8px}' );
}

// -----------------------------
// AJAX: Get IPs
// -----------------------------

add_action( 'wp_ajax_srvip_get_ips', 'srvip_handle_get_ips' );
function srvip_handle_get_ips() {
	check_ajax_referer( SRVIP_NONCE_ACTION, 'security' );

	if ( ! srvip_current_user_can_view() ) {
		wp_send_json_error( array( 'message' => __( 'Unauthorized', 'server-ip-dashboard' ) ), 403 );
	}

	$internal_ip = srvip_resolve_internal_ip();
	$external_ip = srvip_resolve_external_ip();

	$data = array(
		'internal_ip' => $internal_ip ? esc_html( $internal_ip ) : __( 'Unavailable', 'server-ip-dashboard' ),
		'external_ip' => $external_ip ? esc_html( $external_ip ) : __( 'Unavailable', 'server-ip-dashboard' ),
		'timestamp'   => esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ) ),
	);

	wp_send_json_success( $data );
}

// -----------------------------
// IP Resolution Helpers
// -----------------------------

/**
 * Resolve internal IP.
 *
 * @return string|null
 */
function srvip_resolve_internal_ip() {
	// Preferred: filter_input (no direct $_SERVER).
	$ip = filter_input( INPUT_SERVER, 'SERVER_ADDR', FILTER_VALIDATE_IP );
	if ( $ip ) {
		return (string) $ip;
	}

	// Fallback: unslash + validate from $_SERVER (line-scoped ignore only on read).
	if ( isset( $_SERVER['SERVER_ADDR'] ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$raw = wp_unslash( $_SERVER['SERVER_ADDR'] ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$ip  = filter_var( $raw, FILTER_VALIDATE_IP );
		if ( $ip ) {
			return (string) $ip;
		}
	}

	// Hostname resolution.
	$host = function_exists( 'gethostname' ) ? gethostname() : php_uname( 'n' );
	if ( $host ) {
		$resolved = gethostbyname( $host );
		if ( $resolved && $resolved !== $host && filter_var( $resolved, FILTER_VALIDATE_IP ) ) {
			return (string) $resolved;
		}
	}

	// UDP socket probe.
	$socket_ip = srvip_udp_local_ip_probe();
	if ( $socket_ip ) {
		return $socket_ip;
	}

	return null;
}

/**
 * Try to discover the local interface IP by opening a UDP socket (no traffic sent).
 *
 * @return string|null
 */
function srvip_udp_local_ip_probe() {
	$socket = @stream_socket_client( 'udp://8.8.8.8:53', $errno, $errstr, 1 );
	if ( ! $socket ) {
		return null;
	}
	$address = stream_socket_get_name( $socket, false ); // e.g. "192.168.1.10:54321"
	if ( $address && strpos( $address, ':' ) !== false ) {
		$parts = explode( ':', $address );
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing a UDP socket (not file I/O); WP_Filesystem not applicable.
		@fclose( $socket );
		return $parts[0];
	}
	// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing a UDP socket (not file I/O); WP_Filesystem not applicable.
	@fclose( $socket );
	return null;
}

/**
 * Resolve external/public IP with short transient cache.
 *
 * @return string|null
 */
function srvip_resolve_external_ip() {
	$cached = get_transient( SRVIP_EXTERNAL_IP_TRANSIENT );
	if ( is_string( $cached ) && $cached ) {
		return $cached;
	}

	$settings = srvip_get_settings();
	$url      = esc_url_raw( $settings['external_api_url'] );
	if ( empty( $url ) ) {
		$url = SRVIP_DEFAULT_EXT_API;
	}

	$response = wp_remote_get(
		$url,
		array(
			'timeout' => 5,
			'headers' => array( 'Accept' => 'application/json, text/plain, */*' ),
		)
	);

	if ( is_wp_error( $response ) ) {
		return null;
	}

	$code = (int) wp_remote_retrieve_response_code( $response );
	if ( $code < 200 || $code >= 300 ) {
		return null;
	}

	$body = wp_remote_retrieve_body( $response );
	if ( ! $body ) {
		return null;
	}

	$ip = null;

	// Try JSON first, then plain text.
	$data = json_decode( $body, true );
	if ( is_array( $data ) && ! empty( $data['ip'] ) && filter_var( $data['ip'], FILTER_VALIDATE_IP ) ) {
		$ip = $data['ip'];
	} else {
		$maybe = trim( wp_strip_all_tags( $body ) );
		if ( $maybe && filter_var( $maybe, FILTER_VALIDATE_IP ) ) {
			$ip = $maybe;
		}
	}

	if ( $ip ) {
		set_transient( SRVIP_EXTERNAL_IP_TRANSIENT, $ip, 5 * MINUTE_IN_SECONDS );
	}

	return $ip ? $ip : null;
}
