<?php
defined( 'ABSPATH' ) || exit;

class VSMGR_Order {

    public function __construct() {
        // Auto-complete: Trigger on valid payment status only
        add_action( 'woocommerce_payment_complete', array( $this, 'auto_complete_order' ) );
        
        // Assign Serial Keys: Trigger on processing/completed
        add_action( 'woocommerce_order_status_processing', array( $this, 'assign_serials_to_order' ) );
        add_action( 'woocommerce_order_status_completed', array( $this, 'assign_serials_to_order' ) );
        
        // Display Keys
        add_action( 'woocommerce_order_details_after_order_table', array( $this, 'display_keys_order_details' ) );
        add_action( 'woocommerce_email_after_order_table', array( $this, 'display_keys_email' ), 10, 4 );
    }

    /**
     * Auto Complete Order
     */
    public function auto_complete_order( $order_id ) {
        if ( ! $order_id ) return;

        // Check if feature is enabled
        if ( get_option( 'vsmgr_auto_complete_order' ) !== 'yes' ) {
            return;
        }

        $order = wc_get_order( $order_id );
        if ( ! $order ) return;

        // Only complete if not already completed/refunded/cancelled
        if ( ! $order->has_status( array( 'completed', 'refunded', 'cancelled', 'failed' ) ) ) {
            $order->update_status( 'completed' );
        }
    }

    /**
     * Assign Serials Logic
     */
    public function assign_serials_to_order( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) return;
        
        // Idempotency Check: Avoid running multiple times per order
        if ( $order->get_meta( '_vsmgr_keys_assigned' ) ) return;

        global $wpdb;
        $table = VSMGR_DB::get_table_name();
        $keys_assigned = false;

        foreach ( $order->get_items() as $item_id => $item ) {
            $product_id = $item->get_product_id();
            $variation_id = $item->get_variation_id();
            $qty = $item->get_quantity();
            $target_id = $variation_id ? $variation_id : $product_id;

            // 1. Check Enabled Flag from Product Meta
            $enabled = get_post_meta( $target_id, '_vsmgr_enabled', true );
            if ( $enabled !== 'yes' ) {
                continue;
            }

            for ( $i = 0; $i < $qty; $i++ ) {
                // 2. Try to get available key
                $key = $wpdb->get_row( $wpdb->prepare( 
                    "SELECT * FROM $table WHERE product_id = %d AND variation_id = %d AND status = 'available' LIMIT 1",
                    $product_id, $variation_id
                ) );

                if ( $key ) {
                    // Assign Existing Key
                    $wpdb->update( 
                        $table, 
                        array( 
                            'status' => 'sold', 
                            'order_id' => $order_id 
                        ), 
                        array( 'id' => $key->id ) 
                    );
                    $keys_assigned = true;
                } else {
                    // 3. Generator Rule Fallback
                    $generator_rule = get_post_meta( $target_id, '_vsmgr_rule', true );
                    
                    if ( ! empty( $generator_rule ) ) {
                        $new_key = VSMGR_Common::generate_key( $generator_rule );
                        $wpdb->insert( $table, array(
                            'product_id'   => $product_id,
                            'variation_id' => $variation_id,
                            'serial_key'   => $new_key,
                            'status'       => 'sold',
                            'order_id'     => $order_id,
                            'activation_limit' => get_post_meta( $target_id, '_vsmgr_activation_limit', true ),
                            'validity'         => get_post_meta( $target_id, '_vsmgr_validity', true )
                        ) );
                        $keys_assigned = true;
                    } else {
                        // 4. Low Stock / Missing Key Handling
                        $this->handle_low_stock( $target_id );
                        $order->add_note( sprintf( esc_html__( 'Serial Key Missing for Product #%d', 'serial-numbers-license-keys-for-woocommerce' ), $target_id ) );
                    }
                }
            }
        }

        // Mark as processed if we attempted assignments (or found enabled products)
        // Ideally only if we actually assigned or tried to assign. 
        // If we found enabled products, we mark as assigned so we don't retry loop.
        if ( $keys_assigned ) {
             $order->update_meta_data( '_vsmgr_keys_assigned', true );
             $order->save();
        }
    }

    /**
     * Handle Low Stock Notification with Rate Limiting
     */
    private function handle_low_stock( $product_id ) {
        if ( get_option( 'vsmgr_stock_notification_email' ) !== 'yes' ) {
            return;
        }

        // Transient check to prevent email spam (limit to 1 per hour per product)
        $transient_key = 'vsmgr_low_stock_sent_' . $product_id;
        if ( get_transient( $transient_key ) ) {
            return;
        }

        $recipient = get_option( 'vsmgr_notification_recipient', get_option( 'admin_email' ) );
        $subject = sprintf( esc_html__( 'Low Stock Alert: Product #%d', 'serial-numbers-license-keys-for-woocommerce' ), $product_id );
        $message = sprintf( esc_html__( 'Stock is empty for Product #%d. Please replenish keys.', 'serial-numbers-license-keys-for-woocommerce' ), $product_id );
        
        wp_mail( $recipient, $subject, $message );
        
        // Set transient for 1 hour
        set_transient( $transient_key, true, HOUR_IN_SECONDS );
    }

    /**
     * Display Keys in Order Details
     */
    public function display_keys_order_details( $order ) {
        $this->render_keys_html( $order );
    }

    /**
     * Display Keys in Email
     */
    public function display_keys_email( $order, $sent_to_admin, $plain_text, $email ) {
        if ( $plain_text ) return;
        $this->render_keys_html( $order );
    }

    private function render_keys_html( $order ) {
        global $wpdb;
        $table = VSMGR_DB::get_table_name();
        $order_id = $order->get_id();

        $keys = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM $table WHERE order_id = %d", $order_id ) );

        if ( empty( $keys ) ) return;

        echo '<h2 style="margin-top: 20px;">' . esc_html__( 'Serial Keys', 'serial-numbers-license-keys-for-woocommerce' ) . '</h2>';
        echo '<table class="td" cellspacing="0" cellpadding="6" style="width: 100%; font-family: \'Helvetica Neue\', Helvetica, Roboto, Arial, sans-serif; margin-bottom: 40px; border: 1px solid #eee;" border="1">';
        echo '<thead><tr>
                <th class="td" scope="col" style="text-align:left; padding: 10px;">' . esc_html__( 'Product', 'serial-numbers-license-keys-for-woocommerce' ) . '</th>
                <th class="td" scope="col" style="text-align:left; padding: 10px;">' . esc_html__( 'Serial Key', 'serial-numbers-license-keys-for-woocommerce' ) . '</th>
              </tr></thead>';
        echo '<tbody>';
        
        foreach ( $keys as $key ) {
            $product = wc_get_product( $key->variation_id ? $key->variation_id : $key->product_id );
            $name = $product ? $product->get_name() : __( 'Unknown Product', 'serial-numbers-license-keys-for-woocommerce' );
            
            echo '<tr>';
            echo '<td class="td" style="text-align:left; padding: 10px;">' . esc_html( $name ) . '</td>';
            echo '<td class="td" style="text-align:left; padding: 10px;"><code>' . esc_html( $key->serial_key ) . '</code></td>';
            echo '</tr>';
        }
        
        echo '</tbody></table>';
    }
}

new VSMGR_Order();
