<?php
defined( 'ABSPATH' ) || exit;

class VSMGR_Import_Export {

    public static function handle_export() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'Unauthorized', 'serial-numbers-license-keys-for-woocommerce' ) );
        }
        check_admin_referer( 'vsmgr_export_nonce' );

        global $wpdb;
        $table = esc_sql( VSMGR_DB::get_table_name() );
        
        $keys = $wpdb->get_results( "SELECT * FROM $table", ARRAY_A );

        if ( empty( $keys ) ) {
            wp_redirect( admin_url( 'admin.php?page=vsmgr-serial-manager&message=no_keys' ) );
            exit;
        }

        $filename = 'serial-keys-export-' . date( 'Y-m-d' ) . '.csv';

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $filename );

        $output = fopen( 'php://output', 'w' );
        
        // Headers
        fputcsv( $output, array( 'ID', 'Product ID', 'Variation ID', 'Serial Key', 'Activation Limit', 'Validity', 'Status', 'Order ID', 'Created At' ) );

        foreach ( $keys as $key ) {
            $row = array(
                $key['id'],
                $key['product_id'],
                $key['variation_id'],
                $key['serial_key'],
                isset( $key['activation_limit'] ) ? $key['activation_limit'] : 0,
                isset( $key['validity'] ) ? $key['validity'] : 0,
                $key['status'],
                $key['order_id'],
                $key['created_at']
            );

            // CSV Injection Protection
            foreach ( $row as $k => $valid ) {
                if ( is_string( $valid ) && preg_match( '/^[\=\+\-\@]/', $valid ) ) {
                    $row[ $k ] = "'" . $valid;
                }
            }
            fputcsv( $output, $row );
        }

        fclose( $output );
        exit;
    }

    public static function handle_import() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'Unauthorized', 'serial-numbers-license-keys-for-woocommerce' ) );
        }
        check_admin_referer( 'vsmgr_import_nonce' );

        if ( ! isset( $_FILES['import_file'] ) || empty( $_FILES['import_file']['tmp_name'] ) ) {
            wp_die( esc_html__( 'Please upload a file.', 'serial-numbers-license-keys-for-woocommerce' ) );
        }

        $file = $_FILES['import_file']['tmp_name'];
        $handle = fopen( $file, 'r' );
        $imported = 0;
        
        if ( $handle !== FALSE ) {
            global $wpdb;
            $table = VSMGR_DB::get_table_name();
            
            // Skip header row
            fgetcsv( $handle );

            while ( ( $data = fgetcsv( $handle, 1000, ',' ) ) !== FALSE ) {
                // Expected Format: Product ID, Variation ID, Key, Limit, Validity, Status
                $product_id = intval( $data[0] );
                $variation_id = intval( $data[1] );
                $key_str = sanitize_text_field( $data[2] );
                
                // New Fields
                $activation_limit = isset($data[3]) ? intval( $data[3] ) : 0;
                $validity         = isset($data[4]) ? intval( $data[4] ) : 0;
                $status           = !empty($data[5]) ? sanitize_text_field( $data[5] ) : 'available';

                if ( $product_id > 0 && !empty( $key_str ) ) {
                    $inserted = $wpdb->insert( $table, array(
                        'product_id'       => $product_id,
                        'variation_id'     => $variation_id,
                        'serial_key'       => $key_str,
                        'activation_limit' => $activation_limit,
                        'validity'         => $validity,
                        'status'           => $status
                    ) );

                    if ( $inserted ) {
                        $imported++;
                    }
                }
            }
            fclose( $handle );
        }

        wp_redirect( admin_url( 'admin.php?page=vsmgr-serial-manager&tab=import&message=imported&count=' . $imported ) );
        exit;
    }
}
