<?php
defined( 'ABSPATH' ) || exit;

class VSMGR_Admin {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'register_menu' ) );
        add_action( 'admin_post_vsmgr_save_key', array( $this, 'handle_save_key' ) );
        add_action( 'admin_post_vsmgr_bulk_generate', array( $this, 'handle_bulk_generate' ) );
        add_action( 'admin_post_vsmgr_export', array( 'VSMGR_Import_Export', 'handle_export' ) );
        add_action( 'admin_post_vsmgr_import', array( 'VSMGR_Import_Export', 'handle_import' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
    }

    public function enqueue_scripts() {
        wp_enqueue_style( 'vsmgr-admin-style', VSMGR_URL . 'assets/css/admin.css', array(), VSMGR_VERSION );
    }

    public function register_menu() {
        // Updated to use Dashicon instead of Base64
        $icon = 'dashicons-admin-network';

        add_menu_page(
            esc_html__( 'Serial Manager', 'serial-numbers-license-keys-for-woocommerce' ),
            esc_html__( 'Serial Manager', 'serial-numbers-license-keys-for-woocommerce' ),
            'manage_woocommerce',
            'vsmgr-serial-manager',
            array( $this, 'render_dashboard' ),
            $icon,
            56
        );

        // Submenus
        add_submenu_page(
            'vsmgr-serial-manager',
            esc_html__( 'All Keys', 'serial-numbers-license-keys-for-woocommerce' ),
            esc_html__( 'All Keys', 'serial-numbers-license-keys-for-woocommerce' ),
            'manage_woocommerce',
            'vsmgr-serial-manager',
            array( $this, 'render_dashboard' )
        );

        add_submenu_page(
            'vsmgr-serial-manager',
            esc_html__( 'Reports', 'serial-numbers-license-keys-for-woocommerce' ),
            esc_html__( 'Reports', 'serial-numbers-license-keys-for-woocommerce' ),
            'manage_woocommerce',
            'vsmgr-reports',
            array( 'VSMGR_Reports', 'render' )
        );

        add_submenu_page(
            'vsmgr-serial-manager',
            esc_html__( 'Settings', 'serial-numbers-license-keys-for-woocommerce' ),
            esc_html__( 'Settings', 'serial-numbers-license-keys-for-woocommerce' ),
            'manage_woocommerce',
            'vsmgr-settings',
            array( 'VSMGR_Settings', 'render' )
        );

        add_submenu_page(
            'vsmgr-serial-manager',
            esc_html__( 'Upgrade Pro', 'serial-numbers-license-keys-for-woocommerce' ),
            esc_html__( 'Upgrade Pro', 'serial-numbers-license-keys-for-woocommerce' ),
            'manage_woocommerce',
            'vsmgr-upgrade',
            array( $this, 'render_upgrade' )
        );
    }

    public function render_upgrade() {
        ?>
        <div class="wrap vsm-wrapper">
            <h1><?php esc_html_e( 'Upgrade to Pro', 'serial-numbers-license-keys-for-woocommerce' ); ?></h1>
            <div class="vsm-card">
                <h3><?php esc_html_e( 'Upgrade to Pro Version', 'serial-numbers-license-keys-for-woocommerce' ); ?></h3>
                <p><?php esc_html_e( 'Unlock even more powerful features.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                <p><a href="#" class="button button-primary"><?php esc_html_e( 'Buy Now', 'serial-numbers-license-keys-for-woocommerce' ); ?></a></p>
            </div>
        </div>
        <?php
    }

    public function render_dashboard() {
        $tab = isset( $_GET['tab'] ) ? sanitize_key( $_GET['tab'] ) : 'list';
        ?>
        <div class="wrap vsm-wrapper">
            <h1 class="wp-heading-inline"><?php esc_html_e( 'Serial Keys Manager', 'serial-numbers-license-keys-for-woocommerce' ); ?></h1>
            <hr class="wp-header-end">
            
            <?php if ( ! empty( $_GET['message'] ) ) : 
                $message = sanitize_text_field( $_GET['message'] );
                ?>
                <?php if ( $message == 'generated' ) : 
                    $count = isset($_GET['count']) ? intval($_GET['count']) : 0;
                    $failed = isset($_GET['failed']) ? intval($_GET['failed']) : 0;
                    ?>
                    <div class="notice notice-success is-dismissible">
                        <p><?php printf( esc_html__( '%d keys generated successfully.', 'serial-numbers-license-keys-for-woocommerce' ), $count ); ?>
                        <?php if($failed > 0) printf( esc_html__( '%d failed.', 'serial-numbers-license-keys-for-woocommerce' ), $failed ); ?>
                        </p>
                    </div>
                <?php elseif ( $message == 'saved' ) : ?>
                    <div class="notice notice-success is-dismissible"><p><?php esc_html_e( 'Key saved successfully.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p></div>
                <?php elseif ( $message == 'imported' ) : ?>
                    <div class="notice notice-success is-dismissible"><p><?php esc_html_e( 'Keys imported successfully.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p></div>
                <?php elseif ( $message == 'no_keys' ) : ?>
                    <div class="notice notice-error is-dismissible"><p><?php esc_html_e( 'No keys found to export.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p></div>
                <?php endif; ?>
            <?php endif; ?>

            <nav class="nav-tab-wrapper">
                <a href="?page=vsmgr-serial-manager&tab=list" class="nav-tab <?php echo $tab == 'list' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'All Keys', 'serial-numbers-license-keys-for-woocommerce' ); ?></a>
                <a href="?page=vsmgr-serial-manager&tab=add_new" class="nav-tab <?php echo $tab == 'add_new' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Add New', 'serial-numbers-license-keys-for-woocommerce' ); ?></a>
                <a href="?page=vsmgr-serial-manager&tab=generator" class="nav-tab <?php echo $tab == 'generator' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Bulk Generator', 'serial-numbers-license-keys-for-woocommerce' ); ?></a>
                <a href="?page=vsmgr-serial-manager&tab=import" class="nav-tab <?php echo $tab == 'import' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Import', 'serial-numbers-license-keys-for-woocommerce' ); ?></a>
                <a href="?page=vsmgr-serial-manager&tab=export" class="nav-tab <?php echo $tab == 'export' ? 'nav-tab-active' : ''; ?>"><?php esc_html_e( 'Export', 'serial-numbers-license-keys-for-woocommerce' ); ?></a>
            </nav>
            
            <div class="vsm-tab-content">
                <?php 
                if ( $tab == 'list' ) $this->render_list();
                elseif ( $tab == 'add_new' ) $this->render_add_new();
                elseif ( $tab == 'generator' ) $this->render_generator();
                elseif ( $tab == 'import' ) $this->render_import();
                elseif ( $tab == 'export' ) $this->render_export();
                ?>
            </div>
        </div>
        <?php
    }

    /**
     * Tab: Import
     */
    private function render_import() {
        ?>
        <div class="vsm-card">
            <h3><?php esc_html_e( 'Import CSV', 'serial-numbers-license-keys-for-woocommerce' ); ?></h3>
            <p><?php esc_html_e( 'Upload a CSV file with columns:', 'serial-numbers-license-keys-for-woocommerce' ); ?> <code>Product ID, Variation ID, Serial Key, Activation Limit, Validity, Status</code> <?php esc_html_e( '(in that order).', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>" enctype="multipart/form-data">
                <input type="hidden" name="action" value="vsmgr_import">
                <?php wp_nonce_field( 'vsmgr_import_nonce' ); ?>
                <input type="file" name="import_file" required accept=".csv">
                <p class="submit"><input type="submit" class="button button-primary" value="<?php esc_attr_e( 'Import Keys', 'serial-numbers-license-keys-for-woocommerce' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    /**
     * Tab: Export
     */
    private function render_export() {
        ?>
        <div class="vsm-card">
            <h3><?php esc_html_e( 'Export Keys', 'serial-numbers-license-keys-for-woocommerce' ); ?></h3>
            <p><?php esc_html_e( 'Download all serial keys as a CSV file.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                <input type="hidden" name="action" value="vsmgr_export">
                <?php wp_nonce_field( 'vsmgr_export_nonce' ); ?>
                <input type="submit" class="button button-primary" value="<?php esc_attr_e( 'Download CSV', 'serial-numbers-license-keys-for-woocommerce' ); ?>">
            </form>
        </div>
        <?php
    }

    /**
     * List Table of Keys
     */
    private function render_list() {
        global $wpdb;
        $table = VSMGR_DB::get_table_name();
        
        // Basic Pagination
        $page = isset( $_GET['paged'] ) ? max( 1, intval( $_GET['paged'] ) ) : 1;
        $per_page = 20;
        $offset = ( $page - 1 ) * $per_page;

        // Uses prepare for correct SQL
        $keys = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$table} ORDER BY id DESC LIMIT %d OFFSET %d", $per_page, $offset ) );
        $total = (int) $wpdb->get_var( "SELECT COUNT(id) FROM {$table}" );
        $total_pages = ceil( $total / $per_page );
        ?>
        <div class="vsm-list-table" style="margin-top: 20px;">
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                    <th><?php esc_html_e( 'ID', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                    <th><?php esc_html_e( 'Product', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                    <th><?php esc_html_e( 'Variation', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                    <th><?php esc_html_e( 'Key', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                    <th><?php esc_html_e( 'Status', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                    <th><?php esc_html_e( 'Order', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                    <th><?php esc_html_e( 'Date', 'serial-numbers-license-keys-for-woocommerce' ); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php if ( empty( $keys ) ) : ?>
                    <tr><td colspan="7"><?php esc_html_e( 'No keys found.', 'serial-numbers-license-keys-for-woocommerce' ); ?></td></tr>
                <?php else: ?>
                    <?php foreach ( $keys as $key ) : ?>
                        <tr>
                            <td><?php echo esc_html( $key->id ); ?></td>
                            <td><?php echo esc_html( $key->product_id ); ?> (<?php echo esc_html( get_the_title( $key->product_id ) ); ?>)</td>
                            <td><?php echo $key->variation_id ? esc_html( $key->variation_id ) : '-'; ?></td>
                            <td>
                                <?php 
                                $key_display = esc_html( $key->serial_key );
                                if ( get_option( 'vsmgr_hide_keys' ) === 'yes' ) {
                                    $key_display = '<code>' . substr($key->serial_key, 0, 4) . ' •••• ' . substr($key->serial_key, -4) . '</code>';
                                } else {
                                    $key_display = '<code>' . $key_display . '</code>';
                                }
                                echo $key_display; 
                                ?>
                            </td>
                            <td><span class="badge badge-<?php echo esc_attr( $key->status ); ?>"><?php echo esc_html( ucfirst($key->status) ); ?></span></td>
                            <td>
                                <?php if($key->order_id > 0) echo '<a href="'.esc_url( admin_url('post.php?post='.$key->order_id.'&action=edit') ).'">#'.esc_html( $key->order_id ).'</a>'; ?>
                            </td>
                            <td><?php echo esc_html( $key->created_at ); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        <?php if ( $total_pages > 1 ) : ?>
            <div class="tablenav bottom">
                <div class="tablenav-pages">
                    <span class="pagination-links">
                        <?php if($page > 1) echo '<a class="button" href="?page=vsmgr-serial-manager&tab=list&paged='.($page-1).'">&laquo; ' . esc_html__('Prev', 'serial-numbers-license-keys-for-woocommerce') . '</a>'; ?>
                        <span class="paging-input"> <?php echo esc_html( $page ); ?> <?php esc_html_e('of', 'serial-numbers-license-keys-for-woocommerce'); ?> <?php echo esc_html( $total_pages ); ?> </span>
                        <?php if($page < $total_pages) echo '<a class="button" href="?page=vsmgr-serial-manager&tab=list&paged='.($page+1).'">' . esc_html__('Next', 'serial-numbers-license-keys-for-woocommerce') . ' &raquo;</a>'; ?>
                    </span>
                </div>
            </div>
        <?php endif; ?>
        <?php
    }

    /**
     * Tab: Add New (Manual)
     */
    private function render_add_new() {
        // Limited Product Fetch to prevent freezing on large stores
        $products = wc_get_products( array( 'limit' => 50, 'status' => 'publish' ) );
        ?>
        <div class="vsm-card">
            <h3><?php esc_html_e( 'Add New License Key', 'serial-numbers-license-keys-for-woocommerce' ); ?></h3>
            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                <input type="hidden" name="action" value="vsmgr_save_key">
                <?php wp_nonce_field( 'vsmgr_save_key_nonce' ); ?>
                
                <table class="form-table">
                    <tr>
                        <th><label><?php esc_html_e( 'Product', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <select name="product_id" class="regular-text" id="vsm_product_select">
                                <option value=""><?php esc_html_e( 'Select a product...', 'serial-numbers-license-keys-for-woocommerce' ); ?></option>
                                <?php foreach ( $products as $product ) : ?>
                                    <option value="<?php echo esc_attr( $product->get_id() ); ?>"><?php echo esc_html( $product->get_name() ); ?> (#<?php echo esc_html( $product->get_id() ); ?>)</option>
                                    <?php if ( $product->is_type( 'variable' ) ) : 
                                        $variations = $product->get_available_variations();
                                        foreach ( $variations as $variation ) : ?>
                                            <option value="<?php echo esc_attr( $product->get_id() ); ?>:<?php echo esc_attr( $variation['variation_id'] ); ?>">
                                                 — <?php echo esc_html( get_the_title( $variation['variation_id'] ) ); ?> (#<?php echo esc_html( $variation['variation_id'] ); ?>)
                                            </option>
                                        <?php endforeach; 
                                    endif; ?>
                                <?php endforeach; ?>
                            </select>
                            <p class="description"><?php esc_html_e( 'Select the product for which this key is applicable.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Serial Key', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <textarea name="serial_keys" rows="5" class="large-text" placeholder="serial-####-####-####"></textarea>
                            <p class="description"><?php esc_html_e( 'Enter your serial key, also supports multiline.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Activation Limit', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <input type="number" name="activation_limit" value="0" class="small-text">
                            <p class="description"><?php esc_html_e( 'Maximum number of times the key can be used. Leave 0 for unlimited/software default.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Valid For (Days)', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <input type="number" name="validity" value="0" class="small-text">
                            <p class="description"><?php esc_html_e( 'Number of days the key will be valid. Leave 0 for lifetime.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Status', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <select name="status">
                                <option value="available"><?php esc_html_e( 'Available', 'serial-numbers-license-keys-for-woocommerce' ); ?></option>
                                <option value="sold"><?php esc_html_e( 'Sold', 'serial-numbers-license-keys-for-woocommerce' ); ?></option>
                                <option value="expired"><?php esc_html_e( 'Expired', 'serial-numbers-license-keys-for-woocommerce' ); ?></option>
                                <option value="inactive"><?php esc_html_e( 'Inactive', 'serial-numbers-license-keys-for-woocommerce' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'Serial key status.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Order ID', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <input type="number" name="order_id" value="" placeholder="<?php esc_attr_e( 'Optional', 'serial-numbers-license-keys-for-woocommerce' ); ?>" class="regular-text">
                            <p class="description"><?php esc_html_e( 'The order to which the serial number will be assigned.', 'serial-numbers-license-keys-for-woocommerce' ); ?></p>
                        </td>
                    </tr>
                </table>
                <p class="submit"><input type="submit" class="button button-primary" value="<?php esc_attr_e( 'Save Keys', 'serial-numbers-license-keys-for-woocommerce' ); ?>"></p>
            </form>
        </div>
        <script>
            // Simple helper to split variation ID if needed logic added later
            // Currently value is "PID" or "PID:VID"
        </script>
        <?php
    }

    /**
     * Tab: Generator
     */
    private function render_generator() {
        // Fetch Products for Dropdown
        $products = wc_get_products( array( 'limit' => 50, 'status' => 'publish' ) );
        ?>
        <div class="vsm-card">
            <h3><?php esc_html_e( 'Bulk Generator', 'serial-numbers-license-keys-for-woocommerce' ); ?></h3>
            <form method="post" action="<?php echo esc_url( admin_url('admin-post.php') ); ?>">
                <input type="hidden" name="action" value="vsmgr_bulk_generate">
                <?php wp_nonce_field( 'vsmgr_bulk_generate_nonce' ); ?>
                
                <table class="form-table">
                    <tr>
                        <th><label><?php esc_html_e( 'Product', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <select name="product_id" class="regular-text" id="vsm_gen_product_select">
                                <option value=""><?php esc_html_e( 'Select a product...', 'serial-numbers-license-keys-for-woocommerce' ); ?></option>
                                <?php foreach ( $products as $product ) : ?>
                                    <option value="<?php echo esc_attr( $product->get_id() ); ?>"><?php echo esc_html( $product->get_name() ); ?> (#<?php echo esc_html( $product->get_id() ); ?>)</option>
                                    <?php if ( $product->is_type( 'variable' ) ) : 
                                        $variations = $product->get_available_variations();
                                        foreach ( $variations as $variation ) : ?>
                                            <option value="<?php echo esc_attr( $product->get_id() ); ?>:<?php echo esc_attr( $variation['variation_id'] ); ?>">
                                                 — <?php echo esc_html( get_the_title( $variation['variation_id'] ) ); ?> (#<?php echo esc_html( $variation['variation_id'] ); ?>)
                                            </option>
                                        <?php endforeach; 
                                    endif; ?>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    
                    <tr>
                        <th><label><?php esc_html_e( 'Quantity', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td><input type="number" name="quantity" value="10" min="1" max="1000" class="small-text"></td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Activation Limit', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <input type="number" name="activation_limit" value="0" class="small-text">
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Valid For (Days)', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <input type="number" name="validity" value="0" class="small-text">
                        </td>
                    </tr>

                    <tr>
                        <th><label><?php esc_html_e( 'Rule Override', 'serial-numbers-license-keys-for-woocommerce' ); ?></label></th>
                        <td>
                            <input type="text" name="rule" class="regular-text" placeholder="<?php esc_attr_e( 'Leave empty to use Product Settings', 'serial-numbers-license-keys-for-woocommerce' ); ?>">
                            <p class="description"><?php esc_html_e( 'Example:', 'serial-numbers-license-keys-for-woocommerce' ); ?> <code>SALE-{4}-{4}</code></p>
                        </td>
                    </tr>
                </table>
                <p class="submit"><input type="submit" class="button button-primary" value="<?php esc_attr_e( 'Generate Keys', 'serial-numbers-license-keys-for-woocommerce' ); ?>"></p>
            </form>
        </div>
        <?php
    }

    /**
     * Handle: Bulk Generate
     */
    public function handle_bulk_generate() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'Unauthorized', 'serial-numbers-license-keys-for-woocommerce' ) );
        }
        check_admin_referer( 'vsmgr_bulk_generate_nonce' );
        
        // Parse Product/Variation from Select
        $product_val = isset($_POST['product_id']) ? sanitize_text_field( $_POST['product_id'] ) : ''; 
        $parts = explode( ':', $product_val );
        $product_id = intval( $parts[0] );
        $variation_id = isset( $parts[1] ) ? intval( $parts[1] ) : 0;

        // Validation - Products
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
             wp_die( esc_html__( 'Invalid product', 'serial-numbers-license-keys-for-woocommerce' ) );
        }
        if ( $variation_id ) {
            $variation = wc_get_product( $variation_id );
            if ( ! $variation || $variation->get_parent_id() !== $product_id ) {
                 wp_die( esc_html__( 'Invalid variation', 'serial-numbers-license-keys-for-woocommerce' ) );
            }
        }

        $qty = isset($_POST['quantity']) ? intval( $_POST['quantity'] ) : 10;
        $activation_limit = isset($_POST['activation_limit']) ? intval( $_POST['activation_limit'] ) : 0;
        $validity = isset($_POST['validity']) ? intval( $_POST['validity'] ) : 0;
        $rule_override = isset($_POST['rule']) ? sanitize_text_field( $_POST['rule'] ) : '';

        // If override is empty, try fetch from product meta
        if ( empty( $rule_override ) ) {
            $target_id = $variation_id ? $variation_id : $product_id;
            $rule_override = get_post_meta( $target_id, '_vsmgr_rule', true );
        }
        
        // Validation - Rules
        if ( empty( $rule_override ) ) {
            wp_die( esc_html__( 'No generator rule defined. Please check product settings or provide an override.', 'serial-numbers-license-keys-for-woocommerce' ) );
        }

        global $wpdb;
        $table = VSMGR_DB::get_table_name();
        $generated = 0;
        $failed = 0;

        for ( $i = 0; $i < $qty; $i++ ) {
            $key_str = VSMGR_Common::generate_key( $rule_override );
            $result = $wpdb->insert( $table, array(
                'product_id'   => $product_id,
                'variation_id' => $variation_id,
                'serial_key'   => $key_str,
                'status'       => 'available',
                'activation_limit' => $activation_limit,
                'validity'     => $validity
            ) );

            if ( $result === false ) {
                $failed++;
            } else {
                $generated++;
            }
        }

        wp_redirect( admin_url( 'admin.php?page=vsmgr-serial-manager&message=generated&count=' . $generated . '&failed=' . $failed ) );
        exit;
    }
    
    public function handle_save_key() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
             wp_die( esc_html__( 'Unauthorized', 'serial-numbers-license-keys-for-woocommerce' ) );
        }
        check_admin_referer( 'vsmgr_save_key_nonce' );
        
        $product_val = isset($_POST['product_id']) ? sanitize_text_field( $_POST['product_id'] ) : '';
        $parts = explode( ':', $product_val );
        $product_id = intval( $parts[0] );
        $variation_id = isset( $parts[1] ) ? intval( $parts[1] ) : 0;

        // Validation - Products
        $product = wc_get_product( $product_id );
        if ( ! $product ) {
             wp_die( esc_html__( 'Invalid product', 'serial-numbers-license-keys-for-woocommerce' ) );
        }
        if ( $variation_id ) {
             $variation = wc_get_product( $variation_id );
             if ( ! $variation || $variation->get_parent_id() !== $product_id ) {
                  wp_die( esc_html__( 'Invalid variation', 'serial-numbers-license-keys-for-woocommerce' ) );
             }
        }

        $keys_textarea = isset($_POST['serial_keys']) ? sanitize_textarea_field( $_POST['serial_keys'] ) : '';
        $keys = explode( "\n", $keys_textarea );
        
        $activation_limit = isset($_POST['activation_limit']) ? intval( $_POST['activation_limit'] ) : 0;
        $validity = isset($_POST['validity']) ? intval( $_POST['validity'] ) : 0;
        $status = isset($_POST['status']) ? sanitize_text_field( $_POST['status'] ) : 'available';
        $order_id = isset($_POST['order_id']) ? intval( $_POST['order_id'] ) : 0;

        global $wpdb;
        $table = VSMGR_DB::get_table_name();

        foreach ( $keys as $key_str ) {
            $key_str = trim( $key_str );
            if ( empty( $key_str ) ) continue;

            $wpdb->insert( $table, array(
                'product_id'   => $product_id,
                'variation_id' => $variation_id,
                'serial_key'   => $key_str,
                'activation_limit' => $activation_limit,
                'validity'     => $validity,
                'status'       => $status,
                'order_id'     => $order_id
            ) );
        }

        wp_redirect( admin_url( 'admin.php?page=vsmgr-serial-manager&message=saved' ) );
        exit;
    }
}
new VSMGR_Admin();
