<?php

if (!defined('ABSPATH')) {
    exit;
}

class SentWatch {

    protected $api;
    protected $heartbeat;
    protected $admin;

    public function __construct() {
        $this->api = new SentWatch_API();
        $this->heartbeat = new SentWatch_Heartbeat($this->api);
        
        if (is_admin()) {
            $this->admin = new SentWatch_Admin($this->api, $this->heartbeat);
        }
    }

    public function run() {
        // Add Custom Cron Schedules
        add_filter('cron_schedules', [$this, 'add_cron_schedules']);

        // Init Heartbeat Scheduler
        $this->heartbeat->init();

        // Init Admin
        if (is_admin()) {
            $this->admin->init();
        }

        // Register REST API Endpoint for Diagnostics
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Config Sync Task
        add_action('sentwatch_check_config', [$this, 'check_remote_config']);
    }

    /**
     * Register REST API Routes
     */
    public function register_rest_routes() {
        register_rest_route('sentwatch/v1', '/ping', [
            'methods' => 'GET',
            'callback' => [$this, 'handle_ping_request'],
            'permission_callback' => [$this, 'validate_api_request'],
        ]);

        register_rest_route('sentwatch/v1', '/sync', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_sync_request'],
            'permission_callback' => [$this, 'validate_api_request'],
        ]);
    }

    /**
     * Validate API Request - Permission Callback for REST endpoints
     * Checks API Key and Site UUID from request headers
     */
    public function validate_api_request($request) {
        $headers = $request->get_headers();
        $apiKey = $headers['x_api_key'][0] ?? '';
        $requestUuid = $headers['x_site_uuid'][0] ?? '';

        $storedKey = get_option('sentwatch_api_key');
        $storedUuid = get_option('sentwatch_site_uuid');

        // Validate API Key
        if (!$apiKey || $apiKey !== $storedKey) {
            return false;
        }

        // Validate UUID
        if ($requestUuid && $requestUuid !== $storedUuid) {
            return false;
        }

        return true;
    }

    /**
     * Handle Sync Request from Dashboard
     * Authentication already validated by permission_callback
     */
    public function handle_sync_request($request) {
        // Trigger Sync Logic
        // We call check_remote_config which pulls the latest config from the API
        // This is safer than accepting config payload directly
        $this->check_remote_config();

        return new WP_REST_Response([
            'success' => true, 
            'message' => 'Sync triggered',
            'new_interval' => get_option('sentwatch_interval')
        ], 200);
    }

    /**
     * Handle Ping Request from Dashboard
     * Authentication already validated by permission_callback
     */
    public function handle_ping_request($request) {
        // Gather Diagnostic Data
        $data = [
            'status' => 'active',
            'version' => SENTWATCH_VERSION,
            'wp_version' => get_bloginfo('version'),
            'php_version' => phpversion(),
            'server_time' => current_time('mysql'),
            'next_heartbeat' => wp_next_scheduled('sentwatch_send_heartbeat') ? get_date_from_gmt(gmdate('Y-m-d H:i:s', wp_next_scheduled('sentwatch_send_heartbeat')), 'Y-m-d H:i:s') : 'not_scheduled',
            'last_heartbeat' => get_option('sentwatch_last_heartbeat', 'never'),
            'memory_limit' => ini_get('memory_limit'),
        ];

        // 3. Trigger Heartbeat if requested
        if ($request->get_param('trigger_mail')) {
            $this->heartbeat->send_email_heartbeat();
            $data['mail_triggered'] = true;
        }

        return new WP_REST_Response($data, 200);
    }

    /**
     * Add custom cron intervals based on SentWatch settings
     */
    public function add_cron_schedules($schedules) {
        $interval = (int)get_option('sentwatch_interval', 24); // Default 24h
        
        // Aggiungi sempre l'intervallo dinamico "sentwatch_custom"
        // Questo ci permette di avere un nome fisso ma una durata variabile
        $schedules['sentwatch_custom'] = [
            'interval' => $interval * 3600, // Converti ore in secondi
            'display'  => "SentWatch Every {$interval} Hours"
        ];

        return $schedules;
    }

    public static function activate() {
        // Schedule default event if configured
        if (get_option('sentwatch_api_key') && get_option('sentwatch_site_uuid')) {
            if (!wp_next_scheduled('sentwatch_send_heartbeat')) {
                wp_schedule_event(time(), 'sentwatch_custom', 'sentwatch_send_heartbeat');
            }
        }

        // Schedule config sync (Daily)
        if (!wp_next_scheduled('sentwatch_check_config')) {
            wp_schedule_event(time(), 'daily', 'sentwatch_check_config');
        }
    }

    public static function deactivate() {
        // Notify API first (before clearing schedule)
        $apiKey = get_option('sentwatch_api_key');
        $siteUuid = get_option('sentwatch_site_uuid');

        if ($apiKey && $siteUuid) {
            $api = new SentWatch_API();
            $result = $api->deactivate_site($apiKey, $siteUuid);

            if (isset($result['success']) && !$result['success']) {
                // Log error but continue with deactivation
                error_log('SentWatch plugin deactivation: API call failed - ' . ($result['error'] ?? 'Unknown error'));
            }
        }

        // Clear schedule
        wp_clear_scheduled_hook('sentwatch_send_heartbeat');
        wp_clear_scheduled_hook('sentwatch_check_config');
    }

    /**
     * Controlla periodicamente la configurazione remota (Sync)
     * Assicura che l'intervallo di heartbeat sia allineato con il pannello
     */
    public function check_remote_config() {
        $apiKey = get_option('sentwatch_api_key');
        $siteUuid = get_option('sentwatch_site_uuid');

        if (!$apiKey || !$siteUuid) {
            return;
        }

        // Usa l'API esistente
        $api = new SentWatch_API();
        $status = $api->get_status($apiKey, $siteUuid);

        if (isset($status['success']) && $status['success'] && isset($status['heartbeat_interval_hours'])) {
            $current_interval = (int)get_option('sentwatch_interval', 24);
            $remote_interval = (int)$status['heartbeat_interval_hours'];

            // Se l'intervallo è cambiato, aggiorna e rischedula
            if ($remote_interval > 0 && $current_interval !== $remote_interval) {
                update_option('sentwatch_interval', $remote_interval);
                
                // Rischedula heartbeat
                wp_clear_scheduled_hook('sentwatch_send_heartbeat');
                wp_schedule_event(time(), 'sentwatch_custom', 'sentwatch_send_heartbeat');
            }
        }
    }
}
