<?php

if (!defined('ABSPATH')) {
    exit;
}

class SentWatch_API {

    private $api_url = SENTWATCH_API_URL;

    /**
     * Registra il sito su SentWatch
     */
    public function register_site($api_key) {
        $body = [
            'site_url' => get_site_url(),
            'site_name' => get_bloginfo('name'),
            'wp_version' => get_bloginfo('version'),
            'php_version' => phpversion(),
            'plugin_version' => SENTWATCH_VERSION
        ];

        $response = wp_remote_post($this->api_url . 'register-site.php', [
            'headers' => [
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/x-www-form-urlencoded'
            ],
            'body' => $body,
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200) {
            return ['success' => false, 'error' => $body['error'] ?? 'Unknown error'];
        }

        return $body;
    }

    /**
     * Ottiene lo stato del sito
     */
    public function get_status($api_key, $site_uuid) {
        $url = add_query_arg([
            'site_uuid' => $site_uuid,
            'history' => 1 // Request uptime history
        ], $this->api_url . 'status.php');

        $response = wp_remote_get($url, [
            'headers' => [
                'X-API-Key' => $api_key
            ],
            'timeout' => 10
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        return json_decode(wp_remote_retrieve_body($response), true);
    }

    /**
     * Invia heartbeat via HTTP
     */
    public function send_heartbeat($api_key, $site_uuid) {
        $response = wp_remote_post($this->api_url . 'heartbeat.php', [
            'headers' => [
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/x-www-form-urlencoded'
            ],
            'body' => [
                'site_uuid' => $site_uuid
            ],
            'timeout' => 15
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200) {
            return ['success' => false, 'error' => $body['error'] ?? 'Unknown error'];
        }

        return $body;
    }

    /**
     * Notifica disattivazione sito
     */
    public function deactivate_site($api_key, $site_uuid) {
        $response = wp_remote_post($this->api_url . 'deactivate-site.php', [
            'headers' => [
                'X-API-Key' => $api_key,
                'Content-Type' => 'application/x-www-form-urlencoded'
            ],
            'body' => [
                'site_uuid' => $site_uuid
            ],
            'timeout' => 10,
            'blocking' => true // Aspetta risposta per verificare successo
        ]);

        if (is_wp_error($response)) {
            error_log('SentWatch deactivation failed: ' . $response->get_error_message());
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200) {
            $error = $body['error'] ?? 'Unknown error';
            error_log('SentWatch deactivation failed (HTTP ' . $code . '): ' . $error);
            return ['success' => false, 'error' => $error];
        }

        return $body;
    }
}
