<?php

if (!defined('ABSPATH')) {
    exit;
}

class SentWatch_Admin {

    private $api;
    private $heartbeat;

    public function __construct($api, $heartbeat) {
        $this->api = $api;
        $this->heartbeat = $heartbeat;
    }

    public function init() {
        add_action('admin_menu', [$this, 'add_menu_page']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
        add_action('admin_post_sentwatch_connect', [$this, 'handle_connect']);
        add_action('admin_post_sentwatch_disconnect', [$this, 'handle_disconnect']);
        add_action('admin_post_sentwatch_test', [$this, 'handle_test']);
    }

    public function add_menu_page() {
        add_menu_page(
            'SentWatch',
            'SentWatch',
            'manage_options',
            'sentwatch',
            [$this, 'render_settings_page'],
            'dashicons-email-alt',
            100
        );
    }

    public function enqueue_assets($hook) {
        if ($hook !== 'toplevel_page_sentwatch') {
            return;
        }
        wp_enqueue_style('sentwatch-admin', SENTWATCH_PLUGIN_URL . 'admin/css/admin.css', [], SENTWATCH_VERSION);
    }

    public function render_settings_page() {
        // Passa dati alla view
        $connected = get_option('sentwatch_connected', false);
        $api_key = get_option('sentwatch_api_key', '');
        $site_uuid = get_option('sentwatch_site_uuid', '');
        $last_heartbeat = get_option('sentwatch_last_heartbeat', '');
        $status = null;
        
        // Se connesso, recupera stato aggiornato
        if ($connected && $site_uuid) {
            $status = $this->api->get_status($api_key, $site_uuid);

            // Sync interval if changed
            if (isset($status['heartbeat_interval_hours'])) {
                $current_interval = (int)get_option('sentwatch_interval', 24);
                $remote_interval = (int)$status['heartbeat_interval_hours'];
                
                if ($current_interval !== $remote_interval && $remote_interval > 0) {
                    update_option('sentwatch_interval', $remote_interval);
                    
                    // Reschedule with new interval
                    wp_clear_scheduled_hook('sentwatch_send_heartbeat');
                    wp_schedule_event(time(), 'sentwatch_custom', 'sentwatch_send_heartbeat');
                }
            }
        }

        include SENTWATCH_PLUGIN_DIR . 'admin/settings-page.php';
    }

    public function handle_connect() {
        if (!current_user_can('manage_options') || !check_admin_referer('sentwatch_connect_action', 'sentwatch_nonce')) {
            wp_die('Unauthorized');
        }

        $api_key = sanitize_text_field($_POST['api_key']);
        
        if (empty($api_key)) {
            wp_redirect(admin_url('admin.php?page=sentwatch&error=missing_key'));
            exit;
        }

        // Call API
        $result = $this->api->register_site($api_key);

        if ($result['success']) {
            // Save options
            update_option('sentwatch_connected', true);
            update_option('sentwatch_api_key', $api_key);
            update_option('sentwatch_site_uuid', $result['site_uuid']);
            update_option('sentwatch_heartbeat_email', $result['heartbeat_email']);
            update_option('sentwatch_interval', $result['heartbeat_interval_hours']);

            // Schedule Cron
            if (!wp_next_scheduled('sentwatch_send_heartbeat')) {
                // Use custom interval defined in class-sentwatch.php based on sentwatch_interval option
                wp_schedule_event(time(), 'sentwatch_custom', 'sentwatch_send_heartbeat');
            }

            // Send first heartbeat immediately
            $this->heartbeat->send_heartbeat();

            wp_redirect(admin_url('admin.php?page=sentwatch&success=connected'));
        } else {
            wp_redirect(admin_url('admin.php?page=sentwatch&error=' . urlencode($result['error'])));
        }
        exit;
    }

    public function handle_disconnect() {
        if (!current_user_can('manage_options') || !check_admin_referer('sentwatch_disconnect_action', 'sentwatch_nonce')) {
            wp_die('Unauthorized');
        }

        $api_key = get_option('sentwatch_api_key');
        $site_uuid = get_option('sentwatch_site_uuid');

        // Notify API
        if ($api_key && $site_uuid) {
            $this->api->deactivate_site($api_key, $site_uuid);
        }

        // Cleanup
        delete_option('sentwatch_connected');
        delete_option('sentwatch_api_key');
        delete_option('sentwatch_site_uuid');
        delete_option('sentwatch_heartbeat_email');
        delete_option('sentwatch_interval');
        delete_option('sentwatch_last_heartbeat');

        wp_clear_scheduled_hook('sentwatch_send_heartbeat');

        wp_redirect(admin_url('admin.php?page=sentwatch&success=disconnected'));
        exit;
    }

    public function handle_test() {
        if (!current_user_can('manage_options') || !check_admin_referer('sentwatch_test_action', 'sentwatch_nonce')) {
            wp_die('Unauthorized');
        }

        $this->heartbeat->send_heartbeat();
        
        wp_redirect(admin_url('admin.php?page=sentwatch&success=heartbeat_sent'));
        exit;
    }
}
