<?php
function validate_request() {
    // Only intercept GET requests
    if (!isset($_SERVER['REQUEST_METHOD']) || $_SERVER['REQUEST_METHOD'] !== 'GET') {
        return;
    }

    // Exclude admin and AJAX requests
    if (is_admin() || wp_doing_ajax()) {
        return;
    }

    if (!senthor_can_operate()) {
        return;
    }

    // Get data
    $headers = getallheaders();
    $uri = isset($_SERVER['REQUEST_URI']) ? esc_url_raw(wp_unslash($_SERVER['REQUEST_URI'])) : '';
    $ip = get_client_ip();

    $sensitive_headers = [
        'authorization',
        'cookie',
        'set-cookie',
        'x-csrf-token',
    ];
    $filtered_headers = array_filter(
        $headers,
        function($key) use ($sensitive_headers) {
            return !in_array(strtolower($key), $sensitive_headers, true);
        },
        ARRAY_FILTER_USE_KEY
    );

    // Make API call
    $api_response = call_validation_api($filtered_headers, $uri, $ip);

    // Don't break the site if the API has an issue. We only handle known code
    if ($api_response["status"] === 402) {
        if (!empty($api_response['headers']) && is_array($api_response['headers'])) {
            foreach ($api_response['headers'] as $name => $value) {
                header("$name: $value");
            }
        }

        status_header(402);

        echo $api_response["body"];
        exit;
    }
}

function get_client_ip() {
    $client_ip       = wp_unslash($_SERVER['HTTP_CLIENT_IP'] ?? '');
    $forwarded_for   = wp_unslash($_SERVER['HTTP_X_FORWARDED_FOR'] ?? '');
    $remote_addr     = wp_unslash($_SERVER['REMOTE_ADDR'] ?? '');

    if (!empty($client_ip)) {
        return sanitize_text_field($client_ip);
    }

    if (!empty($forwarded_for)) {
        return sanitize_text_field($forwarded_for);
    }

    if (!empty($remote_addr)) {
        return sanitize_text_field($remote_addr);
    }

    return "";
}

function call_validation_api($headers, $uri, $ip) {
    $api_url = 'https://waf-api.senthor.io/api/check-request';

    // Build request payload
    $body_array = array(
        'headers'     => $headers,
        'request_url' => $uri,
        'client_ip'   => $ip,
    );
    $body = json_encode($body_array);

    // Make request
    $response = wp_remote_post($api_url, array(
        'method'  => 'POST',
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body'    => $body,
        'timeout' => 2,
    ));

    if (is_wp_error($response)) {
        return array(
            'status' => 500,
            'body'   => '',
            'error'  => $response->get_error_message(),
        );
    }

    $status_code = wp_remote_retrieve_response_code($response);
    $response_body = wp_remote_retrieve_body($response);
    $response_headers = wp_remote_retrieve_headers($response);

    $crawler_headers = array();
    foreach ($response_headers as $name => $value) {
        if (stripos($name, 'crawler-') === 0) {
            $crawler_headers[$name] = $value;
        }
    }

    return array(
        'status'  => $status_code,
        'body'    => $response_body,
        'headers' => $crawler_headers,
    );
}
?>