<?php
/**
 * Plugin Name: Sengeku Meta Description Manager
 * Plugin URI: https://wordpress.org/plugins/sengeku-meta-description-manager/
 * Description: A lightweight meta description manager with character counter for optimal SEO results.
 * Version: 1.2.0
 * Author: Sen-ge-ku
 * Author URI: https://sen-ge-ku.de
 * License: GPL v2 or later
 * Text Domain: sengeku-meta-description-manager
 * Domain Path: /languages
 */

// Security check
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add meta box for Meta Description
 */
function sengeku_add_meta_box() {
    // For all public post types (Posts, Pages, etc.)
    $post_types = get_post_types(array('public' => true));
    
    foreach ($post_types as $post_type) {
        add_meta_box(
            'sengeku_meta_description_box',
            __('Meta Description', 'sengeku-meta-description-manager'),
            'sengeku_meta_box_html',
            $post_type,
            'side', // Position: 'normal', 'side', or 'advanced'
            'high' // Priority
        );
    }
}
add_action('add_meta_boxes', 'sengeku_add_meta_box');

/**
 * Meta box HTML content
 */
function sengeku_meta_box_html($post) {
    // Nonce for security
    wp_nonce_field('sengeku_meta_description_nonce', 'sengeku_meta_nonce');

    // Get current value
    $value = get_post_meta($post->ID, 'meta_description', true);

    // Character counter
    $char_count = strlen($value);
    $max_chars = 160;
    $char_class = ($char_count > $max_chars) ? 'style="color: #d63638;"' : 'style="color: #007cba;"';
    
    ?>
    <div style="margin-bottom: 10px;">
        <label for="sengeku_meta_description_field">
            <strong><?php esc_html_e('Meta Description for search engines:', 'sengeku-meta-description-manager'); ?></strong>
        </label>
        <p style="color: #666; font-size: 12px; margin: 5px 0;">
            <?php esc_html_e('Ideal length: 150-160 characters', 'sengeku-meta-description-manager'); ?>
        </p>
        <textarea 
            id="sengeku_meta_description_field" 
            name="sengeku_meta_description_field" 
            rows="4" 
            style="width: 100%; margin: 8px 0; padding: 8px; font-size: 13px;" 
            onkeyup="document.getElementById('sengeku_char_count').innerHTML = this.value.length;"
            placeholder="<?php esc_attr_e('Describe the content of this page for search engines...', 'sengeku-meta-description-manager'); ?>"
        ><?php echo esc_textarea($value); ?></textarea>
        
        <div style="text-align: right; font-size: 12px; color: #666;">
            <span id="sengeku_char_count" <?php echo wp_kses_post($char_class); ?>><?php echo absint($char_count); ?></span>
            / <?php echo absint($max_chars); ?> <?php esc_html_e('characters', 'sengeku-meta-description-manager'); ?>
        </div>
    </div>
    
    <?php if ($value) : ?>
    <div style="background: #f0f6fc; border-left: 4px solid #007cba; padding: 8px; margin-top: 10px; font-size: 12px;">
        <strong><?php esc_html_e('Preview:', 'sengeku-meta-description-manager'); ?></strong><br>
        <em><?php echo esc_html($value); ?></em>
    </div>
    <?php endif;
}

/**
 * Save meta box value
 */
function sengeku_save_meta_box($post_id) {
    // Verify nonce
    if (!isset($_POST['sengeku_meta_nonce']) ||
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['sengeku_meta_nonce'])), 'sengeku_meta_description_nonce')) {
        return;
    }
    
    // Check autosave
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    // Check permissions
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }
    
    // Save or delete field value
    if (isset($_POST['sengeku_meta_description_field'])) {
        $meta_description = sanitize_textarea_field(wp_unslash($_POST['sengeku_meta_description_field']));
        update_post_meta($post_id, 'meta_description', $meta_description);
    } else {
        delete_post_meta($post_id, 'meta_description');
    }
}
add_action('save_post', 'sengeku_save_meta_box');

/**
 * Add Meta Description to <head>
 */
function sengeku_add_meta_description() {
    if (is_singular()) {
        global $post;
        $meta_description = get_post_meta($post->ID, 'meta_description', true);

        if (!empty($meta_description)) {
            echo '<meta name="description" content="' . esc_attr($meta_description) . '">' . "\n";
        }
    }
}
add_action('wp_head', 'sengeku_add_meta_description', 1);

/**
 * Admin-Styles und Scripts
 */
function sengeku_admin_enqueue_assets() {
    $screen = get_current_screen();

    if ($screen && ($screen->base === 'post' || $screen->base === 'post-new')) {
        // CSS
        $custom_css = '
            #sengeku_meta_description_box .inside {
                padding: 0 12px 12px 12px;
            }
            #sengeku_meta_description_box .hndle {
                background: #f0f6fc;
                border-bottom: 1px solid #ccd0d4;
            }
        ';

        wp_register_style('sengeku-meta-admin', false, array(), '1.2.0');
        wp_enqueue_style('sengeku-meta-admin');
        wp_add_inline_style('sengeku-meta-admin', $custom_css);

        // JavaScript
        $custom_js = "
            document.addEventListener('DOMContentLoaded', function() {
                var textarea = document.getElementById('sengeku_meta_description_field');
                var charCount = document.getElementById('sengeku_char_count');

                if (textarea && charCount) {
                    textarea.addEventListener('input', function() {
                        var count = this.value.length;
                        charCount.innerHTML = count;

                        if (count > 160) {
                            charCount.style.color = '#d63638';
                        } else {
                            charCount.style.color = '#007cba';
                        }
                    });
                }
            });
        ";

        wp_register_script('sengeku-meta-admin', false, array(), '1.2.0', true);
        wp_enqueue_script('sengeku-meta-admin');
        wp_add_inline_script('sengeku-meta-admin', $custom_js);
    }
}
add_action('admin_enqueue_scripts', 'sengeku_admin_enqueue_assets');

/**
 * Add column to posts overview
 */
function sengeku_add_meta_column($columns) {
    $columns['meta_description'] = __('Meta Description', 'sengeku-meta-description-manager');
    return $columns;
}
add_filter('manage_posts_columns', 'sengeku_add_meta_column');
add_filter('manage_pages_columns', 'sengeku_add_meta_column');

function sengeku_display_meta_column($column, $post_id) {
    if ($column == 'meta_description') {
        $meta = get_post_meta($post_id, 'meta_description', true);
        if (!empty($meta)) {
            echo '<span style="color: green;">✓</span> ' . esc_html(substr($meta, 0, 60)) . '...';
        } else {
            echo '<span style="color: #ccc; font-style: italic;">—</span>';
        }
    }
}
add_action('manage_posts_custom_column', 'sengeku_display_meta_column', 10, 2);
add_action('manage_pages_custom_column', 'sengeku_display_meta_column', 10, 2);

/**
 * Add Quick Edit field
 */
function sengeku_quick_edit_field($column_name, $post_type) {
    if ($column_name != 'meta_description') return;
    ?>
    <fieldset class="inline-edit-col-right">
        <div class="inline-edit-col">
            <label>
                <span class="title">Meta Description</span>
                <span class="input-text-wrap">
                    <textarea name="meta_description" rows="2" style="width: 100%;"></textarea>
                </span>
            </label>
        </div>
    </fieldset>
    <?php
}
add_action('quick_edit_custom_box', 'sengeku_quick_edit_field', 10, 2);

