<?php

/**
 * Plugin Name: Sendit Israel
 * Description: Sendit Israel SMS integration for WordPress, WooCommerce and Contact Form 7.
 * Version: 1.0.0
 * Author: UP System
 * Text Domain: sendit-israel
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) exit;

function sendit_is_woocommerce_active()
{
    return in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins', [])), true);
}

function sendit_sanitize_checkbox($value)
{
    return $value === '1' ? '1' : '0';
}

function sendit_register_settings()
{
    register_setting('sendit_settings', 'sendit_api_token', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => '',
    ]);

    register_setting('sendit_settings', 'sendit_sender', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => '',
    ]);

    register_setting('sendit_settings', 'sendit_enable_status_sms', [
        'type'              => 'string',
        'sanitize_callback' => 'sendit_sanitize_checkbox',
        'default'           => '0',
    ]);

    if (function_exists('wc_get_order_statuses')) {
        foreach (wc_get_order_statuses() as $status_key => $label) {
            register_setting('sendit_settings', 'sendit_sms_enabled_' . $status_key, [
                'type'              => 'string',
                'sanitize_callback' => 'sendit_sanitize_checkbox',
                'default'           => '0',
            ]);

            register_setting('sendit_settings', 'sendit_sms_template_' . $status_key, [
                'type'              => 'string',
                'sanitize_callback' => 'sanitize_textarea_field',
                'default'           => '',
            ]);
        }
    }

    register_setting('sendit_settings', 'sendit_cf7_enable', [
        'type'              => 'string',
        'sanitize_callback' => 'sendit_sanitize_checkbox',
        'default'           => '0',
    ]);

    register_setting('sendit_settings', 'sendit_cf7_phone_field', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => 'phone',
    ]);

    register_setting('sendit_settings', 'sendit_cf7_message', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_textarea_field',
        'default'           => '',
    ]);
}
add_action('admin_init', 'sendit_register_settings');

function sendit_admin_menu()
{
    add_submenu_page(
        'woocommerce',
        'Sendit Israel',
        'Sendit Israel',
        'manage_woocommerce',
        'sendit-israel',
        'sendit_settings_page'
    );
}
add_action('admin_menu', 'sendit_admin_menu');

function sendit_settings_page()
{
    if (!sendit_is_woocommerce_active()) {
        echo '<div class="notice notice-error"><p>תוסף WooCommerce אינו פעיל. אינטגרציית Sendit דורשת WooCommerce פעיל.</p></div>';
    }

    $api_token     = esc_attr(get_option('sendit_api_token', ''));
    $enable_sms    = get_option('sendit_enable_status_sms', '0');
    $statuses      = function_exists('wc_get_order_statuses') ? wc_get_order_statuses() : [];
    $cf7_enable    = get_option('sendit_cf7_enable', '0');
    $cf7_phone     = esc_attr(get_option('sendit_cf7_phone_field', 'phone'));
    $cf7_message   = esc_textarea(get_option('sendit_cf7_message', ''));
?>
    <div class="wrap">
        <h1>Sendit – הגדרות אינטגרציה</h1>

        <form method="post" action="options.php">
            <?php settings_fields('sendit_settings'); ?>
            <?php do_settings_sections('sendit_settings'); ?>

            <h2>חיבור ל-Sendit</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">מפתח API (Token)</th>
                    <td>
                        <input type="text" name="sendit_api_token" class="regular-text ltr" value="<?php echo $api_token; ?>" placeholder="15|XXXXXXXXXXXXXXXXXXXXXXXX">
                    </td>
                </tr>
                <tr>
                    <th scope="row">זיהוי שולח (Sender)</th>
                    <td>
                        <input type="text" name="sendit_sender" class="regular-text"
                            value="<?php echo esc_attr(get_option('sendit_sender', '')); ?>"
                            placeholder="MyBusiness">
                    </td>
                </tr>
            </table>

            <hr>

            <h2>הודעות סמס לפי סטטוס הזמנה</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">הפעלת הודעות סמס</th>
                    <td>
                        <label>
                            <input type="checkbox" name="sendit_enable_status_sms" value="1" <?php checked($enable_sms, '1'); ?>>
                            שליחת הודעת סמס אוטומטית ללקוח בעת שינוי סטטוס הזמנה (בהתאם להגדרות לכל סטטוס).
                        </label>
                    </td>
                </tr>
            </table>

            <?php if (!empty($statuses)) : ?>
                <h3>תבניות הודעות לפי סטטוס הזמנה</h3>
                <p class="description">
                    ניתן להשתמש במשתנים הבאים בתוך הטקסט:
                    <code>{order_id}</code>, <code>{status}</code>, <code>{total}</code>, <code>{currency}</code>,
                    <code>{first_name}</code>, <code>{last_name}</code>, <code>{site_name}</code>
                </p>
                <table class="form-table">
                    <?php foreach ($statuses as $status_key => $label) :
                        $enabled_option = 'sendit_sms_enabled_' . $status_key;
                        $template_option = 'sendit_sms_template_' . $status_key;
                        $enabled_value = get_option($enabled_option, '0');
                        $template_value = esc_textarea(get_option($template_option, ''));
                    ?>
                        <tr>
                            <th scope="row"><?php echo esc_html($label) . ' (' . esc_html($status_key) . ')'; ?></th>
                            <td>
                                <label style="display:flex;align-items:center;gap:8px;margin-bottom:6px;">
                                    <input type="checkbox" name="<?php echo esc_attr($enabled_option); ?>" value="1" <?php checked($enabled_value, '1'); ?>>
                                </label>
                                <textarea name="<?php echo esc_attr($template_option); ?>" rows="2" class="large-text" placeholder="נוסח הודעת ה-SMS לסטטוס זה..."><?php echo $template_value; ?></textarea>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </table>
            <?php endif; ?>

            <hr>

            <h2>Contact Form 7</h2>
            <table class="form-table">
                <tr>
                    <th scope="row">הפעלת אינטגרציה עם Contact Form 7</th>
                    <td>
                        <label>
                            <input type="checkbox" name="sendit_cf7_enable" value="1" <?php checked($cf7_enable, '1'); ?>>
                            שליחת אירועים ל-Sendit בעת שליחת טופס Contact Form 7
                        </label>
                    </td>
                </tr>
                <tr>
                    <th scope="row">שם שדה הטלפון בטופס</th>
                    <td>
                        <input type="text" name="sendit_cf7_phone_field" class="regular-text" value="<?php echo $cf7_phone; ?>" placeholder="phone">
                        <p class="description">שם השדה בטופס המכיל את מספר הטלפון של הפונה (למשל: <code>phone</code>).</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">תבנית הודעת סמס לפונה</th>
                    <td>
                        <textarea name="sendit_cf7_message" rows="2" class="large-text" placeholder="תודה שפניתם אלינו, נחזור אליכם בהקדם."><?php echo $cf7_message; ?></textarea>
                        <p class="description">
                            ניתן להשתמש במשתנים:
                            <code>{site_name}</code>, <code>{form_name}</code>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button('שמירת הגדרות'); ?>
        </form>
    </div>
<?php
}

function sendit_send_event($type, array $payload = [])
{
    $token = trim(get_option('sendit_api_token', ''));
    if ($token === '') {
        return;
    }

    $url = 'https://sendit.co.il/api/wordpress/v1/events';

    $body = [
        'type'    => $type,
        'source'  => 'wordpress',
        'site'    => [
            'url'  => home_url(),
            'name' => get_bloginfo('name'),
        ],
        'payload' => $payload,
    ];

    wp_remote_post($url, [
        'headers' => [
            'Content-Type'  => 'application/json',
            'Authorization' => 'Bearer ' . $token,
        ],
        'body'    => wp_json_encode($body),
        'timeout' => 10,
    ]);
}

function sendit_on_order_status_changed($order_id, $old_status, $new_status, $order)
{
    if (!sendit_is_woocommerce_active()) {
        return;
    }

    $enabled_global = get_option('sendit_enable_status_sms', '0');
    if ($enabled_global !== '1') {
        return;
    }

    if (!$order instanceof WC_Order) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }
    }

    $phone = $order->get_billing_phone();
    if (empty($phone)) {
        return;
    }

    $status_key = 'wc-' . $new_status;
    $enabled_option = 'sendit_sms_enabled_' . $status_key;
    $template_option = 'sendit_sms_template_' . $status_key;

    $enabled_value = get_option($enabled_option, '0');
    if ($enabled_value !== '1') {
        return;
    }

    $template = get_option($template_option, '');
    if (trim($template) === '') {
        return;
    }

    $message = strtr($template, [
        '{order_id}'   => $order->get_id(),
        '{status}'     => $new_status,
        '{total}'      => $order->get_total(),
        '{currency}'   => $order->get_currency(),
        '{first_name}' => $order->get_billing_first_name(),
        '{last_name}'  => $order->get_billing_last_name(),
        '{site_name}'  => get_bloginfo('name'),
    ]);

    if (trim($message) === '') {
        return;
    }

    $payload = [
        'phone'    => $phone,
        'message'  => $message,
        'sender'   => get_option('sendit_sender', ''),
    ];

    sendit_send_event('sms.order_status_changed', $payload);
}
add_action('woocommerce_order_status_changed', 'sendit_on_order_status_changed', 10, 4);

function sendit_cf7_event($contact_form)
{
    if (get_option('sendit_cf7_enable', '0') !== '1') {
        return;
    }

    if (!class_exists('WPCF7_Submission')) {
        return;
    }

    $submission = WPCF7_Submission::get_instance();
    if (!$submission) {
        return;
    }

    $form_id   = $contact_form->id();
    $form_name = $contact_form->title();
    $posted    = $submission->get_posted_data();

    $payload = [
        'form_id'      => $form_id,
        'form_name'    => $form_name,
        'submitted_at' => current_time('mysql'),
        'fields'       => $posted,
        'user'         => [
            'ip'         => $submission->get_meta('remote_ip'),
            'user_agent' => $submission->get_meta('user_agent'),
            'url'        => $submission->get_meta('url'),
        ],
    ];

    sendit_send_event('contact.form_submitted', $payload);

    $phone_field = get_option('sendit_cf7_phone_field', 'phone');
    $sms_template = get_option('sendit_cf7_message', '');

    if (trim($sms_template) === '') {
        return;
    }

    if (!isset($posted[$phone_field]) || empty($posted[$phone_field])) {
        return;
    }

    $phone = $posted[$phone_field];

    $message = strtr($sms_template, [
        '{site_name}' => get_bloginfo('name'),
        '{form_name}' => $form_name,
    ]);

    if (trim($message) === '') {
        return;
    }

    $sms_payload = [
        'form_id'   => $form_id,
        'form_name' => $form_name,
        'phone'     => $phone,
        'message'   => $message,
    ];

    sendit_send_event('sms.contact_form', $sms_payload);
}
add_action('wpcf7_mail_sent', 'sendit_cf7_event');
