<?php
#region Plugin Details
/**
 * This file is read by WordPress to generate the plugin information in the plugin
 * admin area. This file also includes all of the dependencies used by the plugin,
 * registers the activation and deactivation functions, and defines a function
 * that starts the plugin.
 * 
 * @package   Send2CRM Wordpress Plugin
 * @author    FuseIT  support@fuseit.com
 * @copyright 2025 Fuse Information Technologies Ltd
 * @license   GPL v2 or later
 * @link      https://fuseit.com
 * 
 * Plugin Name: Send2CRM
 * Plugin URI:      https://github.com/FuseInfoTech/wordpress-send2crm-plugin/
 * Description:     Easily integrate your WordPress site with your CRM through FuseIT’s official Send2CRM Wordpress plugin.
 * Version: 1.0.1
 * Author: FuseIT
 * Author URI: https://fuseit.com
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires PHP:    8.1
 * Requires at least: 6.5
 * 
 */
#endregion
// In strict mode, only a variable of exact type of the type declaration will be accepted.
declare(strict_types=1);
namespace Send2CRM;

#region Includes
use Send2CRM\Admin\Settings;
use Send2CRM\Admin\VersionManager;
use Send2CRM\Public\Snippet;
#endregion

// If this file is called directly, abort.
if (!defined('ABSPATH')) exit;

// Autoloader allows the use of namespaces and use statement (PSR-4)
require_once plugin_dir_path(__FILE__) . 'Autoloader.php';


#region Constants
/**
 * Current plugin version.
 * Start at version 1.0.0 and use SemVer - https://semver.org
 * Rename this for your plugin and update it as you release new versions.
 */
define('SEND2CRM_VERSION', '1.0.1');

// The string used to uniquely identify this plugin.
define('SEND2CRM_SLUG', 'send2crm');

// The name to for the Settings Menu, Page and Title
define('SEND2CRM_MENU_NAME', 'Send2CRM');
#endregion
class Send2CRM {
    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     */
    protected string $slug;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     */
    protected string $version;

    /**
     * The name to for the Settings Menu, Page and Title
     * 
     * @since    1.0.0
     */
    protected string $menuName;

    /**
     * A reference to the plugin settings class so that is is accesible.
     *
     * @since    1.0.0
     */
    public Settings $settings;

    /**
     * A reference to the Snippet class that inserts the Send2CRM.js.
     *
     * @since    1.0.0
     */
    public Snippet $snippet;

    /**
     * A reference to the VersionManager class that handles version updates in the admin page.
     *
     * @since    1.0.0
     */
    public VersionManager $versionManager;

    /**
     * Indicates if the plugin hooks has been initialized so we don't double hook.
     *  
     */
    public bool $isInitialized = false;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->version = SEND2CRM_VERSION;
        $this->slug = SEND2CRM_SLUG;
        $this->menuName = SEND2CRM_MENU_NAME;
        
        //Register settings,but the hook initialization should only run on Admin area only.
        $this->settings = new Settings($this->slug, $this->menuName);
        $this->snippet = new Snippet($this->settings, $this->version);
        $this->versionManager = new VersionManager($this->settings, $this->version);
        if ($this->isInitialized) return;
        $this->initialize_hooks();
    }

    /**
     * Initialize the hooks of the plugin so plugin functions are called correctly.
     * 
     * @since    1.0.0
     */
    public function initialize_hooks() {

        $isAdmin = is_admin();

        if ($isAdmin)
        {
            add_filter( 'plugin_action_links_' . plugin_basename(__FILE__), array($this,'add_action_links') );
            $this->settings->initialize_hooks($isAdmin);
            $this->versionManager->initialize_hooks($isAdmin);
        }
        $this->snippet->initialize_hooks($isAdmin);
        $this->isInitialized = true;
    }

    #region Callbacks
    
    /** 
     *  Add a settings link to the plugin page that opens the plugin settings page.
     * 
     * @since    1.0.0
     * @param   array   $links  The standard links on to the plugin page.
     * @return  array           The changed list of links with our custom links added.
     */
    public function add_action_links(array $links): array {
        $customLinks = array(
            '<a href="' . admin_url( "admin.php?page={$this->slug}" ) . '">Settings</a>',
        );
        return array_merge( $customLinks,$links);
    }
    #endregion
}

//Start the plugin
new Send2CRM();













