<?php

declare(strict_types=1);

namespace Send2CRM\Admin;

#region Includes
use Send2CRM\Admin\Settings;
#endregion  

// If this file is called directly, abort.
if (!defined('ABSPATH')) exit;

#region Constants
//TODO create a constants.php for shared constants ones required for a specific class should use live in the class and use the contanst keyword instead

define('SEND2CRM_VERSION_MANAGER_FILENAME', 'js/version-manager.js');
define('SEND2CRM_GITHUB_USERNAME', 'FuseInfoTech');
define('SEND2CRM_GITHUB_REPO', 'send2crmjs');
define('SEND2CRM_MINIMUM_VERSION', '1.21.0');
define('SEND2CRM_UPLOAD_FOLDERNAME', '/send2crm-releases/');
define('SEND2CRM_HASH_FILENAME', 'send2crm.sri-hash.sha384');
define('SEND2CRM_JS_FILENAME', 'send2crm.min.js');
define('SEND2CRM_CDN_URL', 'https://cdn.jsdelivr.net'); 
define('SEND2CRM_CDN', SEND2CRM_CDN_URL .'/gh/'. SEND2CRM_GITHUB_USERNAME . '/' . SEND2CRM_GITHUB_REPO);
#endregion

/**
 * Manages fetching, display and installing of Send2CRM versions used for the plugin
 *
 * @since      1.0.0
 */
class VersionManager {

    /** 
     * Instance holding Send2CRM settings so that they can be accessed when hooks are executed
     * 
     * @since 1.0.0
     */
    private Settings $settings;


    /** 
     * Plugin version for registering and enqueuing the send2crm javascript snippet.
     * 
     * @since 1.0.0
     */
    private string $version;

    private string $githubRepo;

    private string $githubUsername;

    private string $minimumVersion;

public function __construct(Settings $settings, string $version) {
        $this->settings = $settings;
        $this->version = $version;
        $this->githubRepo = SEND2CRM_GITHUB_REPO;
        $this->githubUsername = SEND2CRM_GITHUB_USERNAME;
        $this->minimumVersion = SEND2CRM_MINIMUM_VERSION;
        $this->initialize_settings();
    }

    public function initialize_settings() {

        //Create section for cookies settings
        $versionSectionName = $this->settings->add_section(
            'version', 
            'Library Version & Delivery', 
            'Choose which version of the website Send2CRM client to load and how to deliver it.',
        );

        $this->settings->add_field(
            'js_version',
            'Version<span style="color: #d63638;">*</span>', 
            array($this, 'render_version_input'), 
            "Select which version of the website Send2CRM client to use on your site. After selecting a version and pressing Save, the security verification code below will automatically update with the integrity hash for that version.", 
            $versionSectionName,
            type: 'version',
        );

        $this->settings->add_field(
            'use_cdn',
            'Delivery Method', 
            array($this, 'render_cdn_input'), 
            "<strong>✓ Checked (CDN):</strong> Send2CRM will be loaded into your visitors browser from jsDeliver CDN - faster loading, automatic caching, reduced server load <br/><strong>○ Unchecked (Local):</strong> Downloads and serves a local copy from your WordPress server - use this to comply with strict security requirements.", 
            $versionSectionName,
            type: 'checkbox',
        );

        $verificationSectionName = $this->settings->add_section(
            'security-verification',
            'Security Verification', 
            'Subresource Integrity (SRI) hash for the selected library version.',
        );

        $this->settings->add_field(
            'js_hash',
            'File Verification Code', 
            array($this, 'render_hash_input'), 
            "<a href='https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity' target='_blank'>What is this?</a><br/>This is a cryptographic hash specific to the selected version of Send2CRM. It's used for Subresource Integrity (SRI) checks to ensure the JavaScript file hasn't been tampered with. The web browser verifies the file matches this hash before executing it, providing protection against compromised CDN files or man-in-the-middle attacks. This field populates automatically when you select a version and save your settings.", 
            $verificationSectionName, 
        );


    }

    /**
     * Register all the hooks of this class.
     *
     * @since    1.0.0
     * @param  bool    $isAdmin    Whether the current request is for an administrative interface page.
     */
    public function initialize_hooks(bool $isAdmin): void {
        if ($isAdmin) {
            //Hook on admin page to add javascript
            add_action('admin_enqueue_scripts', array($this,'insert_version_manager_scripts'));
            //Hook on ajax call to retrieve send2crm releases
            add_action('wp_ajax_send2crm_fetch_releases', array($this, 'ajax_fetch_releases'));
            //Hook to filter send2crm settings before they are saved to database
            add_filter('pre_update_option_send2crm_settings_option', array($this, 'filter_version_settings'),10,3);
        }
    }

    #region Callbacks
    
    /**
     * Render the Send2CRM.js version input select field.
     *
     * @since    1.0.0
     * @param   array   $arguments  Array of arguments passed to the function by the action hook.
     */
    public function render_version_input($arguments) : void {
        $fieldId = $arguments['id'];
        $fieldDetails = $this->settings->get_field($fieldId);
        // Get the current saved value 
        $optionGroup = $fieldDetails['option_group'];
        $value = $this->settings->get_setting($fieldId, $optionGroup); 
        $settingName = $this->settings->get_setting_name($fieldId, $optionGroup);
        $description = $fieldDetails['description'];
        // Render the input field 
        echo "<select required id='".esc_attr($fieldId)."' name='".esc_attr($settingName)."' data-current-version='".esc_attr($value)."'>";
        echo "<option value='' selected>Click 'Fetch Versions' to load available versions</option>";
        echo "</select>";
        echo "<button id='fetch-releases' class='button button-primary'><span id='fetch-icon' style='vertical-align: sub;' class='dashicons dashicons-update'></span></button>";
        if (empty($description)) {
            return;
        }
        echo "<p class='description'>".esc_html($description)."</p>";
    }

    /**
     * Render the Hash input text field.
     *
     * @since    1.0.0
     * @param   array   $arguments  Array of arguments passed to the function by the action hook.
     */
    public function render_hash_input(array $arguments): void {
        $fieldId = $arguments['id'];
        $fieldDetails = $this->settings->get_field($fieldId);
        // Get the current saved value 
        $optionGroup = $fieldDetails['option_group'];
        $value = $this->settings->get_setting($fieldId); 
        $settingName = $this->settings->get_setting_name($fieldId,$optionGroup);
        $description = $fieldDetails['description'];
        // Render the input field 
        echo "<input class='large-text' placeholder='Select a version above and save changes to generate hash' readonly type='text' id='" . esc_attr($fieldId) . "' name='" . esc_attr($settingName) . "' value='" . esc_attr($value) . "'>";
        if (empty($description)) {
            return;
        }
        echo "<p class='description'>".wp_kses_post($description)."</p>";
    }

    /**
     * Render the Use CDN input checkbox field.
     *
     * @since    1.0.0
     * @param   array   $arguments  Array of arguments passed to the function by the action hook.
     */
    public function render_cdn_input(array $arguments): void {
        $fieldId = $arguments['id'];
        $fieldDetails = $this->settings->get_field($fieldId);
        // Get the current saved value 
        $optionGroup = $fieldDetails['option_group'];
        $value = $this->settings->get_setting($fieldId,$optionGroup); 
        $settingName = $this->settings->get_setting_name($fieldId, $optionGroup);
        $description = $fieldDetails['description'];
        // Render the input field 
        $checked = checked($value, 1, false);
        echo "<input type='checkbox' id='" . esc_attr($fieldId) . "' name='" . esc_attr($settingName) . "' value='1' " . esc_html($checked) . "> Use Content Delivery Network";
        if (empty($description)) {
            return;
        }
        echo "<p class='description'>". wp_kses_post($description). "</p>";
    }

    /**
     * Callback for the filter pre_update_option hook that updates the hash and version based on the input.
     * 
     * @since    1.0.0
     * @param  array    $newValue   The new value of the option.
     * @param  array    $currentValue   The current value of the option.
     * @param  string   $optionName   The name of the option.
     * @return array    The updated value of the option which will be saved to the database.
     */
    public function filter_version_settings(array $newValue = [], array | string $currentValue  = "", $optionName  = "") : array {
        if ( empty($optionName) || empty($newValue) || $optionName != 'send2crm_settings_option') {
            return $newValue;
        }
        $currentVersion = '';
        $currentUseCDN = false;
        if (empty($currentValue) === false) {
            $currentVersion = array_key_exists('js_version', $currentValue) ? $currentValue['js_version'] : '';
            $currentUseCDN = (bool)($currentValue['use_cdn'] ?? false);
        }
        $newVersion = array_key_exists('js_version', $newValue) ? $newValue['js_version'] : $currentVersion;
        //We don't fall back to the currentUseCDN value because this is a checkbox so the if `use_cdn` is not set then the checkbox was not ticked. There is never a need to fall back to the current value.
        $newUseCDN =(bool)($newValue['use_cdn'] ?? false);
        $updateHash = false;
        $downloadJS = false;
        $removeJS = false;
        if ($currentVersion !== $newVersion) {
            $updateHash = true;
        } else if ($newUseCDN !== $currentUseCDN) {
            if ($newUseCDN && $this->release_file_exists($newVersion)) { 
                //TODO Decide if we want to remove the Javascript file if we aren't using the CDN. Would we want to remove all versions or only the current one?
                $removeJS = true;
            }
        }
        // If we aren't using the CDN but JS file is missing then download it anyway to avoid errors finding the JS.
        if ($newUseCDN === false && $this->release_file_exists($newVersion) === false) {
                $downloadJS = true;
        }

        if ($updateHash) {
            $newHash = $this->get_hash(SEND2CRM_CDN . "@{$newVersion}/");
            if (empty($newHash)) {
                add_settings_error( 'js_version', esc_attr( 'settings_updated' ), "Unable to update to {$newVersion}", 'error' );
                $newValue['js_version'] = $currentVersion;
                $newValue['use_cdn'] = $currentUseCDN ? '1' : '';
                return $newValue;
            }
            $newValue['js_hash'] = $newHash; //TODO Check the hash is valid before saving?
        }

        if ($downloadJS && $this->check_integrity()) {
            $results = $this->download_release_files($newVersion);
            if ($results['success'] === false) {
                $newValue['js_version'] = $currentVersion;
                $newValue['use_cdn'] = $currentUseCDN ? '1' : '';
                add_settings_error( 'js_version', esc_attr( 'settings_updated' ), "Unable to update to {$newVersion}", 'error' );       
                return $newValue;
            }
            add_settings_error( 'js_version', esc_attr( 'settings_updated' ), $newVersion . ' downloaded successfully!', 'updated' );
        }

        return $newValue;
    }
    /**
     * Check the integrity of the Send2CRM JS file at the provided location using the hash stored in the database.
     * This should be an extra security check in future, but currently not required.
     * 
     * 
     * @since 1.0.0
     * @return bool whether the Send2CRM JS file is valid.
     */
    public function check_integrity() :bool {
         //TODO implement integrity check fully
        return true;
    }

    /**
     * Remove the Send2CRM JS file at the provided location.
     *  
     * @since 1.0.0
     * 
     * @param  string  $version  The version of the Send2CRM JS file to remove
     * @return bool whether the Send2CRM JS file was successfully removed
     */
    public function remove_release_files($version) : bool {
        $upload_dir = wp_upload_dir();
        $success = false;

        if (file_exists($upload_dir['basedir'] . SEND2CRM_UPLOAD_FOLDERNAME . $version . '/' . SEND2CRM_JS_FILENAME)) {
            $success = wp_delete_file($upload_dir['basedir'] . SEND2CRM_UPLOAD_FOLDERNAME . $version . '/' . SEND2CRM_JS_FILENAME) ?? false;
        }
        return $success;
    }


    /**
     * Check if the Send2CRM JS file at the provided location exists.
     *  
     * @since 1.0.0
     * 
     * @param  string  $version  The version of the Send2CRM JS file to check
     * @return bool whether the Send2CRM JS file exists
     */
    public function release_file_exists($version): bool {
        $upload_dir = wp_upload_dir();
        return file_exists($upload_dir['basedir'] . SEND2CRM_UPLOAD_FOLDERNAME . $version . '/' . SEND2CRM_JS_FILENAME);
    }

    /**
     * Add the Send2CRM JS version manager script to the admin page.
     * 
     * @since 1.0.0
     * 
     * @param mixed $hook_suffix The admin page suffix to determin which page the function is running on.
     */
    public function insert_version_manager_scripts(mixed $hook_suffix) : void {
        if( $hook_suffix !== 'settings_page_' . $this->settings->pluginSlug ) {
            return;
        };
        $versionManagerJSUrl = plugin_dir_url( __FILE__ ) . SEND2CRM_VERSION_MANAGER_FILENAME;
        $versionManagerJsPath = plugin_dir_path( __FILE__ ) . SEND2CRM_VERSION_MANAGER_FILENAME;
        $versionManagerJSId = "{$this->settings->pluginSlug}-version-manager";
        $versionManagerJSVersion = file_exists($versionManagerJsPath) ? filemtime($versionManagerJsPath) : $this->version;

        if (wp_register_script( $versionManagerJSId, $versionManagerJSUrl, array('jquery'), $versionManagerJSVersion, false ) === false) 
        {
            add_settings_error( 'js_version', esc_attr('settings_updated'), "Unable to register Send2CRM version manager script.", 'error' );
            return;
        }
        
        wp_enqueue_script(
            $versionManagerJSId,
            $versionManagerJSUrl,
            array('jquery'),
            $this->version,
            false
        );

        $upload_dir = wp_upload_dir();
        //TODO update this script and object to descripe its function that that only fetches releases
        wp_localize_script($versionManagerJSId, 'send2crmReleases', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('send2crm_releases_nonce'),
            //TODO: Remove hard coded element id and reference it from the field
            'version_element_id' => "js_version",
        ));
    }

    #endregion

    #region Private Functions
    /**
     * Get the content of the hash file at the provided location(folder).It will check for the SEND2CRM_HASH_FILENAME file.
     * 
     * @since 1.0.0
     * 
     * @param  string  $location  The folder where the hash file is located of the hash file.
     * @return string the content of the hash file or an empty string if the file does not exist
     */
    private function get_hash(string $location): string {
        //TODO Add checks for bad paths to prevent critical errors
        $hash = file_get_contents($location . SEND2CRM_HASH_FILENAME);
        if (!$hash) {
            return '';
        }
        return $hash;
    }

     /**
     * AJAX handler for fetching releases
     * 
     * @since 1.0.0
     * 
     */
    public function ajax_fetch_releases() {
        check_ajax_referer('send2crm_releases_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $result = $this->fetch_releases();
        wp_send_json($result);
    }


    /**
     * Fetch releases from GitHub API
     * 
     * @since 1.0.0
     * @return array the result of fetching the releases including if the request was successful and releases as an array.
     */
    public function fetch_releases(): array {
        $api_url = "https://api.github.com/repos/{$this->githubUsername}/{$this->githubRepo}/releases";
        
        $response = wp_remote_get($api_url, array(
            'headers' => array(
                'Accept' => 'application/vnd.github.v3+json',
                'User-Agent' => 'WordPress-Plugin'
            ),
            'timeout' => 15
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $releases = json_decode($body, true);
        
        if (!is_array($releases)) {
            return array(
                'success' => false,
                'message' => 'Invalid response from GitHub API'
            );
        }
        
        // Filter releases by minimum version
        $filtered_releases = $this->filter_by_minimum_version($releases);
        return array(
            'success' => true,
            'releases' => $filtered_releases
        );
    }

    /**
     * Filter releases by minimum version
     * 
     * @since 1.0.0
     * 
     * @param array $releases the array of releases to filter
     * @return array the filtered array of releases that are greater than or equal to the SEND2CRM_MINIMUM_VERSION constant
     */
    private function filter_by_minimum_version(array $releases) : array {
        if (empty($this->minimumVersion)) {
            return $releases;
        }
        
        $filtered = array();
        
        foreach ($releases as $release) {
            $version = ltrim($release['tag_name'], 'v');
            
            if (version_compare($version, $this->minimumVersion, '>=')) {
                $filtered[$release['tag_name']] = $release;
            }
        }
        
        return $filtered;
    }

    /**
     * Download specific files from a Send2CRM release
     * 
     * @since 1.0.0
     * 
     * @param string $tag_name the tag name of the release to download
     * @return array the result of downloading the files including if the request was successful and file paths as an array
     */
    public function download_release_files(string $tag_name): array {
        // Files to download
        $files_to_download = array(
            SEND2CRM_JS_FILENAME,
        );
        
        // Create a downloads directory in wp-content/uploads
        $upload_dir = wp_upload_dir();
        $download_dir = $upload_dir['basedir'] . SEND2CRM_UPLOAD_FOLDERNAME . $tag_name;
        
        if (!file_exists($download_dir)) {
            wp_mkdir_p($download_dir);
        }
        
        $results = array();
        $all_success = true;
        
        foreach ($files_to_download as $filename) {
            // Construct raw file URL
            $file_url = "https://raw.githubusercontent.com/{$this->githubUsername}/{$this->githubRepo}/{$tag_name}/{$filename}";

            $file_path = $download_dir . '/' . $filename;
            
            // Check if file already exists
            if (file_exists($file_path)) {
                $results[$filename] = array(
                    'success' => true,
                    'message' => 'File already exists',
                    'file_path' => $file_path,
                    'file_url' => $upload_dir['baseurl'] . SEND2CRM_UPLOAD_FOLDERNAME . $tag_name . '/' . $filename,
                    'skipped' => true
                );
                continue;
            }
            
            // Download the file
            $response = wp_remote_get($file_url, array(
                'timeout' => 60,
                'headers' => array(
                    'User-Agent' => 'WordPress-Plugin'
                )
            ));
            
            if (is_wp_error($response)) {
                $results[$filename] = array(
                    'success' => false,
                    'message' => 'Download failed: ' . $response->get_error_message()
                );
                $all_success = false;
                continue;
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            
            if ($response_code !== 200) {
                $results[$filename] = array(
                    'success' => false,
                    'message' => 'File not found (HTTP ' . $response_code . ')'
                );
                $all_success = false;
                continue;
            }
            
            // Save file content
            $file_content = wp_remote_retrieve_body($response);
            $saved = file_put_contents($file_path, $file_content);
            
            if ($saved === false) {
                $results[$filename] = array(
                    'success' => false,
                    'message' => 'Failed to save file'
                );
                $all_success = false;
                continue;
            }

            $results[$filename] = array(
                'success' => true,
                'message' => 'Downloaded successfully',
                'file_path' => $file_path,
                'file_url' => $upload_dir['baseurl'] . SEND2CRM_UPLOAD_FOLDERNAME . $tag_name . '/' . $filename,
                'file_size' => size_format(filesize($file_path))
            );
        }

        return array(
            'success' => $all_success, 
            'message' => $all_success ? 'All files downloaded successfully' : 'Some files failed to download',
            'files' => $results,
            'download_dir' => $download_dir,
            'upload_url' => $upload_dir['baseurl'] . SEND2CRM_UPLOAD_FOLDERNAME . $tag_name  . '/',
            'version' => $tag_name
        );
    }

}