<?php
/**
 * Plugin Name: Send Your Prayers
 * Description: A prayer submission system with one-time payments
 * Author: DigitalME
 * Author URI: https://www.digitalme.cc
 * Version: 1.6.2
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: send-your-prayers
 * Domain Path: /languages
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
if (!defined('SYPRAYER_PLUGIN_URL')) {
    define('SYPRAYER_PLUGIN_URL', plugin_dir_url(__FILE__));
}
if (!defined('SYPRAYER_PLUGIN_PATH')) {
    define('SYPRAYER_PLUGIN_PATH', plugin_dir_path(__FILE__));
}

// Get version from plugin header
function syprayer_get_version() {
    if (!function_exists('get_plugin_data')) {
        require_once(ABSPATH . 'wp-admin/includes/plugin.php');
    }
    $plugin_data = get_plugin_data(__FILE__);
    return $plugin_data['Version'];
}

class Send_Your_Prayers_Plugin {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        
        add_action('plugins_loaded', array($this, 'init'));
    }
    
    public function activate() {
        if (!current_user_can('activate_plugins')) {
            return;
        }
        
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-db.php';
        
        SYPRAYER_DB::create_tables();
        
        $default_options = array(
            'currency' => 'USD',
            'simple_amount' => '1',
            'prayer_limit' => 5,
            'form_description' => 'Share your prayer request with our community. Your prayer will be received and considered by our church.',
            'stripe_test_mode' => 'yes',
            'stripe_test_pk' => '',
            'stripe_test_sk' => '',
            'stripe_live_pk' => '',
            'stripe_live_sk' => '',
            'paypal_test_mode' => 'yes',
            'paypal_client_id' => '',
            'paypal_client_secret' => '',
        );
        
        if (false === get_option('syprayer_settings')) {
            add_option('syprayer_settings', $default_options);
        }
        
        // Add version to database for future updates
        update_option('syprayer_version', syprayer_get_version());
        
        return true;
    }
    
    public function deactivate() {
        if (!current_user_can('activate_plugins')) {
            return;
        }
        return true;
    }
    
    public function init() {
    $this->includes();
    
    new SYPRAYER_Admin();
    new SYPRAYER_Form();
    new SYPRAYER_Payments();
    new SYPRAYER_PayPal();
    new SYPRAYER_ThankYou();
    new SYPRAYER_Payment_Handler();
    
    add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
    add_action('admin_notices', array($this, 'admin_notices'));
    
    add_shortcode('send_your_prayers', array('SYPRAYER_Form', 'render_form'));
    
    // Add cleanup action to clear payment cookies on page load
    add_action('init', array($this, 'maybe_clear_payment_cookies'));
}
    
    private function includes() {
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-db.php';
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-admin.php';
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-form.php';
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-payments.php';
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-paypal.php';
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-thankyou.php';
        require_once SYPRAYER_PLUGIN_PATH . 'includes/class-syprayer-payment-handler.php';
    }
    
    public function enqueue_scripts() {
        $version = syprayer_get_version();
        
        // Register and enqueue frontend CSS
        wp_register_style('syprayer-style', SYPRAYER_PLUGIN_URL . 'assets/css/style.css', array(), $version);
        
        global $post;
        $should_enqueue_ajax = false;
        
        // Check if we're on a SYPRAYER page
        if (is_a($post, 'WP_Post')) {
            $has_main_form = has_shortcode($post->post_content, 'send_your_prayers');
            $has_thank_you = has_shortcode($post->post_content, 'syprayer_thank_you');
            
            $should_enqueue_ajax = $has_main_form || $has_thank_you;
            
            if ($has_main_form) {
                // Register Stripe with async attribute (WordPress 6.3+)
                wp_register_script('stripe-js', 'https://js.stripe.com/v3/', array(), '3.0', array(
                    'strategy' => 'async',
                    'in_footer' => true
                ));
                wp_enqueue_script('stripe-js');
            }
        }
        
        // Always enqueue style on SYPRAYER pages
        if ($should_enqueue_ajax) {
            wp_enqueue_style('syprayer-style');
            
            // Register and enqueue frontend JS - Load in footer to ensure Stripe is loaded
            wp_register_script('syprayer-script', SYPRAYER_PLUGIN_URL . 'assets/js/script.js', array('jquery'), $version, true);
            wp_enqueue_script('syprayer-script');
            
            $settings = get_option('syprayer_settings');
            
            // Get page URLs from settings
            $form_page_id = isset($settings['form_page_id']) ? $settings['form_page_id'] : 0;
            $thank_you_page_id = isset($settings['thank_you_page_id']) ? $settings['thank_you_page_id'] : 0;
            
            // Determine which key to use
            $stripe_test_mode = isset($settings['stripe_test_mode']) ? $settings['stripe_test_mode'] : 'yes';
            $stripe_pk = ($stripe_test_mode === 'yes') ? 
                (isset($settings['stripe_test_pk']) ? $settings['stripe_test_pk'] : '') : 
                (isset($settings['stripe_live_pk']) ? $settings['stripe_live_pk'] : '');
            
            // Get button colors with defaults
            $button_color = isset($settings['button_color']) ? $settings['button_color'] : '#27ae60';
            $button_hover_color = isset($settings['button_hover_color']) ? $settings['button_hover_color'] : '#219a52';
            
            // Check if PayPal is available
            $paypal_available = !empty($settings['paypal_client_id']) && !empty($settings['paypal_client_secret']);
            
            wp_localize_script('syprayer-script', 'syprayer_ajax', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('syprayer_nonce'),
                'stripe_pk' => sanitize_text_field($stripe_pk),
                'stripe_test_mode' => sanitize_text_field($stripe_test_mode),
                'currency' => isset($settings['currency']) ? sanitize_text_field($settings['currency']) : 'USD',
                'simple_amount' => isset($settings['simple_amount']) ? floatval($settings['simple_amount']) : 1,
                'form_page_url' => $form_page_id ? esc_url(get_permalink($form_page_id)) : esc_url(home_url('/')),
                'thank_you_page_url' => $thank_you_page_id ? esc_url(get_permalink($thank_you_page_id)) : esc_url(home_url('/thank-you')),
                'button_color' => sanitize_hex_color($button_color),
                'button_hover_color' => sanitize_hex_color($button_hover_color),
                'paypal_available' => $paypal_available,
                'debug' => false // Set to false for production
            ));
            
            // Add inline CSS for button colors with proper escaping
            $button_css = sprintf(
                ':root {
                    --syprayer-button-color: %s;
                    --syprayer-button-hover: %s;
                }
                
                #send-your-prayers-form .syprayer-submit-button,
                .syprayer-thank-you-page .syprayer-button {
                    background: %s !important;
                }
                
                #send-your-prayers-form .syprayer-submit-button:hover,
                .syprayer-thank-you-page .syprayer-button:hover {
                    background: %s !important;
                }',
                esc_html($button_color),
                esc_html($button_hover_color),
                esc_html($button_color),
                esc_html($button_hover_color)
            );
            wp_add_inline_style('syprayer-style', $button_css);
        }
    }
    
    public function maybe_clear_payment_cookies() {
        // Clear payment cookies if we're on the form page and not processing a payment
        global $post;
        if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'send_your_prayers')) {
            // Check if we're not coming from a payment redirect
            if (!isset($_GET['syprayer_success']) && !isset($_GET['gateway'])) {
                // Clear any existing payment cookies
                if (isset($_COOKIE['syprayer_payment_token'])) {
                    unset($_COOKIE['syprayer_payment_token']);
                    setcookie('syprayer_payment_token', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
                }
            }
        }
    }
    
    public function admin_notices() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return;
        }
        
        $settings = get_option('syprayer_settings');
        
        if (empty($settings['stripe_test_pk']) && empty($settings['stripe_live_pk']) && empty($settings['paypal_client_id'])) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <strong><?php echo esc_html__('Send Your Prayers:', 'send-your-prayers'); ?></strong> 
                    <?php 
                    printf(
                        /* translators: %s: Settings page URL */
                        esc_html__('Please configure your payment gateway settings in the %s to enable payments.', 'send-your-prayers'),
                        '<a href="' . esc_url(admin_url('admin.php?page=send-your-prayers')) . '">' . esc_html__('settings page', 'send-your-prayers') . '</a>'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }
}
Send_Your_Prayers_Plugin::get_instance();
?>