<?php
if (!defined('ABSPATH')) {
    exit;
}

class SYPRAYER_ThankYou {
    
    public function __construct() {
        // Add shortcode for thank you page
        add_shortcode('syprayer_thank_you', array($this, 'render_thank_you_page'));
        
        // Clear cookies on thank you page
        add_action('wp', array($this, 'clear_payment_cookies_on_thankyou'));
    }
    
    private function get_payment_info() {
        // First try to get from cookie
        if (isset($_COOKIE['syprayer_payment_token'])) {
            $token = sanitize_text_field(wp_unslash($_COOKIE['syprayer_payment_token']));
            $payment_data = get_transient('syprayer_payment_' . $token);
            
            if ($payment_data) {
                // Clean up after retrieval
                delete_transient('syprayer_payment_' . $token);
                $this->clear_payment_cookie();
                return $payment_data;
            } else {
                // Transient expired, clear cookie
                $this->clear_payment_cookie();
            }
        }
        
        // Check for direct URL parameters (for Stripe direct redirects)
        if (isset($_GET['gateway']) && isset($_GET['_wpnonce'])) {
            // Verify nonce for thank you access
            if (wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'syprayer_thankyou_access')) {
                $gateway = sanitize_text_field(wp_unslash($_GET['gateway']));
                return array(
                    'gateway' => $gateway,
                    'timestamp' => time()
                );
            }
        }
        
        return false;
    }
    
    private function clear_payment_cookie() {
        if (isset($_COOKIE['syprayer_payment_token'])) {
            unset($_COOKIE['syprayer_payment_token']);
        }
        setcookie('syprayer_payment_token', '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
    }
    
    public function clear_payment_cookies_on_thankyou() {
        global $post;
        // If we're on the thank you page, clear cookies after displaying content
        if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'syprayer_thank_you')) {
            $this->clear_payment_cookie();
        }
    }
    
    public function render_thank_you_page() {
        // Get payment data from secure storage
        $payment_data = $this->get_payment_info();
        
        if (empty($payment_data)) {
            return $this->render_no_payment_found();
        }
        
        return $this->render_thank_you_content($payment_data);
    }
    
    public function render_thank_you_content($payment_data) {
        // Ensure payment_data is sanitized
        $payment_data = array_map('sanitize_text_field', $payment_data);
        
        $settings = get_option('syprayer_settings');
        $form_page_id = isset($settings['form_page_id']) ? $settings['form_page_id'] : 0;
        $form_page_url = $form_page_id ? get_permalink($form_page_id) : home_url('/');
        
        ob_start();
        ?>
        <div class="syprayer-thank-you-page">
            <div class="syprayer-thank-you-header">
                <h1><?php echo esc_html__('Thank You for Your Prayer Submission!', 'send-your-prayers'); ?> 🙏</h1>
                <div class="syprayer-success-icon">✓</div>
            </div>
            
            <div class="syprayer-thank-you-content">
                <p><?php echo esc_html__('Your prayer has been received and your payment was successful.', 'send-your-prayers'); ?></p>
                
                <div class="syprayer-payment-details">
                    <h3><?php echo esc_html__('Payment Details', 'send-your-prayers'); ?></h3>
                    <ul>
                        <li><strong><?php echo esc_html__('Payment Gateway:', 'send-your-prayers'); ?></strong> <?php echo esc_html(ucfirst($payment_data['gateway'])); ?></li>
                        <li><strong><?php echo esc_html__('Date:', 'send-your-prayers'); ?></strong> <?php echo esc_html(gmdate('F j, Y, g:i a', $payment_data['timestamp'])); ?></li>
                    </ul>
                </div>
                
                <div class="syprayer-next-steps">
                    <h3><?php echo esc_html__('What Happens Next?', 'send-your-prayers'); ?></h3>
                    <ul>
                        <li><?php echo esc_html__('Your prayer will be reviewed by our team', 'send-your-prayers'); ?></li>
                        <li><?php echo esc_html__('You will receive a confirmation email shortly', 'send-your-prayers'); ?></li>
                        <li><?php echo esc_html__('Our community will include your prayer in our services', 'send-your-prayers'); ?></li>
                    </ul>
                </div>
                
                <div class="syprayer-actions">
                    <a href="<?php echo esc_url($form_page_url); ?>" class="syprayer-button"><?php echo esc_html__('Submit Another Prayer', 'send-your-prayers'); ?></a>
                </div>
            </div>
        </div>
        <?php
        
        return ob_get_clean();
    }
    
    private function render_no_payment_found() {
        // Clear any stuck cookies
        $this->clear_payment_cookie();
        
        ob_start();
        ?>
        <div class="syprayer-thank-you-page">
            <div class="syprayer-thank-you-header">
                <h1><?php echo esc_html__('Payment Status', 'send-your-prayers'); ?></h1>
            </div>
            
            <div class="syprayer-thank-you-content">
                <p><?php echo esc_html__('No active payment session found.', 'send-your-prayers'); ?></p>
                
                <div class="syprayer-next-steps">
                    <h3><?php echo esc_html__('Possible reasons:', 'send-your-prayers'); ?></h3>
                    <ul>
                        <li><?php echo esc_html__('Your payment session has expired', 'send-your-prayers'); ?></li>
                        <li><?php echo esc_html__('You have already completed your payment', 'send-your-prayers'); ?></li>
                        <li><?php echo esc_html__('You accessed this page directly without completing a payment', 'send-your-prayers'); ?></li>
                    </ul>
                </div>
                
                <div class="syprayer-actions">
                    <?php
                    $settings = get_option('syprayer_settings');
                    $form_page_id = isset($settings['form_page_id']) ? $settings['form_page_id'] : 0;
                    $form_page_url = $form_page_id ? get_permalink($form_page_id) : home_url('/');
                    ?>
                    <a href="<?php echo esc_url($form_page_url); ?>" class="syprayer-button"><?php echo esc_html__('Submit a Prayer', 'send-your-prayers'); ?></a>
                    <a href="<?php echo esc_url(home_url('/')); ?>" class="syprayer-button syprayer-button-secondary"><?php echo esc_html__('Return to Home', 'send-your-prayers'); ?></a>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}
?>