<?php
if (!defined('ABSPATH')) {
    exit;
}

class SYPRAYER_PayPal {
    
    private $client_id;
    private $client_secret;
    private $test_mode;
    private $is_configured = false;
    
    public function __construct() {
        add_action('wp_ajax_syprayer_process_paypal', array($this, 'process_payment'));
        add_action('wp_ajax_nopriv_syprayer_process_paypal', array($this, 'process_payment'));
        
        $this->init_paypal();
    }
    
    private function init_paypal() {
        $settings = get_option('syprayer_settings', array());
        
        // Check if PayPal is configured
        if (empty($settings['paypal_client_id']) || empty($settings['paypal_client_secret'])) {
            $this->is_configured = false;
            return;
        }
        
        $this->test_mode = (!empty($settings['paypal_test_mode']) && $settings['paypal_test_mode'] === 'yes');
        $this->client_id = isset($settings['paypal_client_id']) ? $settings['paypal_client_id'] : '';
        $this->client_secret = isset($settings['paypal_client_secret']) ? $settings['paypal_client_secret'] : '';
        $this->is_configured = !empty($this->client_id) && !empty($this->client_secret);
    }
    
    public static function is_available() {
        $settings = get_option('syprayer_settings', array());
        return !empty($settings['paypal_client_id']) && !empty($settings['paypal_client_secret']);
    }
    
    public function get_access_token() { // Changed from private to public
        if (!$this->is_configured) {
            throw new Exception('PayPal is not configured. Please check your settings.');
        }
        
        $url = $this->test_mode ? 
            'https://api.sandbox.paypal.com/v1/oauth2/token' : 
            'https://api.paypal.com/v1/oauth2/token';
        
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Accept' => 'application/json',
                'Accept-Language' => 'en_US',
                'Authorization' => 'Basic ' . base64_encode($this->client_id . ':' . $this->client_secret),
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
            'body' => 'grant_type=client_credentials',
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('PayPal API request failed: ' . esc_html($response->get_error_message()));
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (isset($result['access_token'])) {
            return $result['access_token'];
        } else {
            throw new Exception('Failed to get PayPal access token. Please check your credentials.');
        }
    }
    
    public function capture_payment($access_token, $order_id) { // Changed from private to public
        $url = $this->get_api_base_url() . '/v2/checkout/orders/' . $order_id . '/capture';
        
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
            ),
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('Payment capture failed: ' . esc_html($response->get_error_message()));
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (isset($result['status']) && $result['status'] === 'COMPLETED') {
            return array(
                'status' => $result['status'],
                'id' => $order_id
            );
        }
        
        $error_message = isset($result['message']) ? esc_html($result['message']) : 'Unknown error';
        throw new Exception('Payment not completed: ' . esc_html($error_message));
    }
    
    public function save_prayer_submission($payment_data, $payment_id) { // Changed from private to public
        try {
            $syprayer_db = new SYPRAYER_DB();
            
            $prayer_data = array(
                'user_email' => $payment_data['email'],
                'prayer_text' => $payment_data['prayer'],
                'amount' => $payment_data['amount'],
                'currency' => $payment_data['currency'],
                'payment_intent_id' => $payment_id,
                'payment_gateway' => 'paypal',
                'payment_status' => 'succeeded'
            );
            
            $insert_id = $syprayer_db->insert_prayer($prayer_data);
            
            if (!$insert_id) {
                throw new Exception('Failed to save prayer submission to database.');
            }
            
            return $insert_id;
            
        } catch (Exception $e) {
            throw $e;
        }
    }
    
    public function process_payment() {
        // Verify nonce for AJAX request
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'syprayer_nonce')) {
            wp_send_json_error('Security check failed. Invalid nonce.');
            return;
        }
        
        // Check if PayPal is configured
        if (!$this->is_configured) {
            wp_send_json_error('PayPal is not configured. Please contact the site administrator.');
        }
        
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $prayer = isset($_POST['prayer']) ? sanitize_textarea_field(wp_unslash($_POST['prayer'])) : '';
        $amount = isset($_POST['amount']) ? floatval(wp_unslash($_POST['amount'])) : 0;
        
        // Validate inputs
        if (!is_email($email)) {
            wp_send_json_error('Please enter a valid email address.');
        }
        
        if (empty($prayer)) {
            wp_send_json_error('Please enter your prayer request.');
        }
        
        if ($amount <= 0) {
            wp_send_json_error('Please enter a valid donation amount.');
        }
        
        // Check submission limits
        $syprayer_db = new SYPRAYER_DB();
        $prayer_count = $syprayer_db->get_prayer_count($email);
        $settings = get_option('syprayer_settings');
        $limit = isset($settings['prayer_limit']) ? intval($settings['prayer_limit']) : 5;
        
        if ($prayer_count >= $limit) {
            wp_send_json_error('You have reached your monthly submission limit of ' . $limit . ' prayers.');
        }
        
        try {
            $access_token = $this->get_access_token();
            $settings = get_option('syprayer_settings');
            $currency = isset($settings['currency']) ? $settings['currency'] : 'USD';
            
            // Store payment data
            $payment_data = array(
                'email' => $email,
                'prayer' => $prayer,
                'amount' => $amount,
                'currency' => $currency
            );
            
            // Create one-time payment
            $payment_result = $this->create_one_time_payment($access_token, $amount, $currency, $payment_data);
            
            // Store payment data
            $transient_key = 'syprayer_paypal_' . $payment_result['order_id'];
            set_transient($transient_key, $payment_data, HOUR_IN_SECONDS);
            
            // Generate nonce for PayPal return URL
            $return_nonce = wp_create_nonce('syprayer_paypal_return');
            
            // Add nonce to approval URL
            $approval_url = add_query_arg(
                array('_wpnonce' => $return_nonce),
                $payment_result['approval_url']
            );
            
            wp_send_json_success(array(
                'approval_url' => $approval_url,
                'order_id' => $payment_result['order_id'],
                'type' => 'one_time'
            ));
            
        } catch (Exception $e) {
            wp_send_json_error('PayPal payment failed: ' . esc_html($e->getMessage()));
        }
    }
    
public function process_paypal_return($token) {
    try {
        // Get stored payment data
        $transient_key = 'syprayer_paypal_' . $token;
        $payment_data = get_transient($transient_key);
        
        if (!$payment_data) {
            throw new Exception('Payment session expired.');
        }
        
        $access_token = $this->get_access_token();
        $capture_result = $this->capture_payment($access_token, $token);
        
        if ($capture_result['status'] === 'COMPLETED') {
            // Save prayer submission
            $this->save_prayer_submission($payment_data, $capture_result['id']);
            
            // Clean up transient
            delete_transient($transient_key);
            
            return true;
        } else {
            throw new Exception('Payment not completed. Status: ' . $capture_result['status']);
        }
        
    } catch (Exception $e) {
        throw $e;
    }
}
    
    private function create_one_time_payment($access_token, $amount, $currency, $payment_data) {
        $url = $this->get_api_base_url() . '/v2/checkout/orders';
        
        // Generate nonce for return URL
        $return_nonce = wp_create_nonce('syprayer_paypal_return');
        
        $return_url = add_query_arg(array(
            'syprayer_success' => '1',
            'gateway' => 'paypal',
            '_wpnonce' => $return_nonce
        ), home_url('/'));
        
        $order_data = array(
            'intent' => 'CAPTURE',
            'purchase_units' => array(
                array(
                    'amount' => array(
                        'currency_code' => $currency,
                        'value' => number_format($amount, 2, '.', '')
                    ),
                    'description' => 'Prayer Submission'
                )
            ),
            'application_context' => array(
                'return_url' => $return_url,
                'cancel_url' => home_url('/'),
                'brand_name' => get_bloginfo('name'),
                'user_action' => 'PAY_NOW'
            )
        );
        
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $access_token,
                'PayPal-Request-Id' => 'syprayer-' . time(),
            ),
            'body' => json_encode($order_data),
            'timeout' => 30
        );
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('PayPal order creation failed: ' . esc_html($response->get_error_message()));
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (isset($result['id']) && isset($result['links'])) {
            foreach ($result['links'] as $link) {
                if ($link['rel'] === 'approve') {
                    return array(
                        'approval_url' => $link['href'],
                        'order_id' => $result['id']
                    );
                }
            }
        }
        
        throw new Exception('Failed to create PayPal order.');
    }
    
    private function get_api_base_url() {
        return $this->test_mode ? 
            'https://api.sandbox.paypal.com' : 
            'https://api.paypal.com';
    }
}