<?php
if (!defined('ABSPATH')) {
    exit;
}

class SYPRAYER_Payments {
    
    private $stripe_secret_key;
    
    public function __construct() {
        add_action('wp_ajax_syprayer_process_payment', array($this, 'process_payment'));
        add_action('wp_ajax_nopriv_syprayer_process_payment', array($this, 'process_payment'));
        
        $this->init_stripe();
    }
    
    private function init_stripe() {
        $settings = get_option('syprayer_settings');
        
        // Get test mode setting
        $test_mode = isset($settings['stripe_test_mode']) ? $settings['stripe_test_mode'] : 'yes';
        
        // Determine which key to use
        if ($test_mode === 'yes') {
            $this->stripe_secret_key = isset($settings['stripe_test_sk']) ? $settings['stripe_test_sk'] : '';
        } else {
            $this->stripe_secret_key = isset($settings['stripe_live_sk']) ? $settings['stripe_live_sk'] : '';
        }
        
    }
    
    private function stripe_request($endpoint, $method = 'POST', $data = array()) {
        // Check if secret key is set
        if (empty($this->stripe_secret_key)) {
            throw new Exception('Stripe secret key is not configured.');
        }
        
        $url = 'https://api.stripe.com/v1/' . $endpoint;
        
        // Convert boolean values to strings for URL encoding
        $formatted_data = array();
        foreach ($data as $key => $value) {
            if (is_bool($value)) {
                $formatted_data[$key] = $value ? 'true' : 'false';
            } elseif (is_array($value)) {
                // Handle nested arrays (for metadata, etc.)
                foreach ($value as $sub_key => $sub_value) {
                    $formatted_data[$key . '[' . $sub_key . ']'] = $sub_value;
                }
            } else {
                $formatted_data[$key] = $value;
            }
        }
        
        $args = array(
            'method' => $method,
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->stripe_secret_key,
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
            'body' => $formatted_data,
            'timeout' => 30
        );
        
        $response = wp_remote_request($url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('Stripe API request failed: ' . esc_html($response->get_error_message()));
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if ($response_code !== 200) {
            if (isset($result['error']['message'])) {
                throw new Exception(esc_html($result['error']['message']));
            } else {
                throw new Exception('Stripe API returned status code: ' . esc_html($response_code));
            }
        }
        
        return $result;
    }
    
    public function process_payment() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'syprayer_nonce')) {
            wp_send_json_error('Security check failed. Please refresh the page and try again.');
        }
        
        // Sanitize and validate inputs
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $prayer = isset($_POST['prayer']) ? sanitize_textarea_field(wp_unslash($_POST['prayer'])) : '';
        $amount = isset($_POST['amount']) ? floatval(wp_unslash($_POST['amount'])) : 0;
        $payment_gateway = isset($_POST['payment_gateway']) ? sanitize_text_field(wp_unslash($_POST['payment_gateway'])) : '';
        $payment_method_id = isset($_POST['payment_method_id']) ? sanitize_text_field(wp_unslash($_POST['payment_method_id'])) : '';
        
        // Validate inputs
        if (!is_email($email)) {
            wp_send_json_error('Please enter a valid email address.');
        }
        
        if (empty($prayer)) {
            wp_send_json_error('Please enter your prayer request.');
        }
        
        if ($amount <= 0) {
            wp_send_json_error('Please enter a valid donation amount.');
        }
        
        // Check submission limits
        $syprayer_db = new SYPRAYER_DB();
        $prayer_count = $syprayer_db->get_prayer_count($email);
        $settings = get_option('syprayer_settings');
        $limit = isset($settings['prayer_limit']) ? intval($settings['prayer_limit']) : 5;
        
        if ($prayer_count >= $limit) {
            wp_send_json_error('You have reached your monthly submission limit of ' . $limit . ' prayers.');
        }
        
        // Handle different payment gateways
        if ($payment_gateway === 'paypal') {
            wp_send_json_error('PayPal payments are handled separately.');
            return;
        }
        
        try {
            $settings = get_option('syprayer_settings');
            $currency = strtolower($settings['currency']);
            
            // Handle one-time payment
            $this->process_stripe_payment($email, $prayer, $amount, $currency, $payment_method_id);
            
        } catch (Exception $e) {
            wp_send_json_error('Payment processing failed: ' . esc_html($e->getMessage()));
        }
    }
    
    private function process_stripe_payment($email, $prayer, $amount, $currency, $payment_method_id) {
    try {
        // Generate a nonce for the success redirect
        $success_nonce = wp_create_nonce('syprayer_payment_redirect');
        
        // Create payment intent
        $payment_intent_data = array(
            'amount' => intval($amount * 100), // Convert to cents
            'currency' => strtolower($currency),
            'payment_method' => $payment_method_id,
            'confirmation_method' => 'automatic',
            'confirm' => 'true', // String instead of boolean
            'return_url' => add_query_arg(array(
                'syprayer_success' => '1',
                'gateway' => 'stripe',
                '_wpnonce' => $success_nonce
            ), home_url('/thank-you')),
            'metadata[email]' => $email,
            'metadata[type]' => 'prayer_submission'
        );
        
        $payment_intent = $this->stripe_request('payment_intents', 'POST', $payment_intent_data);
        
        // Store prayer submission
        $prayer_data = array(
            'user_email' => $email,
            'prayer_text' => $prayer,
            'amount' => $amount,
            'currency' => strtoupper($currency),
            'payment_intent_id' => $payment_intent['id'],
            'payment_gateway' => 'stripe',
            'payment_status' => $payment_intent['status']
        );
        
        $syprayer_db = new SYPRAYER_DB();
        $insert_result = $syprayer_db->insert_prayer($prayer_data);
        
        if (!$insert_result) {
            throw new Exception('Failed to save prayer submission.');
        }
        
        // Check if payment requires additional action
        if ($payment_intent['status'] === 'requires_action' || $payment_intent['status'] === 'requires_confirmation') {
            wp_send_json_success(array(
                'requires_action' => true,
                'client_secret' => $payment_intent['client_secret'],
                'payment_intent_id' => $payment_intent['id']
            ));
        } elseif ($payment_intent['status'] === 'succeeded') {
            // Payment succeeded immediately - create redirect with nonce
            $redirect_nonce = wp_create_nonce('syprayer_payment_redirect');
            $redirect_url = add_query_arg(array(
                'syprayer_success' => '1',
                'gateway' => 'stripe',
                'payment_intent' => $payment_intent['id'],
                '_wpnonce' => $redirect_nonce
            ), home_url('/thank-you'));
            
            wp_send_json_success(array(
                'redirect_url' => $redirect_url
            ));
        } else {
            throw new Exception('Payment status: ' . $payment_intent['status']);
        }
        
    } catch (Exception $e) {
        throw new Exception('Stripe payment failed: ' . esc_html($e->getMessage()));
    }
}
}
?>