<?php
if (!defined('ABSPATH')) {
    exit;
}

class SYPRAYER_Payment_Handler {
    
    public function __construct() {
        // Handle all payment returns - hook VERY early
        add_action('init', array($this, 'handle_payment_returns'), 0);
    }
    
    public function handle_payment_returns() {
        // Only process if we have payment parameters
        if (!isset($_GET['syprayer_success']) || $_GET['syprayer_success'] !== '1') {
            return;
        }
        
        // Check for gateway
        if (!isset($_GET['gateway'])) {
            return;
        }
        
        $gateway = sanitize_text_field(wp_unslash($_GET['gateway']));
        
        // Route to appropriate handler
        if ($gateway === 'paypal') {
            $this->handle_paypal_return();
        } elseif ($gateway === 'stripe') {
            $this->handle_stripe_return();
        }
    }
    
    private function handle_paypal_return() {
    // Check for required PayPal parameters
    if (!isset($_GET['_wpnonce']) || !isset($_GET['token'])) {
        return;
    }
    
    // Verify nonce for PayPal return
    if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'syprayer_paypal_return')) {
        wp_die('Security check failed. Invalid PayPal return.');
    }
    
    $token = sanitize_text_field(wp_unslash($_GET['token']));
    
    // Get PayPal instance
    $paypal = new SYPRAYER_PayPal();
    
    try {
        // Process PayPal return using the new public method
        if ($paypal->process_paypal_return($token)) {
            // Store success in cookie for thank you page
            $this->store_payment_success('paypal');
            
            // Redirect to thank you page
            $this->redirect_to_thankyou('paypal');
        }
        
    } catch (Exception $e) {
        wp_die('PayPal payment error: ' . esc_html($e->getMessage()));
    }
}
    
    private function handle_stripe_return() {
        // For Stripe, we just need to store success and redirect
        if (!isset($_GET['_wpnonce'])) {
            return;
        }
        
        // Verify nonce for Stripe return
        if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'syprayer_payment_redirect')) {
            wp_die('Security check failed. Invalid Stripe return.');
        }
        
        // Store success in cookie for thank you page
        $this->store_payment_success('stripe');
        
        // Redirect to thank you page
        $this->redirect_to_thankyou('stripe');
    }
    
    private function store_payment_success($gateway) {
        // Generate a unique token for this session
        $token = wp_generate_password(32, false);
        
        // Store payment info in a transient with a short expiration (10 minutes)
        $payment_data = array(
            'gateway' => $gateway,
            'timestamp' => time(),
            'token' => $token
        );
        
        set_transient('syprayer_payment_' . $token, $payment_data, 600); // 10 minutes
        
        // Store token in a secure cookie
        if (!headers_sent()) {
            setcookie('syprayer_payment_token', $token, time() + 600, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true);
            $_COOKIE['syprayer_payment_token'] = $token;
        }
    }
    
    private function redirect_to_thankyou($gateway) {
        // Generate a nonce for the thank you page
        $nonce = wp_create_nonce('syprayer_thankyou_access');
        
        // Build redirect URL
        $redirect_url = add_query_arg(array(
            'gateway' => $gateway,
            '_wpnonce' => $nonce
        ), home_url('/thank-you'));
        
        // Redirect
        wp_redirect($redirect_url);
        exit;
    }
}