<?php
if (!defined('ABSPATH')) {
    exit;
}

class SYPRAYER_DB {
    
    public static function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Prayer submissions table
        $table_name = $wpdb->prefix . 'syprayer_submissions';
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            user_email varchar(100) NOT NULL,
            prayer_text text NOT NULL,
            prayer_excerpt varchar(500) DEFAULT '',
            amount decimal(10,2) NOT NULL,
            currency varchar(3) NOT NULL,
            payment_status varchar(20) DEFAULT 'pending',
            payment_intent_id varchar(255),
            payment_gateway varchar(20) DEFAULT 'stripe',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY user_email (user_email),
            KEY payment_status (payment_status),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public function get_prayer_count($email) {
        global $wpdb;
        
        $current_month = gmdate('Y-m-01');
        
        // Get cache version
        $cache_version = get_option('syprayer_cache_version', 1);
        $cache_key = 'syprayer_prayer_count_v' . $cache_version . '_' . md5($email . '_' . $current_month);
        
        $count = wp_cache_get($cache_key, 'send-your-prayers');
        
        if (false === $count) {
            // Direct database query for custom table - requires ignore comment
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $count = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->prefix}syprayer_submissions WHERE user_email = %s AND created_at >= %s",
                    $email,
                    $current_month
                )
            );
            
            // Cache for 15 minutes
            wp_cache_set($cache_key, $count, 'send-your-prayers', 900);
        }
        
        return $count ? intval($count) : 0;
    }
    
    public function insert_prayer($data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'syprayer_submissions';
        
        // Create excerpt from prayer text (first 150 characters)
        if (isset($data['prayer_text'])) {
            $prayer_text = $data['prayer_text'];
            $excerpt = wp_trim_words($prayer_text, 25, '...');
            $data['prayer_excerpt'] = $excerpt;
        }
        
        // Direct database query for custom table - requires ignore comment
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $result = $wpdb->insert($table_name, $data);
        
        if (false === $result) {
            // Return false without logging
            return false;
        }
        
        $insert_id = $wpdb->insert_id;
        
        // Clear relevant caches when new prayer is inserted
        $this->clear_caches_on_insert($data, $insert_id);
        
        return $insert_id;
    }
    
    private function clear_caches_on_insert($data, $insert_id) {
        // Increment cache version to invalidate all caches
        $current_version = get_option('syprayer_cache_version', 1);
        update_option('syprayer_cache_version', $current_version + 1, false);
        
        // Clear specific caches that should be immediate
        $currency = isset($data['currency']) ? $data['currency'] : 'USD';
        wp_cache_delete('syprayer_total_submissions', 'send-your-prayers');
        
        // Clear any cached single prayer view
        wp_cache_delete('syprayer_prayer_' . $insert_id, 'send-your-prayers');
    }
    
    private function clear_submissions_caches() {
        // Increment cache version to invalidate all caches
        $current_version = get_option('syprayer_cache_version', 1);
        update_option('syprayer_cache_version', $current_version + 1, false);
    }
    
    public function get_submissions($filters = array()) {
        global $wpdb;
        
        // Get current cache version
        $cache_version = get_option('syprayer_cache_version', 1);
        
        // Use versioned cache key
        $cache_key = 'syprayer_submissions_v' . $cache_version . '_' . md5(serialize($filters));
        $results = wp_cache_get($cache_key, 'send-your-prayers');
        
        if (false === $results) {
            $where = array('1=1');
            $params = array();
            
            // Build WHERE clause and parameters
            if (!empty($filters['payment_status'])) {
                $where[] = 'payment_status = %s';
                $params[] = sanitize_text_field($filters['payment_status']);
            }
            
            if (!empty($filters['date_from'])) {
                $where[] = 'created_at >= %s';
                $params[] = sanitize_text_field($filters['date_from']);
            }
            
            if (!empty($filters['date_to'])) {
                $where[] = 'created_at <= %s';
                $params[] = sanitize_text_field($filters['date_to'] . ' 23:59:59');
            }
            
            if (!empty($filters['user_email'])) {
                $where[] = 'user_email = %s';
                $params[] = sanitize_email($filters['user_email']);
            }
            
            $where_clause = implode(' AND ', $where);
            
            if (!empty($params)) {
                // Direct database query for custom table
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $results = $wpdb->get_results(
                    $wpdb->prepare(
                        "SELECT * FROM {$wpdb->prefix}syprayer_submissions WHERE {$where_clause} ORDER BY created_at DESC",
                        $params
                    )
                );
            } else {
                // No parameters
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $results = $wpdb->get_results(
                    "SELECT * FROM {$wpdb->prefix}syprayer_submissions WHERE {$where_clause} ORDER BY created_at DESC"
                );
            }
            
            // Cache for 5 minutes
            wp_cache_set($cache_key, $results, 'send-your-prayers', 300);
        }
        
        return $results;
    }
    
    public function get_total_submissions() {
        global $wpdb;
        
        // Get cache version
        $cache_version = get_option('syprayer_cache_version', 1);
        $cache_key = 'syprayer_total_submissions_v' . $cache_version;
        
        $total = wp_cache_get($cache_key, 'send-your-prayers');
        
        if (false === $total) {
            // Direct database query for custom table - requires ignore comment
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $total = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$wpdb->prefix}syprayer_submissions"
            );
            
            // Cache for 5 minutes
            wp_cache_set($cache_key, $total, 'send-your-prayers', 300);
        }
        
        return $total;
    }
    
    public function get_total_revenue() {
        global $wpdb;
        
        $settings = get_option('syprayer_settings');
        $currency = isset($settings['currency']) ? $settings['currency'] : 'USD';
        
        // Get cache version
        $cache_version = get_option('syprayer_cache_version', 1);
        $cache_key = 'syprayer_total_revenue_v' . $cache_version . '_' . $currency;
        
        $revenue = wp_cache_get($cache_key, 'send-your-prayers');
        
        if (false === $revenue) {
            // Direct database query for custom table - requires ignore comment
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $revenue = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT SUM(amount) FROM {$wpdb->prefix}syprayer_submissions WHERE payment_status = %s",
                    'succeeded'
                )
            );
            
            // Cache for 5 minutes
            wp_cache_set($cache_key, $revenue, 'send-your-prayers', 300);
        }
        
        return $currency . number_format($revenue ?: 0, 2);
    }
}
?>