<?php
if (!defined('ABSPATH')) {
    exit;
}

class SYPRAYER_Admin {
    
    private $settings_tabs = array();
    
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));
        
        // Initialize tabs
        $this->settings_tabs = array(
            'general' => 'General Settings',
            'pages' => 'Page Mapping',
            'stripe' => 'Stripe Settings',
            'paypal' => 'PayPal Settings'
        );
    }
    
    public function add_admin_menu() {
        // Main menu
        add_menu_page(
            'Send Your Prayers',
            'Prayers',
            'manage_options',
            'send-your-prayers',
            array($this, 'dashboard_page'),
            $this->get_prayer_icon_svg(),
            30
        );
        
        // Submenus
        add_submenu_page(
            'send-your-prayers',
            'Dashboard',
            'Dashboard',
            'manage_options',
            'send-your-prayers',
            array($this, 'dashboard_page')
        );
        
        add_submenu_page(
            'send-your-prayers',
            'Prayer Submissions',
            'Prayer Submissions',
            'manage_options',
            'syprayer-submissions',
            array($this, 'submissions_page')
        );
        
        add_submenu_page(
            'send-your-prayers',
            'Settings',
            'Settings',
            'manage_options',
            'syprayer-settings',
            array($this, 'settings_page')
        );
    }
    
    private function get_prayer_icon_svg() {
        // Prayer hands icon - white color for admin menu
        $svg = '<?xml version="1.0" encoding="utf-8"?>
<svg width="40" height="40" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
<path d="M8.96173 18.9109L9.42605 18.3219L8.96173 18.9109ZM12 5.50063L11.4596 6.02073C11.601 6.16763 11.7961 6.25063 12 6.25063C12.2039 6.25063 12.399 6.16763 12.5404 6.02073L12 5.50063ZM15.0383 18.9109L15.5026 19.4999L15.0383 18.9109ZM7.00061 16.4209C6.68078 16.1577 6.20813 16.2036 5.94491 16.5234C5.68169 16.8432 5.72758 17.3159 6.04741 17.5791L7.00061 16.4209ZM2.34199 13.4115C2.54074 13.7749 2.99647 13.9084 3.35988 13.7096C3.7233 13.5108 3.85677 13.0551 3.65801 12.6917L2.34199 13.4115ZM2.75 9.1371C2.75 6.98623 3.96537 5.18252 5.62436 4.42419C7.23607 3.68748 9.40166 3.88258 11.4596 6.02073L12.5404 4.98053C10.0985 2.44352 7.26409 2.02539 5.00076 3.05996C2.78471 4.07292 1.25 6.42503 1.25 9.1371H2.75ZM8.49742 19.4999C9.00965 19.9037 9.55954 20.3343 10.1168 20.6599C10.6739 20.9854 11.3096 21.25 12 21.25V19.75C11.6904 19.75 11.3261 19.6293 10.8736 19.3648C10.4213 19.1005 9.95208 18.7366 9.42605 18.3219L8.49742 19.4999ZM15.5026 19.4999C16.9292 18.3752 18.7528 17.0866 20.1833 15.4758C21.6395 13.8361 22.75 11.8026 22.75 9.1371H21.25C21.25 11.3345 20.3508 13.0282 19.0617 14.4798C17.7469 15.9603 16.0896 17.1271 14.574 18.3219L15.5026 19.4999ZM22.75 9.1371C22.75 6.42503 21.2153 4.07292 18.9992 3.05996C16.7359 2.02539 13.9015 2.44352 11.4596 4.98053L12.5404 6.02073C14.5983 3.88258 16.7639 3.68748 18.3756 4.42419C20.0346 5.18252 21.25 6.98623 21.25 9.1371H22.75ZM14.574 18.3219C14.0479 18.7366 13.5787 19.1005 13.1264 19.3648C12.6739 19.6293 12.3096 19.75 12 19.75V21.25C12.6904 21.25 13.3261 20.9854 13.8832 20.6599C14.4405 20.3343 14.9903 19.9037 15.5026 19.4999L14.574 18.3219ZM9.42605 18.3219C8.63014 17.6945 7.82129 17.0963 7.00061 16.4209L6.04741 17.5791C6.87768 18.2624 7.75472 18.9144 8.49742 19.4999L9.42605 18.3219ZM3.65801 12.6917C3.0968 11.6656 2.75 10.5033 2.75 9.1371H1.25C1.25 10.7746 1.66995 12.1827 2.34199 13.4115L3.65801 12.6917Z" fill="white"/>
</svg>';
        
        // Return as data URI
        return 'data:image/svg+xml;base64,' . base64_encode($svg);
    }
    
    public function admin_scripts($hook) {
        // Load admin CSS on all SYPRAYER pages
        if (strpos($hook, 'send-your-prayers') !== false || strpos($hook, 'syprayer-') !== false) {
            // Get plugin version
            $version = function_exists('syprayer_get_version') ? syprayer_get_version() : '1.6.0';
            
            // Register and enqueue admin CSS
            wp_register_style('syprayer-admin-style', SYPRAYER_PLUGIN_URL . 'assets/css/admin.css', array(), $version);
            wp_enqueue_style('syprayer-admin-style');
            
            // Load admin JS only on settings page
            if (strpos($hook, 'syprayer-settings') !== false) {
                wp_register_script('syprayer-admin-script', SYPRAYER_PLUGIN_URL . 'assets/js/admin.js', array('jquery', 'wp-color-picker'), $version, true);
                wp_enqueue_script('syprayer-admin-script');
                wp_enqueue_style('wp-color-picker');
            }
            
            // Register and enqueue common admin JS for all SYPRAYER pages
            wp_register_script('syprayer-admin-common', '', array('jquery'), $version, true);
            wp_enqueue_script('syprayer-admin-common');
            
            // Add inline script for admin functions
            $admin_js = '
                function copyToClipboard(text) {
                    const textArea = document.createElement("textarea");
                    textArea.value = text;
                    document.body.appendChild(textArea);
                    textArea.select();
                    document.execCommand("copy");
                    document.body.removeChild(textArea);
                    alert("Prayer text copied to clipboard!");
                }
                
                // Add admin-specific helper functions
                if (typeof syprayerAdminHelpers === "undefined") {
                    var syprayerAdminHelpers = {
                        formatCurrency: function(amount, currency) {
                            return currency + parseFloat(amount).toFixed(2);
                        },
                        formatDate: function(dateString) {
                            const date = new Date(dateString);
                            return date.toLocaleDateString() + " " + date.toLocaleTimeString();
                        }
                    };
                }
            ';
            
            wp_add_inline_script('syprayer-admin-common', $admin_js);
            
            // Localize script for AJAX
            wp_localize_script('syprayer-admin-common', 'syprayer_admin', array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('syprayer_admin_nonce'),
                'text' => array(
                    'copy_success' => 'Copied to clipboard!',
                    'copy_error' => 'Failed to copy',
                    'confirm_delete' => 'Are you sure you want to delete this? This action cannot be undone.'
                )
            ));
        }
    }
    
    public function dashboard_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }
        
        // Add nonce verification for admin pages
        if (isset($_POST['action'])) {
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'syprayer_admin_action')) {
                wp_die('Security check failed.');
            }
        }
        
        $syprayer_db = new SYPRAYER_DB();
        
        // Get stats
        $total_submissions = $syprayer_db->get_total_submissions();
        $revenue = $syprayer_db->get_total_revenue();
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Send Your Prayers Dashboard', 'send-your-prayers'); ?></h1>
            
            <div class="syprayer-dashboard-stats">
                <div class="syprayer-stat-card">
                    <h3><?php echo esc_html__('Total Submissions', 'send-your-prayers'); ?></h3>
                    <div class="syprayer-stat-number"><?php echo esc_html($total_submissions); ?></div>
                </div>
                
                <div class="syprayer-stat-card">
                    <h3><?php echo esc_html__('Total Revenue', 'send-your-prayers'); ?></h3>
                    <div class="syprayer-stat-number"><?php echo esc_html($revenue); ?></div>
                </div>
            </div>
            
            <div class="syprayer-recent-submissions">
                <h2><?php echo esc_html__('Recent Prayer Submissions', 'send-your-prayers'); ?></h2>
                <?php $this->display_recent_submissions(); ?>
            </div>
        </div>
        <?php
    }
    
    public function submissions_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }
        
        // Check if we're viewing a single prayer with proper nonce verification
        if (isset($_GET['view_prayer']) && is_numeric($_GET['view_prayer'])) {
            // Verify nonce for single prayer view
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'syprayer_view_prayer')) {
                wp_die('Security check failed. Please use the proper navigation.');
            }
            
            $this->view_single_prayer(intval($_GET['view_prayer']));
            return;
        }
        
        $syprayer_db = new SYPRAYER_DB();
        $filters = array();
        
        // Handle filter submission with nonce verification
        if (isset($_GET['syprayer_filter_nonce'])) {
            if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['syprayer_filter_nonce'])), 'syprayer_filter_submissions')) {
                wp_die('Security check failed. Invalid filter request.');
            }
        }
        
        // Sanitize and validate GET parameters
        if (isset($_GET['date_from'])) {
            $filters['date_from'] = sanitize_text_field(wp_unslash($_GET['date_from']));
        }
        
        if (isset($_GET['date_to'])) {
            $filters['date_to'] = sanitize_text_field(wp_unslash($_GET['date_to']));
        }
        
        if (isset($_GET['user_email'])) {
            $filters['user_email'] = sanitize_email(wp_unslash($_GET['user_email']));
        }
        
        $submissions = $syprayer_db->get_submissions($filters);
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Prayer Submissions', 'send-your-prayers'); ?></h1>
            
            <div class="syprayer-filters">
                <form method="get">
                    <?php wp_nonce_field('syprayer_filter_submissions', 'syprayer_filter_nonce'); ?>
                    <input type="hidden" name="page" value="syprayer-submissions">
                    
                    <input type="email" name="user_email" value="<?php echo isset($_GET['user_email']) ? esc_attr(sanitize_email(wp_unslash($_GET['user_email']))) : ''; ?>" placeholder="<?php echo esc_attr__('Filter by email', 'send-your-prayers'); ?>">
                    
                    <input type="date" name="date_from" value="<?php echo isset($_GET['date_from']) ? esc_attr(sanitize_text_field(wp_unslash($_GET['date_from']))) : ''; ?>" placeholder="<?php echo esc_attr__('From Date', 'send-your-prayers'); ?>">
                    <input type="date" name="date_to" value="<?php echo isset($_GET['date_to']) ? esc_attr(sanitize_text_field(wp_unslash($_GET['date_to']))) : ''; ?>" placeholder="<?php echo esc_attr__('To Date', 'send-your-prayers'); ?>">
                    
                    <button type="submit" class="button"><?php echo esc_html__('Filter', 'send-your-prayers'); ?></button>
                    <a href="?page=syprayer-submissions" class="button"><?php echo esc_html__('Clear', 'send-your-prayers'); ?></a>
                </form>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('ID', 'send-your-prayers'); ?></th>
                        <th><?php echo esc_html__('Email', 'send-your-prayers'); ?></th>
                        <th><?php echo esc_html__('Prayer', 'send-your-prayers'); ?></th>
                        <th><?php echo esc_html__('Amount', 'send-your-prayers'); ?></th>
                        <th><?php echo esc_html__('Status', 'send-your-prayers'); ?></th>
                        <th><?php echo esc_html__('Date', 'send-your-prayers'); ?></th>
                        <th><?php echo esc_html__('Actions', 'send-your-prayers'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($submissions): ?>
                        <?php foreach ($submissions as $submission): ?>
                        <tr>
                            <td><?php echo esc_html($submission->id); ?></td>
                            <td><?php echo esc_html($submission->user_email); ?></td>
                            <td>
                                <div class="syprayer-prayer-excerpt">
                                    <?php 
                                    $excerpt = !empty($submission->prayer_excerpt) ? 
                                        $submission->prayer_excerpt : 
                                        wp_trim_words($submission->prayer_text, 15, '...');
                                    echo esc_html($excerpt); 
                                    ?>
                                </div>
                            </td>
                            <td><?php echo esc_html($submission->currency . number_format($submission->amount, 2)); ?></td>
                            <td>
                                <span class="syprayer-status syprayer-status-<?php echo esc_attr($submission->payment_status); ?>">
                                    <?php echo esc_html(ucfirst($submission->payment_status)); ?>
                                </span>
                            </td>
                            <td><?php echo esc_html(gmdate('M j, Y g:i A', strtotime($submission->created_at))); ?></td>
                            <td>
                                <?php 
                                $view_nonce = wp_create_nonce('syprayer_view_prayer');
                                $view_url = add_query_arg(array(
                                    'page' => 'syprayer-submissions',
                                    'view_prayer' => $submission->id,
                                    '_wpnonce' => $view_nonce
                                ), admin_url('admin.php'));
                                ?>
                                <a href="<?php echo esc_url($view_url); ?>" 
                                   class="button button-small"><?php echo esc_html__('View', 'send-your-prayers'); ?></a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7"><?php echo esc_html__('No submissions found.', 'send-your-prayers'); ?></td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }
    
    private function view_single_prayer($prayer_id) {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }
        
        global $wpdb;
        
        // Use caching for single prayer query
        $cache_key = 'syprayer_prayer_' . $prayer_id;
        $prayer = wp_cache_get($cache_key, 'send-your-prayers');
        
        if (false === $prayer) {
            // Direct database query for custom table - requires ignore comment
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $prayer = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM {$wpdb->prefix}syprayer_submissions WHERE id = %d",
                    $prayer_id
                )
            );
            
            // Cache the result for 1 hour
            if ($prayer) {
                wp_cache_set($cache_key, $prayer, 'send-your-prayers', 3600);
            }
        }
        
        if (!$prayer) {
            wp_die('Prayer not found.');
        }

        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Prayer Submission Details', 'send-your-prayers'); ?></h1>
            
            <div class="syprayer-prayer-details">
                <div class="syprayer-back-link">
                    <?php 
                    $back_nonce = wp_create_nonce('syprayer_view_prayer_back');
                    $back_url = add_query_arg(array(
                        'page' => 'syprayer-submissions',
                        '_wpnonce' => $back_nonce
                    ), admin_url('admin.php'));
                    ?>
                    <a href="<?php echo esc_url($back_url); ?>" class="button">&larr; <?php echo esc_html__('Back to Submissions', 'send-your-prayers'); ?></a>
                </div>
                
                <div class="syprayer-prayer-detail-card">
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('ID:', 'send-your-prayers'); ?></strong> <?php echo esc_html($prayer->id); ?>
                    </div>
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('Email:', 'send-your-prayers'); ?></strong> <?php echo esc_html($prayer->user_email); ?>
                    </div>
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('Amount:', 'send-your-prayers'); ?></strong> <?php echo esc_html($prayer->currency . number_format($prayer->amount, 2)); ?>
                    </div>
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('Status:', 'send-your-prayers'); ?></strong> 
                        <span class="syprayer-status syprayer-status-<?php echo esc_attr($prayer->payment_status); ?>">
                            <?php echo esc_html(ucfirst($prayer->payment_status)); ?>
                        </span>
                    </div>
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('Date:', 'send-your-prayers'); ?></strong> <?php echo esc_html(gmdate('M j, Y g:i A', strtotime($prayer->created_at))); ?>
                    </div>
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('Payment Gateway:', 'send-your-prayers'); ?></strong> <?php echo esc_html(ucfirst($prayer->payment_gateway)); ?>
                    </div>
                    <?php if ($prayer->payment_intent_id): ?>
                    <div class="syprayer-detail-row">
                        <strong><?php echo esc_html__('Payment ID:', 'send-your-prayers'); ?></strong> <?php echo esc_html($prayer->payment_intent_id); ?>
                    </div>
                    <?php endif; ?>
                    
                    <div class="syprayer-prayer-content">
                        <h3><?php echo esc_html__('Prayer Request', 'send-your-prayers'); ?></h3>
                        <div class="syprayer-prayer-text">
                            <?php echo wp_kses_post(nl2br(esc_html($prayer->prayer_text))); ?>
                        </div>
                        <div class="syprayer-prayer-stats">
                            <small>
                                <?php 
                                printf(
                                    /* translators: 1: Character count, 2: Word count */
                                    esc_html__('Characters: %1$s | Words: %2$s', 'send-your-prayers'),
                                    esc_html(strlen($prayer->prayer_text)),
                                    esc_html(str_word_count($prayer->prayer_text))
                                );
                                ?>
                            </small>
                        </div>
                    </div>
                    
                    <div class="syprayer-prayer-actions">
                        <button onclick="window.print()" class="button"><?php echo esc_html__('Print', 'send-your-prayers'); ?></button>
                        <button onclick="copyToClipboard('<?php echo esc_js($prayer->prayer_text); ?>')" class="button"><?php echo esc_html__('Copy Text', 'send-your-prayers'); ?></button>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    public function settings_page() {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            wp_die('You do not have sufficient permissions to access this page.');
        }
        
        // Add nonce verification for admin pages
        if (isset($_POST['action'])) {
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'syprayer_admin_action')) {
                wp_die('Security check failed.');
            }
        }
        
        $current_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'general';
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Send Your Prayers Settings', 'send-your-prayers'); ?></h1>
            
            <h2 class="nav-tab-wrapper">
                <?php foreach ($this->settings_tabs as $tab_key => $tab_label): ?>
                    <a href="?page=syprayer-settings&tab=<?php echo esc_attr($tab_key); ?>" 
                       class="nav-tab <?php echo $current_tab === $tab_key ? 'nav-tab-active' : ''; ?>">
                        <?php echo esc_html($tab_label); ?>
                    </a>
                <?php endforeach; ?>
            </h2>
            
            <form method="post" action="options.php">
                <?php
                // Output the appropriate settings based on current tab
                switch ($current_tab) {
                    case 'general':
                        settings_fields('syprayer_settings_general');
                        do_settings_sections('syprayer_settings_general');
                        break;
                    case 'pages':
                        settings_fields('syprayer_settings_pages');
                        do_settings_sections('syprayer_settings_pages');
                        break;
                    case 'stripe':
                        settings_fields('syprayer_settings_stripe');
                        do_settings_sections('syprayer_settings_stripe');
                        break;
                    case 'paypal':
                        settings_fields('syprayer_settings_paypal');
                        do_settings_sections('syprayer_settings_paypal');
                        break;
                }
                
                // Add nonce field for settings
                wp_nonce_field('syprayer_save_settings', 'syprayer_settings_nonce');
                
                // Output save submit button
                submit_button(__('Save Settings', 'send-your-prayers'));
                ?>
            </form>
        </div>
        <?php
    }
    
    public function settings_init() {
        // General Settings
        register_setting('syprayer_settings_general', 'syprayer_settings', array($this, 'sanitize_settings'));
        $this->init_general_settings();
        
        // Pages Settings
        register_setting('syprayer_settings_pages', 'syprayer_settings', array($this, 'sanitize_settings'));
        $this->init_pages_settings();
        
        // Stripe Settings
        register_setting('syprayer_settings_stripe', 'syprayer_settings', array($this, 'sanitize_settings'));
        $this->init_stripe_settings();
        
        // PayPal Settings
        register_setting('syprayer_settings_paypal', 'syprayer_settings', array($this, 'sanitize_settings'));
        $this->init_paypal_settings();
    }
    
    private function init_general_settings() {
        add_settings_section(
            'syprayer_general_section',
            __('General Settings', 'send-your-prayers'),
            array($this, 'general_section_callback'),
            'syprayer_settings_general'
        );
        
        $general_fields = array(
            'currency' => array(
                'label' => __('Currency', 'send-your-prayers'),
                'callback' => 'currency_render',
                'description' => ''
            ),
            'simple_amount' => array(
                'label' => __('Prayer Submission Amount', 'send-your-prayers'),
                'callback' => 'simple_amount_render',
                'description' => __('Fixed amount for prayer submissions', 'send-your-prayers')
            ),
            'prayer_limit' => array(
                'label' => __('Monthly Submission Limit', 'send-your-prayers'),
                'callback' => 'prayer_limit_render',
                'description' => __('Maximum number of submissions per email per month', 'send-your-prayers')
            ),
            'form_description' => array(
                'label' => __('Form Description', 'send-your-prayers'),
                'callback' => 'form_description_render',
                'description' => __('This text will appear below the form title on the prayer submission form', 'send-your-prayers')
            ),
            'button_color' => array(
                'label' => __('Submit Button Color', 'send-your-prayers'),
                'callback' => 'button_color_render',
                'description' => __('Choose the color for the submit button', 'send-your-prayers')
            ),
            'button_hover_color' => array(
                'label' => __('Submit Button Hover Color', 'send-your-prayers'),
                'callback' => 'button_hover_color_render',
                'description' => __('Choose the hover color for the submit button', 'send-your-prayers')
            )
        );
        
        foreach ($general_fields as $field => $config) {
            add_settings_field(
                $field,
                $config['label'],
                array($this, $config['callback']),
                'syprayer_settings_general',
                'syprayer_general_section'
            );
        }
    }
    
    private function init_pages_settings() {
        add_settings_section(
            'syprayer_pages_section',
            __('Page Mapping Settings', 'send-your-prayers'),
            array($this, 'pages_section_callback'),
            'syprayer_settings_pages'
        );
        
        $pages_fields = array(
            'form_page_id' => array(
                'label' => __('Prayer Form Page', 'send-your-prayers'),
                'callback' => 'form_page_render',
                'description' => __('Select the page where the prayer form is displayed', 'send-your-prayers')
            ),
            'thank_you_page_id' => array(
                'label' => __('Thank You Page', 'send-your-prayers'),
                'callback' => 'thank_you_page_render',
                'description' => __('Select the page for thank you message after submission', 'send-your-prayers')
            )
        );
        
        foreach ($pages_fields as $field => $config) {
            add_settings_field(
                $field,
                $config['label'],
                array($this, $config['callback']),
                'syprayer_settings_pages',
                'syprayer_pages_section'
            );
        }
    }
    
    private function init_stripe_settings() {
        add_settings_section(
            'syprayer_stripe_section',
            __('Stripe Payment Settings', 'send-your-prayers'),
            array($this, 'stripe_section_callback'),
            'syprayer_settings_stripe'
        );
        
        $stripe_fields = array(
            'stripe_test_mode' => array(
                'label' => __('Test Mode', 'send-your-prayers'),
                'callback' => 'stripe_test_mode_render',
                'description' => __('Use test keys for development and testing', 'send-your-prayers')
            ),
            'stripe_test_pk' => array(
                'label' => __('Test Publishable Key', 'send-your-prayers'),
                'callback' => 'stripe_test_pk_render',
                'description' => __('Your Stripe test publishable key (starts with pk_test_)', 'send-your-prayers')
            ),
            'stripe_test_sk' => array(
                'label' => __('Test Secret Key', 'send-your-prayers'),
                'callback' => 'stripe_test_sk_render',
                'description' => __('Your Stripe test secret key (starts with sk_test_)', 'send-your-prayers')
            ),
            'stripe_live_pk' => array(
                'label' => __('Live Publishable Key', 'send-your-prayers'),
                'callback' => 'stripe_live_pk_render',
                'description' => __('Your Stripe live publishable key (starts with pk_live_)', 'send-your-prayers')
            ),
            'stripe_live_sk' => array(
                'label' => __('Live Secret Key', 'send-your-prayers'),
                'callback' => 'stripe_live_sk_render',
                'description' => __('Your Stripe live secret key (starts with sk_live_)', 'send-your-prayers')
            )
        );
        
        foreach ($stripe_fields as $field => $config) {
            add_settings_field(
                $field,
                $config['label'],
                array($this, $config['callback']),
                'syprayer_settings_stripe',
                'syprayer_stripe_section'
            );
        }
    }
    
    private function init_paypal_settings() {
        add_settings_section(
            'syprayer_paypal_section',
            __('PayPal Payment Settings', 'send-your-prayers'),
            array($this, 'paypal_section_callback'),
            'syprayer_settings_paypal'
        );
        
        $paypal_fields = array(
            'paypal_test_mode' => array(
                'label' => __('Test Mode (Sandbox)', 'send-your-prayers'),
                'callback' => 'paypal_test_mode_render',
                'description' => __('Use PayPal sandbox for development and testing', 'send-your-prayers')
            ),
            'paypal_client_id' => array(
                'label' => __('Client ID', 'send-your-prayers'),
                'callback' => 'paypal_client_id_render',
                'description' => __('Your PayPal Client ID', 'send-your-prayers')
            ),
            'paypal_client_secret' => array(
                'label' => __('Client Secret', 'send-your-prayers'),
                'callback' => 'paypal_client_secret_render',
                'description' => __('Your PayPal Client Secret', 'send-your-prayers')
            )
        );
        
        foreach ($paypal_fields as $field => $config) {
            add_settings_field(
                $field,
                $config['label'],
                array($this, $config['callback']),
                'syprayer_settings_paypal',
                'syprayer_paypal_section'
            );
        }
    }
    
    public function sanitize_settings($input) {
        // Check user capabilities
        if (!current_user_can('manage_options')) {
            return get_option('syprayer_settings', array());
        }
        
        // Verify nonce
        if (!isset($_POST['syprayer_settings_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['syprayer_settings_nonce'])), 'syprayer_save_settings')) {
            return get_option('syprayer_settings', array());
        }
        
        // Get existing settings first
        $existing_settings = get_option('syprayer_settings', array());
        $sanitized = $existing_settings; // Start with existing settings
        
        // Check which tab is being saved based on the fields present
        $is_stripe_tab = isset($input['stripe_test_mode']) || isset($input['stripe_test_pk']) || 
                         isset($input['stripe_test_sk']) || isset($input['stripe_live_pk']) || 
                         isset($input['stripe_live_sk']);
        
        $is_paypal_tab = isset($input['paypal_test_mode']) || isset($input['paypal_client_id']) || 
                         isset($input['paypal_client_secret']);
        
        $is_general_tab = isset($input['currency']) || isset($input['simple_amount']) || 
                          isset($input['prayer_limit']) || isset($input['form_description']) || 
                          isset($input['button_color']) || isset($input['button_hover_color']);
        
        $is_pages_tab = isset($input['form_page_id']) || isset($input['thank_you_page_id']);
        
        // Only update fields from the current tab being saved
        
        // General Settings
        if ($is_general_tab) {
            if (isset($input['currency'])) {
                $sanitized['currency'] = sanitize_text_field(wp_unslash($input['currency']));
            }
            
            if (isset($input['simple_amount'])) {
                $sanitized['simple_amount'] = floatval(wp_unslash($input['simple_amount']));
            }
            
            if (isset($input['prayer_limit'])) {
                $sanitized['prayer_limit'] = absint(wp_unslash($input['prayer_limit']));
            }
            
            if (isset($input['form_description'])) {
                $sanitized['form_description'] = sanitize_textarea_field(wp_unslash($input['form_description']));
            }
            
            if (isset($input['button_color'])) {
                $sanitized['button_color'] = sanitize_hex_color(wp_unslash($input['button_color']));
            }
            
            if (isset($input['button_hover_color'])) {
                $sanitized['button_hover_color'] = sanitize_hex_color(wp_unslash($input['button_hover_color']));
            }
        }
        
        // Page Mapping Settings
        if ($is_pages_tab) {
            if (isset($input['form_page_id'])) {
                $sanitized['form_page_id'] = absint(wp_unslash($input['form_page_id']));
            }
            
            if (isset($input['thank_you_page_id'])) {
                $sanitized['thank_you_page_id'] = absint(wp_unslash($input['thank_you_page_id']));
            }
        }
        
        // Stripe Settings - only update if we're on Stripe tab
        if ($is_stripe_tab) {
            // Test mode checkbox
            if (isset($input['stripe_test_mode'])) {
                $sanitized['stripe_test_mode'] = (wp_unslash($input['stripe_test_mode']) === 'yes') ? 'yes' : 'no';
            } else {
                // If checkbox is not in input (unchecked), set to 'no'
                $sanitized['stripe_test_mode'] = 'no';
            }
            
            // Only update other fields if they exist in input
            if (isset($input['stripe_test_pk'])) {
                $sanitized['stripe_test_pk'] = sanitize_text_field(wp_unslash($input['stripe_test_pk']));
            }
            
            if (isset($input['stripe_test_sk'])) {
                $sanitized['stripe_test_sk'] = sanitize_text_field(wp_unslash($input['stripe_test_sk']));
            }
            
            if (isset($input['stripe_live_pk'])) {
                $sanitized['stripe_live_pk'] = sanitize_text_field(wp_unslash($input['stripe_live_pk']));
            }
            
            if (isset($input['stripe_live_sk'])) {
                $sanitized['stripe_live_sk'] = sanitize_text_field(wp_unslash($input['stripe_live_sk']));
            }
        }
        
        // PayPal Settings - only update if we're on PayPal tab
        if ($is_paypal_tab) {
            // Test mode checkbox
            if (isset($input['paypal_test_mode'])) {
                $sanitized['paypal_test_mode'] = (wp_unslash($input['paypal_test_mode']) === 'yes') ? 'yes' : 'no';
            } else {
                // If checkbox is not in input (unchecked), set to 'no'
                $sanitized['paypal_test_mode'] = 'no';
            }
            
            // Only update other fields if they exist in input
            if (isset($input['paypal_client_id'])) {
                $sanitized['paypal_client_id'] = sanitize_text_field(wp_unslash($input['paypal_client_id']));
            }
            
            if (isset($input['paypal_client_secret'])) {
                $sanitized['paypal_client_secret'] = sanitize_text_field(wp_unslash($input['paypal_client_secret']));
            }
        }
        
        return $sanitized;
    }
    
    // Section callbacks
    public function general_section_callback() {
        echo '<p>' . esc_html__('Configure the general settings for the Send Your Prayers system.', 'send-your-prayers') . '</p>';
    }
    
    public function pages_section_callback() {
        echo '<p>' . esc_html__('Map your pages for the Send Your Prayers system.', 'send-your-prayers') . '</p>';
    }
    
    public function stripe_section_callback() {
        echo '<p>';
        printf(
            /* translators: %s: Stripe Dashboard URL */
            esc_html__('Configure Stripe payment gateway settings. Get your API keys from %s.', 'send-your-prayers'),
            '<a href="https://dashboard.stripe.com" target="_blank" rel="noopener noreferrer">' . esc_html__('Stripe Dashboard', 'send-your-prayers') . '</a>'
        );
        echo '</p>';
    }
    
    public function paypal_section_callback() {
        echo '<p>';
        printf(
            /* translators: %s: PayPal Developer URL */
            esc_html__('Configure PayPal payment gateway settings. Get your API credentials from %s.', 'send-your-prayers'),
            '<a href="https://developer.paypal.com" target="_blank" rel="noopener noreferrer">' . esc_html__('PayPal Developer', 'send-your-prayers') . '</a>'
        );
        echo '</p>';
    }
    
    // Field render methods
    public function currency_render() {
        $options = get_option('syprayer_settings');
        $currencies = array(
            'USD' => 'US Dollar ($)',
            'EUR' => 'Euro (€)',
            'GBP' => 'British Pound (£)'
        );
        ?>
        <select name="syprayer_settings[currency]">
            <?php foreach ($currencies as $code => $name): ?>
            <option value="<?php echo esc_attr($code); ?>" <?php selected(isset($options['currency']) ? $options['currency'] : 'USD', $code); ?>>
                <?php echo esc_html($name); ?>
            </option>
            <?php endforeach; ?>
        </select>
        <?php
    }
    
    public function simple_amount_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['simple_amount']) ? $options['simple_amount'] : 1;
        ?>
        <input type="number" step="0.01" min="0.01" name="syprayer_settings[simple_amount]" 
               value="<?php echo esc_attr($value); ?>" />
        <p class="description"><?php echo esc_html__('Fixed amount for prayer submissions', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function prayer_limit_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['prayer_limit']) ? $options['prayer_limit'] : 5;
        ?>
        <input type="number" min="1" name="syprayer_settings[prayer_limit]" 
               value="<?php echo esc_attr($value); ?>" />
        <p class="description"><?php echo esc_html__('Maximum number of submissions per email per month', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function form_description_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['form_description']) ? $options['form_description'] : __('Share your prayer request with our community. Your prayer will be received and considered by our church.', 'send-your-prayers');
        ?>
        <textarea name="syprayer_settings[form_description]" rows="4" cols="50" class="large-text"><?php echo esc_textarea($value); ?></textarea>
        <p class="description"><?php echo esc_html__('This text will appear below the form title on the prayer submission form.', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function button_color_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['button_color']) ? $options['button_color'] : '#27ae60';
        ?>
        <input type="text" name="syprayer_settings[button_color]" value="<?php echo esc_attr($value); ?>" 
               class="color-picker" data-default-color="#27ae60" />
        <p class="description"><?php echo esc_html__('Choose the color for the submit button.', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function button_hover_color_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['button_hover_color']) ? $options['button_hover_color'] : '#219a52';
        ?>
        <input type="text" name="syprayer_settings[button_hover_color]" value="<?php echo esc_attr($value); ?>" 
               class="color-picker" data-default-color="#219a52" />
        <p class="description"><?php echo esc_html__('Choose the hover color for the submit button.', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function form_page_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['form_page_id']) ? $options['form_page_id'] : 0;
        $this->render_page_select('form_page_id', $value);
    }
    
    public function thank_you_page_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['thank_you_page_id']) ? $options['thank_you_page_id'] : 0;
        $this->render_page_select('thank_you_page_id', $value);
    }
    
    private function render_page_select($field_name, $selected_value) {
        $pages = get_pages(array(
            'post_status' => 'publish'
        ));
        ?>
        <select name="syprayer_settings[<?php echo esc_attr($field_name); ?>]">
            <option value="0"><?php echo esc_html__('-- Select a Page --', 'send-your-prayers'); ?></option>
            <?php foreach ($pages as $page): ?>
                <option value="<?php echo esc_attr($page->ID); ?>" <?php selected($selected_value, $page->ID); ?>>
                    <?php echo esc_html($page->post_title); ?>
                </option>
            <?php endforeach; ?>
        </select>
        <p class="description"><?php 
            if ($selected_value > 0) {
                echo esc_html__('Current URL: ', 'send-your-prayers') . esc_url(get_permalink($selected_value));
            }
        ?></p>
        <?php
    }
    
    public function stripe_test_mode_render() {
        $options = get_option('syprayer_settings');
        $checked = (isset($options['stripe_test_mode']) && $options['stripe_test_mode'] === 'yes') ? 'yes' : 'no';
        ?>
        <label>
            <input type="checkbox" name="syprayer_settings[stripe_test_mode]" value="yes" 
                   <?php checked($checked, 'yes'); ?> />
            <?php echo esc_html__('Enable Stripe test mode', 'send-your-prayers'); ?>
        </label>
        <p class="description"><?php echo esc_html__('Use Stripe test keys for development and testing', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function stripe_test_pk_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['stripe_test_pk']) ? $options['stripe_test_pk'] : '';
        ?>
        <input type="text" name="syprayer_settings[stripe_test_pk]" 
               value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php echo esc_html__('Your Stripe test publishable key (starts with pk_test_)', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function stripe_test_sk_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['stripe_test_sk']) ? $options['stripe_test_sk'] : '';
        ?>
        <input type="password" name="syprayer_settings[stripe_test_sk]" 
               value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php echo esc_html__('Your Stripe test secret key (starts with sk_test_)', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function stripe_live_pk_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['stripe_live_pk']) ? $options['stripe_live_pk'] : '';
        ?>
        <input type="text" name="syprayer_settings[stripe_live_pk]" 
               value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php echo esc_html__('Your Stripe live publishable key (starts with pk_live_)', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function stripe_live_sk_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['stripe_live_sk']) ? $options['stripe_live_sk'] : '';
        ?>
        <input type="password" name="syprayer_settings[stripe_live_sk]" 
               value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php echo esc_html__('Your Stripe live secret key (starts with sk_live_)', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function paypal_test_mode_render() {
        $options = get_option('syprayer_settings');
        $checked = (isset($options['paypal_test_mode']) && $options['paypal_test_mode'] === 'yes') ? 'yes' : 'no';
        ?>
        <label>
            <input type="checkbox" name="syprayer_settings[paypal_test_mode]" value="yes" 
                   <?php checked($checked, 'yes'); ?> />
            <?php echo esc_html__('Enable PayPal test mode (Sandbox)', 'send-your-prayers'); ?>
        </label>
        <p class="description"><?php echo esc_html__('Use PayPal sandbox for development and testing', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function paypal_client_id_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['paypal_client_id']) ? $options['paypal_client_id'] : '';
        ?>
        <input type="text" name="syprayer_settings[paypal_client_id]" 
               value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php echo esc_html__('Your PayPal Client ID', 'send-your-prayers'); ?></p>
        <?php
    }
    
    public function paypal_client_secret_render() {
        $options = get_option('syprayer_settings');
        $value = isset($options['paypal_client_secret']) ? $options['paypal_client_secret'] : '';
        ?>
        <input type="password" name="syprayer_settings[paypal_client_secret]" 
               value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php echo esc_html__('Your PayPal Client Secret', 'send-your-prayers'); ?></p>
        <?php
    }
    
    private function display_recent_submissions() {
        $syprayer_db = new SYPRAYER_DB();
        $submissions = $syprayer_db->get_submissions(array('date_from' => gmdate('Y-m-d', strtotime('-7 days'))));
        
        if ($submissions) {
            echo '<ul class="syprayer-recent-list">';
            foreach (array_slice($submissions, 0, 5) as $submission) {
                echo '<li>';
                echo '<strong>' . esc_html($submission->user_email) . '</strong> - ';
                echo esc_html(wp_trim_words($submission->prayer_text, 5));
                echo ' <span class="syprayer-amount">(' . esc_html($submission->currency . $submission->amount) . ')</span>';
                echo '</li>';
            }
            echo '</ul>';
        } else {
            echo '<p>' . esc_html__('No recent submissions.', 'send-your-prayers') . '</p>';
        }
    }
}
?>