(function($) {
    'use strict';
    
    $(document).ready(function() {
        if (!$('#send-your-prayers-form').length) {
            return;
        }
        
        // Clear any existing payment data on page load
        clearPaymentData();
        
        // Initialize the payment system with delay to ensure Stripe.js is loaded
        setTimeout(function() {
            initPaymentSystem();
        }, 500);
    });
    
    function clearPaymentData() {
        // Clear any payment data that might be stuck
        if (typeof Storage !== 'undefined') {
            localStorage.removeItem('syprayer_payment_processing');
        }
    }
    
    function initPaymentSystem() {
        // Check if we have the AJAX object
        if (!window.syprayer_ajax) {
            console.error('syprayer_ajax object not found');
            showError('Payment system initialization failed. Please refresh the page.');
            return;
        }
        
        // Check if Stripe credentials are available
        if (!window.syprayer_ajax.stripe_pk || window.syprayer_ajax.stripe_pk.trim() === '') {
            console.error('Stripe public key not configured');
            // Don't show error if only PayPal is configured
            if (!window.syprayer_ajax.paypal_available) {
                showError('Payment gateway is not configured. Please contact the site administrator.');
            }
            // Continue initialization for PayPal-only setups
        }
        
        // Initialize based on available gateways
        if (window.syprayer_ajax.stripe_pk && window.syprayer_ajax.stripe_pk.trim() !== '') {
            initStripe();
        }
        
        // Setup form handlers (works for both Stripe and PayPal)
        setupFormHandlers();
        
        // Set initial state based on selected payment method
        updatePaymentMethodVisibility();
    }
    
    function initStripe() {
        // Check if Stripe.js is loaded
        if (typeof Stripe === 'undefined') {
            console.error('Stripe.js not loaded');
            // Try to load it again
            loadStripeJS();
            return;
        }
        
        try {
            // Initialize Stripe
            window.syprayerStripe = Stripe(window.syprayer_ajax.stripe_pk);
            
            // Create and mount card element
            createCardElement(window.syprayerStripe);
            
        } catch (error) {
            console.error('Failed to initialize Stripe:', error);
            showError('Failed to initialize payment system. Please refresh.');
        }
    }
    
    function loadStripeJS() {
        // Check if Stripe script is already in the page
        if ($('script[src*="stripe.com/v3"]').length === 0) {
            // Load Stripe.js dynamically
            var script = document.createElement('script');
            script.src = 'https://js.stripe.com/v3/';
            script.async = true;
            script.onload = function() {
                console.log('Stripe.js loaded dynamically');
                initStripe();
            };
            script.onerror = function() {
                console.error('Failed to load Stripe.js');
                showError('Failed to load payment gateway. Please refresh.');
            };
            document.head.appendChild(script);
        } else {
            // Script is already loading, wait for it
            var checkStripe = setInterval(function() {
                if (typeof Stripe !== 'undefined') {
                    clearInterval(checkStripe);
                    initStripe();
                }
            }, 100);
            
            // Timeout after 5 seconds
            setTimeout(function() {
                clearInterval(checkStripe);
                if (typeof Stripe === 'undefined') {
                    console.error('Stripe.js failed to load');
                    showError('Payment gateway failed to load. Please refresh.');
                }
            }, 5000);
        }
    }
    
    function createCardElement(stripe) {
        const container = $('#syprayer-payment-element');
        if (!container.length) {
            console.error('Payment element container not found');
            return;
        }
        
        try {
            // Create elements instance
            const elements = stripe.elements();
            
            // Create card element
            const cardElement = elements.create('card', {
                style: {
                    base: {
                        fontSize: '16px',
                        color: '#32325d',
                        fontFamily: '-apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif',
                        fontSmoothing: 'antialiased',
                        '::placeholder': {
                            color: '#aab7c4'
                        }
                    },
                    invalid: {
                        color: '#fa755a',
                        iconColor: '#fa755a'
                    }
                },
                hidePostalCode: true
            });
            
            // Mount the card element
            cardElement.mount('#syprayer-payment-element');
            
            // Store for later use
            window.syprayerCardElement = cardElement;
            
        } catch (error) {
            console.error('Failed to create card element:', error);
            showError('Failed to load payment form: ' + error.message);
        }
    }
    
    function setupFormHandlers() {
        // Toggle payment method visibility when selection changes
        $('input[name="payment_gateway"]').on('change', function() {
            updatePaymentMethodVisibility();
        });
        
        // Form submission
        $('#syprayer-form').on('submit', async function(e) {
            e.preventDefault();
            
            // Validate form
            if (!validateForm()) {
                return;
            }
            
            // Get form data
            const formData = getFormData();
            
            // Additional validation based on payment method
            if (formData.payment_gateway === 'stripe') {
                // Check if Stripe is available
                if (!window.syprayerStripe || !window.syprayerCardElement) {
                    showError('Credit card payment is not available. Please try PayPal or refresh the page.');
                    return;
                }
            } else if (formData.payment_gateway === 'paypal') {
                // Check if PayPal is available
                if (!window.syprayer_ajax.paypal_available) {
                    showError('PayPal is not available. Please use credit card or contact support.');
                    return;
                }
            }
            
            // Set loading state
            setLoading(true);
            
            // Process based on payment method
            if (formData.payment_gateway === 'paypal') {
                processPayPal(formData);
            } else {
                await processStripe(formData);
            }
        });
    }
    
    function updatePaymentMethodVisibility() {
        const method = $('input[name="payment_gateway"]:checked').val();
        
        if (method === 'stripe') {
            // Show Stripe form with animation
            $('#syprayer-stripe-payment-group').slideDown(300, function() {
                // Refresh card element if needed
                if (window.syprayerCardElement && $('#syprayer-payment-element').is(':empty')) {
                    try {
                        window.syprayerCardElement.mount('#syprayer-payment-element');
                    } catch (error) {
                        console.error('Failed to mount card element:', error);
                    }
                }
            });
        } else {
            // Hide Stripe form
            $('#syprayer-stripe-payment-group').slideUp(300);
        }
    }
    
    function validateForm() {
        // Check email
        const email = $('#syprayer-email').val().trim();
        if (!email || !isValidEmail(email)) {
            showError('Please enter a valid email address.');
            return false;
        }
        
        // Check prayer text
        const prayer = $('#syprayer-prayer').val().trim();
        if (!prayer) {
            showError('Please enter your prayer request.');
            return false;
        }
        
        // Check terms
        if (!$('#syprayer-terms').is(':checked')) {
            showError('Please accept the terms to continue.');
            return false;
        }
        
        return true;
    }
    
    function getFormData() {
        return {
            email: $('#syprayer-email').val().trim(),
            prayer: $('#syprayer-prayer').val().trim(),
            payment_gateway: $('input[name="payment_gateway"]:checked').val(),
            amount: window.syprayer_ajax.simple_amount,
            nonce: window.syprayer_ajax.nonce
        };
    }
    
    async function processStripe(formData) {
        // Check if card element exists
        if (!window.syprayerCardElement) {
            showError('Payment form not loaded. Please refresh.');
            setLoading(false);
            return;
        }
        
        try {
            // Create payment method
            const { paymentMethod, error } = await window.syprayerStripe.createPaymentMethod({
                type: 'card',
                card: window.syprayerCardElement,
                billing_details: {
                    email: formData.email
                }
            });
            
            if (error) {
                showError('Card error: ' + error.message);
                setLoading(false);
                return;
            }
            
            // Create payment via AJAX
            $.ajax({
                url: window.syprayer_ajax.ajax_url,
                type: 'POST',
                dataType: 'json',
                data: {
                    action: 'syprayer_process_payment',
                    email: formData.email,
                    prayer: formData.prayer,
                    amount: formData.amount,
                    payment_gateway: formData.payment_gateway,
                    payment_method_id: paymentMethod.id,
                    nonce: formData.nonce
                },
                success: function(response) {
                    if (response.success) {
                        if (response.data.requires_action) {
                            // Handle 3D Secure or other required actions
                            handlePaymentAction(response.data.client_secret);
                        } else if (response.data.redirect_url) {
                            // Redirect to success page
                            window.location.href = response.data.redirect_url;
                        } else {
                            // Default redirect
                            window.location.href = window.syprayer_ajax.thank_you_page_url + '?gateway=stripe';
                        }
                    } else {
                        showError(response.data || 'Payment failed.');
                        setLoading(false);
                    }
                },
                error: function(xhr, status, error) {
                    showError('Server error. Please try again.');
                    setLoading(false);
                }
            });
            
        } catch (error) {
            showError('Payment processing failed.');
            setLoading(false);
        }
    }
    
    function handlePaymentAction(clientSecret) {
        window.syprayerStripe.confirmCardPayment(clientSecret).then(function(result) {
            if (result.error) {
                showError(result.error.message);
                setLoading(false);
            } else {
                // Payment succeeded
                window.location.href = window.syprayer_ajax.thank_you_page_url + '?gateway=stripe';
            }
        });
    }
    
    function processPayPal(formData) {
        showMessage('Creating PayPal payment...', 'info');
        
        $.ajax({
            url: window.syprayer_ajax.ajax_url,
            type: 'POST',
            dataType: 'json',
            data: {
                action: 'syprayer_process_paypal',
                email: formData.email,
                prayer: formData.prayer,
                amount: formData.amount,
                nonce: formData.nonce
            },
            success: function(response) {
                if (response.success && response.data && response.data.approval_url) {
                    // Redirect to PayPal
                    window.location.href = response.data.approval_url;
                } else {
                    showError(response.data || 'Failed to create PayPal payment.');
                    setLoading(false);
                }
            },
            error: function(xhr, status, error) {
                showError('Connection to server failed. Please try again.');
                setLoading(false);
            }
        });
    }
    
    function showMessage(message, type) {
        const div = $('#syprayer-form-messages');
        div.removeClass().addClass('syprayer-message-' + type).text(message).show();
        
        if (type === 'info') {
            setTimeout(() => div.fadeOut(), 3000);
        }
    }
    
    function showError(message) {
        showMessage(message, 'error');
    }
    
    function setLoading(loading) {
        const btn = $('#syprayer-submit-prayer');
        if (loading) {
            btn.prop('disabled', true).addClass('loading');
            btn.find('.syprayer-button-text').hide();
            btn.find('.syprayer-spinner').show();
        } else {
            btn.prop('disabled', false).removeClass('loading');
            btn.find('.syprayer-button-text').show();
            btn.find('.syprayer-spinner').hide();
        }
    }
    
    function isValidEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }
    
})(jQuery);