import {useEffect} from "react";
import {useSelect} from '@wordpress/data';
import {useState} from '@wordpress/element';
import {Button, Modal, Panel, PanelBody, PanelRow} from '@wordpress/components';
import {__} from "@wordpress/i18n";
import {store as editorStore} from '@wordpress/editor';
import {RelatedPostsClient} from "../../../../clients/related-posts-client";
import {LinkingRelatedPostsTable} from "./linking-related-posts-table";
import RelatedPostsMetaBox from "../../../../meta-box/related-posts/meta-box";

const getTableData = (setLoading, setError, setEmpty, setRelatedPosts, postId, postContent, postIntro, postTitle) => {
    setLoading(true);
    setError(null);

    RelatedPostsClient.fetchRelatedPosts({
        postId: postId,
        content: postContent,
        introduction: postIntro,
        title: postTitle
    }).then((relatedPostsResponse) => {
        setEmpty(relatedPostsResponse.posts.length === 0);
        setRelatedPosts(relatedPostsResponse.posts);
        setLoading(false);
    }).catch((error) => {
        console.error(error);
        setError(error);
        setLoading(false);
    });
}

export const LinkingModal = ({currentPost, pluginSettings}) => {
    const [isOpen, setIsOpen] = useState(false);
    const [loading, setLoading] = useState(true);
    const [empty, setEmpty] = useState(false);
    const [error, setError] = useState(null);
    const [modalView, setModalView] = useState(null);
    const [isPluginCorrectlyConfigured, setIsPluginCorrectlyConfigured] = useState(false);

    const [relatedPosts, setRelatedPosts] = useState([]);

    const postContent = useSelect(
        (select) => select(editorStore).getEditedPostContent('postType', currentPost.type, currentPost.id),
        []
    );
    const postId = wp.data.select("core/editor").getCurrentPostId();
    const postIntro = (currentPost.excerpt !== "" ? currentPost.excerpt : currentPost.title);
    const postTitle = currentPost.title;

    useEffect(() => {
        if (isOpen) {
            getTableData(setLoading, setError, setEmpty, setRelatedPosts, postId, postContent, postIntro, postTitle)
        }
    }, [isOpen]);

    const onValidate = () => {
        RelatedPostsClient.saveRelatedPosts({
            postId: postId,
            posts: relatedPosts.map((post) => post.ID)
        })
            .then(([relatedPostsResponse]) => {
                RelatedPostsMetaBox.replaceRelatedPostsTableContent(relatedPostsResponse)

                setLoading(false);
                setIsOpen(false);
            });
    };


    const modalViews = {
        loading: () => (<div style={{textAlign: "center"}}><span className={"spinner semlinks-plugin-spinner"}/></div>),
        error: () => (<div>{__("An error occurred while finding links", "semlinks")}</div>),
        empty: () => (<div>{__("No links were found", "semlinks")}</div>),
        result: () => (
            <Panel>
                <PanelBody
                    title={__("Related posts", "semlinks")}
                    initialOpen={true}
                >
                    <PanelRow>
                        <LinkingRelatedPostsTable
                            posts={relatedPosts}
                            setPosts={setRelatedPosts}
                        />
                    </PanelRow>
                </PanelBody>
            </Panel>
        )
    }

    useEffect(() => {
        let currentView = "result";
        if (error) {
            currentView = "error";
        }
        if (loading) {
            currentView = "loading";
        }
        if (empty) {
            currentView = "empty";
        }
        setModalView(modalViews[currentView]())
    }, [modalView, loading, error, empty]);

    useEffect(() => {
        if (pluginSettings !== undefined) {
            setIsPluginCorrectlyConfigured(
                pluginSettings.is_api_key_valid === "true"
                && pluginSettings.allowed_features.includes("LOOKALIKE")
            )
        }
    }, [pluginSettings]);

    const onClick = (e) => {
        if (!isPluginCorrectlyConfigured) {
            e.preventDefault();
            alert(__("SemLinks is not correctly configured, please contact your administrator.", "semlinks"))
            return;
        }

        setIsOpen(true)
    }

    return (
        <div>
            <button
                className="button button-primary semlinks-plugin-meta-box-action-button"
                onClick={onClick}
            >
                {__('Find links', 'semlinks')}
            </button>
            {isOpen && (
                <Modal
                    title={__('SemLinks', 'semlinks')}
                    onRequestClose={() => setIsOpen(false)}
                    style={{minWidth: "90%", paddingBottom: "64px"}}
                >
                    {modalView}

                    <div style={{
                        position: "fixed",
                        bottom: "10px",
                        right: "10px",
                        backgroundColor: "white",
                    }}>
                        <Button
                            style={{marginRight: "1em"}}
                            variant={"secondary"}
                            onClick={() => setIsOpen(false)}
                        >
                            {__('Cancel', 'semlinks')}
                        </Button>

                        <Button
                            variant={"primary"}
                            onClick={onValidate}
                            disabled={loading || error}
                        >
                            {__('Create', 'semlinks')}
                        </Button>
                    </div>
                </Modal>
            )}
        </div>
    );
}