<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\ViewModel;

use Automattic\WooCommerce\Admin\Overrides\Order;
use Selfnamed\Model\Orders_Sync;
use Selfnamed\Model\Setup;
use WC_Order;

require_once plugin_dir_path(WP_PLUGIN_DIR) . 'plugins/woocommerce/includes/wc-order-functions.php';

/**
 * Orders class.
 */
class Orders {
	/**
	 * Setup model.
	 *
	 * @var Setup
	 */
	private $setup;

	/**
	 * Orders Sync model.
	 *
	 * @var Orders_Sync
	 */
	private $orders_sync;

	/**
	 * Selfnamed orders.
	 *
	 * @var array
	 */
	public $selfnamed_orders = array();

	/**
	 * Current page for pagination.
	 *
	 * @var int
	 */
	public $current_page = 1;

	/**
	 * Orders per page for pagination.
	 *
	 * @var int
	 */
	public $per_page = 20;

	/**
	 * Init total pages for pagination.
	 *
	 * @var int
	 */
	public $total_pages = 1;

	/**
	 * Total items for pagination.
	 *
	 * @var int
	 */
	public $total_items = 0;

	/**
	 * Args to get products.
	 *
	 * @var array
	 */
	private $args = array(
		'orderby'  => 'date',
		'order'    => 'DESC',
		'paginate' => true,
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->setup       	= new Setup();
		$this->orders_sync 	= new Orders_Sync();
		$this->per_page 	= $this->get_per_page();
		$this->current_page = array_key_exists( 'paged', $_GET ) ? max( 1, (int) $_GET['paged'] ) : 1;
	}

	/**
	 * Handle actions.
	 *
	 * @return void
	 */
	public function handle_actions(): void {
	if ( array_key_exists( 'per_page', $_POST ) ) {
		check_admin_referer( 'selfnamed-orders-per-page', 'selfnamed_nonce' );
		$per_page = (int) $_POST['per_page'];	
		$this->update_per_page( $per_page );
		$this->current_page = 1;
		wp_safe_redirect( $this->get_page_url() );
		exit;
	}

	if ( array_key_exists( 'action', $_POST ) && $_POST['action'] === 'sync_selected' ) {
		check_admin_referer( 'selfnamed-orders-sync', 'selfnamed_nonce' );
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Insufficient permissions.', 'selfnamed-cosmetics-on-demand' ) );
		}
			$order_ids = array_key_exists( 'order_ids', $_POST ) ? array_map( 'intval', $_POST['order_ids'] ) : array();
			$this->sync_orders( $order_ids );
		}
	}

	/**
	 * Get list of orders.
	 *
	 * @return void
	 */
	public function get_orders_list(): void {
		$this->selfnamed_orders = array();
		$args                   = $this->args;
		$args['limit']          = $this->per_page;
		$args['paged']          = $this->current_page;
		
		$args['meta_key']     	= '_is_selfnamed_order';
		$args['meta_value']   	= '1';
		$args['meta_compare'] 	= '=';

		$result = wc_get_orders( $args );

		if ( empty( $result->orders ) ) {
			echo '<div class="notice notice-info"><p>No orders found.</p></div>';
			return;
		}

		$this->total_pages = $result->max_num_pages;
		$this->total_items = isset( $result->total ) ? (int) $result->total : 0;

		foreach ( $result->orders as $order ) {
			$woo_order = null;
			if ( $order instanceof Order ) {
				$woo_order = $order;
			} else {
				$order_id  = $order->get_id();
				$woo_order = new WC_Order( $order_id );
			}

			$shipping = $woo_order->get_address( 'shipping' );

			$this->selfnamed_orders[] = [
				'id'               => $woo_order->get_id(),
				'link'             => $woo_order->get_edit_order_url(),
				'name'             => '#' . $woo_order->get_id(),
				'customer'         => $shipping['first_name'] . ' ' . $shipping['last_name'],
				'planned_delivery' => $woo_order->get_meta( '_selfnamed_order_planned_delivery' ),
				'updated_at'       => $order->get_meta( '_selfnamed_order_updated_at' ),
				'message'          => $order->get_meta( '_selfnamed_order_message' )
			];
		}
	}

	/**
	 * Get page URL.
	 *
	 * @return string
	 */
	public function get_page_url($base = false) {
		$args = array(
			'page' => 'self-named',
			'tab' => 'orders',
		);

		if ( !$base && $this->current_page > 1 ) {
			$args['paged'] = $this->current_page;
		}

		return add_query_arg( $args, admin_url( 'admin.php' ) );
	}

	/**
	 * Format date.
	 *
	 * @param string $time The time.
	 *
	 * @return false|string|null
	 */
	public function format_date( string $time ) {
		if ( empty( $time ) ) {
			return null;
		}

		return gmdate( 'H:i d/m/Y', $time );
	}

	/**
	 * Get order status based on selfnamed status.
	 *
	 * @param string $order_id The order id.
	 *
	 * @return string
	 */
	public function get_order_status( $order_id ): string {
		$order  = wc_get_order( $order_id );
		$status = $order->get_meta( '_selfnamed_order_status' );

		switch ( $status ) {
			case 'PAYMENT_PENDING':
				$message = '<span class="">Payment Pending</span>';
				break;
			case 'ACCEPTED':
				$message = '<span class="success">Accepted</span>';
				break;
			case 'CANCELLED':
				if ( $order->get_status() !== 'processing' ) {
					$message = '<span class="alert">Cancelled</span>';
				} else {
					// When changing the status back to precessing after cancelled, reset selfnamed meta information.
					$this->orders_sync->reset_selfnamed_meta( $order );
					$message = '<span class="alert">Not Synced</span>';
				}
				break;
			case 'FILE_ISSUES':
				$message = '<span class="alert">File Issues</span>';
				break;
			case 'FILE_ACCEPTED':
				$message = '<span class="">File Accepted</span>';
				break;
			case 'PRINTING':
				$message = '<span class="">Printing</span>';
				break;
			case 'ASSEMBLING':
				$message = '<span class="">Assembling</span>';
				break;
			case 'SHIPPED':
				$message = '<span class="success">Shipped</span>';
				break;
			case 'DELIVERED':
				$message = '<span class="success">Delivered</span>';
				break;
			case 'FILE_SENT':
				$message = '<span class="">File Sent</span>';
				break;
			case 'AWAITING_CANCELLATION':
				$message = '<span class="">Awaiting Cancellation</span>';
				break;
			case 'CANCELLATION_FAILED':
				$message = '<span class="alert">Cancellation Failed</span>';
				break;
			case 'DESIGN_CHECK':
				$message = '<span class="">Design Check</span>';
				break;
			default:
				$message = '<span class="alert">Not Synced</span>';
		}

		return $message;
	}

	/**
	 * Get tracking numbers returned from Selfnamed in format for the orders table.
	 *
	 * @param int $order_id The WooCommerce order id.
	 *
	 * @return array
	 */
	public function get_tracking_numbers( int $order_id ): array {
		$tracking_numbers_list = array();
		$order                 = wc_get_order( $order_id );

		$tracking_numbers_meta = $order->get_meta( 'selfnamed_order_tracking_numbers' );
		$tracking_urls_meta    = $order->get_meta( 'selfnamed_order_tracking_urls' );

		if ( isset( $tracking_numbers_meta ) && isset( $tracking_urls_meta ) ) {
			$tracking_numbers = explode( ', ', $tracking_numbers_meta );
			$tracking_urls    = explode( ', ', $tracking_urls_meta );

			foreach ( $tracking_numbers as $index => $number ) {
				if ( ! empty( $number ) ) {
					$url = $tracking_urls[ $index ];

					$tracking_numbers_list[] = array(
						'number' => $number,
						'url'    => $url,
					);
				}
			}
		}

		return $tracking_numbers_list;
	}

	/**
	 * Sync order action.
	 *
	 * @return false|void
	 */
	private function sync_orders( array $order_ids = array() ) {
		if ( ! $this->setup->is_ready() ) {
			return false;
		}

		$this->orders_sync->sync( $order_ids );
	}

	/**
	 * Get per-page preference from current user's meta.
	 *
	 * @return int
	 */
	private function get_per_page(): int {
		$user_id  = get_current_user_id();
		$per_page = (int) get_user_meta( $user_id, 'selfnamed_orders_per_page', true );

		return $per_page > 0 ? $per_page : $this->per_page;
	}

	/**
	 * Update per-page preference and persist to current user's meta.
	 *
	 * @param int $per_page The items per page value.
	 *
	 * @return void
	 */
	private function update_per_page( int $per_page ): void {
		if ( $per_page > 0 ) {
			$this->per_page = $per_page;
			update_user_meta( get_current_user_id(), 'selfnamed_orders_per_page', $per_page );
		}
	}
}
