<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

/**
 * Upgrader class.
 */
class Upgrader {
	/**
	 * Option name used to store current plugin version.
	 */
	const VERSION_OPTION = 'selfnamed_plugin_version';

	/**
	 * Option name used to store upgrader state/progress.
	 */
	const UPGRADE_STATE_OPTION = 'selfnamed_upgrade_state';

	/**
	 * Public entrypoint to initialize upgrade checks.
	 *
	 * @return void
	 */
	public static function create() {
		new self();
	}

	/**
	 * Constructor: registers hooks.
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'maybe_upgrade' ) );
	}

	/**
	 * Check if an upgrade is needed and perform it in small batches.
	 *
	 * @return void
	 */
	public function maybe_upgrade(): void {
		$current_version = \selfnamed_get_current_plugin_version();
		$stored_version  = get_option( self::VERSION_OPTION, '0.0.0' );

		if ( empty( $current_version ) || version_compare( $stored_version, $current_version, '>=' ) ) {
			return;
		}

		// Run upgrade routines in batches. If completed, persist the new version.
		$completed = $this->run_upgrade_routines( $stored_version, $current_version );
		if ( $completed ) {
			update_option( self::VERSION_OPTION, $current_version );
			delete_option( self::UPGRADE_STATE_OPTION );
		}
	}

	/**
	 * Execute upgrade routines between $from and $to. Returns true when done.
	 *
	 * @param string $from Previous installed version.
	 * @param string $to   Current plugin version.
	 *
	 * @return bool True if all routines completed.
	 */
	private function run_upgrade_routines( string $from, string $to ): bool {
		$state = get_option( self::UPGRADE_STATE_OPTION, array() );

		// Backfill _selfnamed_order meta for existing orders.
		if ( ! isset( $state['backfill_selfnamed_flag_done'] ) || true !== (bool) $state['backfill_selfnamed_flag_done'] ) {
			$done = $this->backfill_selfnamed_flag( $state );
			$state['backfill_selfnamed_flag_done'] = $done;
			update_option( self::UPGRADE_STATE_OPTION, $state );
			if ( ! $done ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Backfill `_selfnamed_order` meta for orders that contain Selfnamed products.
	 * Runs in small batches and persists progress in an option.
	 *
	 * @param array $state Mutable upgrade state.
	 *
	 * @return bool True if backfill completed.
	 */
	private function backfill_selfnamed_flag( array &$state ): bool {
		$page  = isset( $state['backfill_page'] ) ? max( 1, (int) $state['backfill_page'] ) : 1;
		$limit = 100;

		$args = array(
			'limit'    => $limit,
			'page'     => $page,
			'orderby'  => 'date',
			'order'    => 'DESC',
			'paginate' => true,
			'return'   => 'ids'
		);

		$result = wc_get_orders( $args );

		if ( empty( $result ) || empty( $result->orders ) ) {
			return true;
		}

		$order_ids = $result->orders;

		foreach ( $order_ids as $order_id ) {
			$order = wc_get_order( $order_id );
			if ( ! $order || ! is_a( $order, 'WC_Order' ) ) {
				continue;
			}

			if ($order->get_meta( '_is_selfnamed_order' ) == 1 ) {
				continue;
			}

			if ($order->get_meta( '_selfnamed_order_id' ) || $this->contains_selfnamed_products( $order ) ) {
				$order->update_meta_data( '_is_selfnamed_order', 1 );
				$order->save();
			}
		}

		if ( $page < $result->max_num_pages ) {
			$state['backfill_page'] = $page + 1;
			return false;
		}

		return true;
	}

	/**
	 * Checks whether an order contains any Selfnamed product.
	 *
	 * @param \WC_Order $order The order.
	 *
	 * @return bool
	 */
	private function contains_selfnamed_products( $order ): bool {
		$items = $order->get_items();
		if ( empty( $items ) ) {
			return false;
		}

		foreach ( $items as $item ) {
			if ( is_a( $item, 'WC_Order_Item_Product' ) ) {
				$product = $item->get_product();
				if ( $product && $product->get_meta( '_selfnamed_product_id' ) ) {
					return true;
				}
			}
		}

		return false;
	}
}


