<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

use WC_Shipping_Method;
use Selfnamed\Model\Product;

/**
 * Selfnamed Shipping Methods Class
 */
class Shipping_Method extends WC_Shipping_Method {
	const METHOD_ID    = 'selfnamed_shipping';
	const METHOD_TITLE = 'Selfnamed Shipping';
	const PACKAGE_ID   = 'selfnamed';

	/**
	 * Api model.
	 *
	 * @var Api
	 */
	private $api;

	/**
	 * Setup model.
	 *
	 * @var Setup
	 */
	private $setup;

	/**
	 * Selfnamed shipping zones.
	 *
	 * @var array|null
	 */
	private $selfnamed_zones = null;

	/**
	 * Selfnamed shipping methods.
	 *
	 * @var array|null
	 */
	private $selfnamed_methods = null;

	/**
	 * Initialization.
	 *
	 * @return void
	 */
	public static function create() {
		new self();
	}

	/**
	 * Constructor
	 *
	 * @param number $instance_id The unique instance id.
	 */
	public function __construct( $instance_id = 0 ) {
		parent::__construct( $instance_id );

		$this->api          = new Api();
		$this->setup        = new Setup();
		$this->id           = self::METHOD_ID;
		$this->method_title = self::METHOD_TITLE;
		$this->title        = self::METHOD_TITLE;

		$this->init_form_fields();
		$this->init_settings();

		$this->enabled = $this->get_option( 'enabled' );

		add_action(
			'woocommerce_update_options_shipping_' . $this->id,
			array( $this, 'process_admin_options' )
		);

		add_action(
			'woocommerce_load_shipping_methods',
			array( $this, 'limit_method_for_sn_products' ),
			1000
		);

		add_action(
			'woocommerce_cart_shipping_packages',
			array( $this, 'mark_selfnamed_packages' ),
			1000
		);
	}

	/**
	 * Check if the selfnamed shipping methods are enabled.
	 *
	 * @return bool
	 */
	public function is_enabled(): bool {
		return parent::is_enabled() || ( 'no' !== $this->enabled && $this->enabled );
	}

	/**
	 * Show selfnamed shipping methods only for the selfnamed products.
	 *
	 * @param array $package The current order's package.
	 *
	 * @return void
	 */
	public function limit_method_for_sn_products( array $package = array() ): void {
		if ( $package && array_key_exists( self::PACKAGE_ID, $package ) && $package[ self::PACKAGE_ID ] ) {
			if ( $this->is_enabled() ) {
				WC()->shipping()->unregister_shipping_methods();
				WC()->shipping()->register_shipping_method( $this );
			}
		} elseif ( ! $package ) {
			WC()->shipping()->register_shipping_method( $this );
		}
	}

	/**
	 * Calculate the products total cost.
	 *
	 * @param array $content The cart content.
	 *
	 * @return float
	 */
	private function calc_content_cost( array $content = array() ): float {
		$cost = 0;

		foreach ( $content as $line_item ) {
			$cost += $line_item['line_total'];
		}

		return $cost;
	}

	/**
	 * Mark selfnamed packages.
	 *
	 * @param array $packages The current order's package.
	 *
	 * @return array
	 */
	public function mark_selfnamed_packages( array $packages = array() ): array {
		if ( ! $this->is_enabled() ) {
			return $packages;
		}

		$return_packages = array();

		foreach ( $packages as $package ) {
			$sn_contents = array();
			$wc_contents = array();

			foreach ( $package['contents'] as $id => $line_item ) {
				$is_sn = (bool) get_post_meta( $line_item['product_id'], Product::PRODUCT_IDENTIFIER_KEY, true );

				if ( $is_sn ) {
					$sn_contents[ $id ] = $line_item;
				} else {
					$wc_contents[ $id ] = $line_item;
				}
			}

			if ( count( $sn_contents ) > 0 ) {
				$sn_package                     = $package;
				$sn_package[ self::PACKAGE_ID ] = true;
				$sn_package['contents']         = $sn_contents;
				$sn_package['contents_cost']    = $this->calc_content_cost( $sn_contents );
				$return_packages[]              = $sn_package;
			}

			if ( count( $wc_contents ) > 0 ) {
				$wc_package                     = $package;
				$wc_package[ self::PACKAGE_ID ] = false;
				$wc_package['contents']         = $wc_contents;
				$wc_package['contents_cost']    = $this->calc_content_cost( $wc_contents );
				$return_packages[]              = $wc_package;
			}
		}

		return $return_packages;
	}

	/**
	 * Get selfnamed shipping zones.
	 *
	 * @return array
	 */
	private function get_selfnamed_zones(): array {
		if ( ! $this->selfnamed_zones ) {
			$api_response = $this->api->get_all_shipping_zones();

			if ( ! $api_response->error ) {
				$this->selfnamed_zones = $api_response->data;
			}
		}

		return $this->selfnamed_zones;
	}

	/**
	 * Calculate the order's items weight
	 *
	 * @param array $package The current order's package.
	 *
	 * @return float
	 */
	private function calc_package_weight( array $package = array() ): float {
		$weight_kg = 0;

		foreach ( $package['contents'] as $line_item ) {
			$product           = $line_item['data'];
			$product_weight_kg = (float) $product->get_weight();
			$weight_kg        += ( $product_weight_kg * $line_item['quantity'] );
		}

		$weight_grams = $weight_kg * 1000;

		return $weight_grams;
	}

	/**
	 * Calculate the shipping method.
	 *
	 * @param array $package The current order's package.
	 *
	 * @return void
	 */
	public function calculate_shipping( $package = array() ): void {
		$zones = $this->get_selfnamed_zones();

		$available_zones = array();
		foreach ( $zones as $zone ) {
			if ( is_array( $zone->countries ) ) {
				$match = array_intersect( array( $package['destination']['country'] ), $zone->countries );

				if ( count( $match ) ) {
					$available_zones[] = $zone;
				}
			}
		}

		if ( count( $available_zones ) ) {
			$weight = $this->calc_package_weight( $package );

			foreach ( $available_zones as $zone ) {
				$price = null;

				foreach ( $zone->rates as $rate ) {
					if ( $weight >= $rate->{'totalWeightFrom'} && $weight <= $rate->{'totalWeightTo'} ) {
						$price = $rate->price;
					}
				}

				if ( ! is_null( $price ) ) {
					$this->add_rate(
						array(
							'id'        => $this->id . '_' . $zone->id,
							'label'     => ucfirst( strtolower( $zone->{'shippingGroup'} ) ),
							'cost'      => $price,
							'calc_tax'  => 'per_order',
							'meta_data' => array(
								'_selfnamed_method_shipping_group' => $zone->{'shippingGroup'},
							),
						)
					);
				}
			}
		}
	}

	/**
	 * Init form fields.
	 *
	 * @return void
	 */
	public function init_form_fields(): void {
		if ( ! $this->setup->is_ready() ) {
			$admin_url                 = admin_url( 'admin.php?page=self-named&tab=getting-started' );
			$this->form_fields['info'] = array(
				'type'        => 'title',
				'description' => "Selfnamed integration is not fully configured yet. 
    Please complete the <a href=\"$admin_url\">Getting Started</a> section for Selfnamed plugin.",
			);

			return;
		}

		$this->form_fields = array(
			'info'    => array(
				'type'        => 'title',
				'description' => 'Enabling this shipping method <b>will disable</b> other shipping methods for products fulfilled by Selfnamed',
			),
			'enabled' => array(
				'title'   => __( 'Enable/Disable', 'selfnamed-cosmetics-on-demand' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this shipping method', 'selfnamed-cosmetics-on-demand' ),
				'default' => 'no',
			),
		);
	}
}
