<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

/**
 * Setup class.
 */
class Setup {
	const API_KEY_PATH                     = 'selfnamed_api_key';
	const PLATFORM_ACCESS_TOKEN            = 'selfnamed_platform_access_token';
	const IS_USING_CUSTOM_SHIPPING_PROFILE = 'selfnamed_is_using_custom_shipping_profile';

	/**
	 * The persisted selfnamed api key.
	 *
	 * @var string
	 */
	private $api_key;

	/**
	 * The menu slug.
	 *
	 * @var string
	 */
	public $menu_slug = 'self-named';

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_option( self::API_KEY_PATH );

		$this->api_key = (string) get_option( self::API_KEY_PATH );
	}

	/**
	 * Check if the api key is set.
	 *
	 * @return bool
	 */
	public function is_ready(): bool {
		return (bool) $this->api_key;
	}

	/**
	 * Get the persisted api key.
	 *
	 * @return string
	 */
	public function get_api_key(): string {
		return $this->api_key;
	}

	/**
	 * Send the api key for creating a new store to the Selfnamed.
	 *
	 * @param string $key The api key from the Selfnamed.
	 *
	 * @return void
	 */
	public function set_api_key( string $key ): void {
		if ( $key ) {
			$platform_access_token = wp_generate_uuid4();

			$site_url = get_option( 'siteurl' );
			// Remove trailing slash from API_URL if added.
			if ( substr( $site_url, - 1 ) === '/' ) {
				$site_url = substr( $site_url, 0, - 1 );
			}
			$callback_url = $site_url . '/wp-json/sn/notification';

			$result =
				API::send_api_key(
					$site_url,
					get_option( 'home' ),
					$callback_url,
					$platform_access_token,
					$key,
					get_option( 'blogname' )
				);

			if ( ! $result->error ) {
				update_option( self::API_KEY_PATH, $key );
				update_option( self::PLATFORM_ACCESS_TOKEN, $platform_access_token );
				update_option( self::IS_USING_CUSTOM_SHIPPING_PROFILE, wp_json_encode( $result->data->{'isUsingCustomShippingProfile'} ) );
			}
		} else {
			delete_option( self::API_KEY_PATH );
			add_option( self::API_KEY_PATH );
		}

		$this->api_key = (string) get_option( self::API_KEY_PATH );
	}

	/**
	 * Save the data from the billing information form.
	 *
	 * @param string $name      The company's name.
	 * @param string $country   The company's country.
	 * @param string $city      The company's city.
	 * @param string $street    The company's street.
	 * @param string $street_no The company's street number.
	 * @param string $flat_no   The company's flat number.
	 * @param string $zip       The company's zip.
	 * @param string $vat_nr    The company's vat number.
	 * @param string $reg_nr    The company's registration number.
	 * @param string $phone     The company's phone number.
	 * @param string $email     The company's email.
	 *
	 * @return void
	 */
	public function set_billing_information(
		string $name,
		string $country,
		string $city,
		string $street,
		string $street_no,
		string $flat_no,
		string $zip,
		string $vat_nr,
		string $reg_nr,
		string $phone,
		string $email
	) {
		update_option(
			'_selfnamed_billing_information',
			serialize(
				array(
					'name'     => $name,
					'country'  => $country,
					'city'     => $city,
					'street'   => $street,
					'streetNo' => $street_no,
					'flatNo'   => $flat_no,
					'zip'      => $zip,
					'vatNr'    => $vat_nr,
					'regNr'    => $reg_nr,
					'phone'    => $phone,
					'email'    => $email,
				)
			)
		);
	}

	/**
	 * Get data from the billing information form.
	 *
	 * @return mixed|string[]
	 */
	public function get_billing_information() {
		$serialized_data = get_option( '_selfnamed_billing_information' );

		if ( ! $serialized_data ) {
			return array(
				'name'     => '',
				'country'  => '',
				'city'     => '',
				'street'   => '',
				'streetNo' => '',
				'flatNo'   => '',
				'zip'      => '',
				'vatNr'    => '',
				'regNr'    => '',
				'phone'    => '',
				'email'    => '',
			);
		}

		$data = unserialize( $serialized_data );

		return array(
			'name'     => $data['name'] ?? '',
			'country'  => $data['country'] ?? '',
			'city'     => $data['city'] ?? '',
			'street'   => $data['street'] ?? '',
			'streetNo' => $data['streetNo'] ?? '',
			'flatNo'   => $data['flatNo'] ?? '',
			'zip'      => $data['zip'] ?? '',
			'vatNr'    => $data['vatNr'] ?? '',
			'regNr'    => $data['regNr'] ?? '',
			'phone'    => $data['phone'] ?? '',
			'email'    => $data['email'] ?? '',
		);
	}
}
