<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

use Selfnamed\ViewModel\Pagination;
use stdClass;
use WP_Filesystem_Direct;

/**
 * Product class.
 */
class Product {
	const PRODUCT_IDENTIFIER_KEY = '_selfnamed_product_id';

	/**
	 * If the section is ready/visible.
	 *
	 * @var bool
	 */
	public $ready = false;

	/**
	 * Pagination view model.
	 *
	 * @var Pagination
	 */
	public $pagination;

	/**
	 * Api model.
	 *
	 * @var Api
	 */
	private $api;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->ready      = (bool) get_option( Setup::API_KEY_PATH );
		$this->api        = new Api();
		$this->pagination = new Pagination();
	}

	/**
	 * Extract products ids from products.
	 *
	 * @param array|null $products The products.
	 *
	 * @return array
	 */
	public function extract_product_ids( ?array $products = array() ): array {
		$ids = array();

		if ( is_array( $products ) ) {
			foreach ( $products as $product ) {
				$ids[] = (string) $product->id;
			}
		}

		return $ids;
	}

	/**
	 * Get selfnamed products.
	 *
	 * @param int $page The page number.
	 * @param int $size The products count per page.
	 *
	 * @return stdClass
	 */
	public function get_products( int $page = 1, int $size = Api::PER_PAGE ): stdClass {
		$response = $this->api->get_products( $page, $size );

		if ( $response->error ) {
			return $response;
		}

		$sn_products          = $response->data;
		$wc_products          = wc_get_products(
			array(
				'limit'        => - 1,
				'meta_key'     => self::PRODUCT_IDENTIFIER_KEY,
				'meta_value'   => $this->extract_product_ids( $sn_products ),
				'meta_compare' => 'IN',
			)
		);
		$wc_products_by_sn_id = array();

		foreach ( $wc_products as $wc_product ) {
			$sn_id                          =
				get_post_meta( $wc_product->get_id(), self::PRODUCT_IDENTIFIER_KEY, true );
			$wc_products_by_sn_id[ $sn_id ] = $wc_product;
		}

		$products = array();

		if ( is_array( $sn_products ) ) {
			foreach ( $sn_products as $product ) {
				$final_product              = $product;
				$final_product->is_in_stock = $product->{'isInStock'};
				$final_product->synced      = array_key_exists( $product->id, $wc_products_by_sn_id );
				$final_product->wc_id       = null;

				if ( $final_product->synced ) {
					$wc_product           = $wc_products_by_sn_id[ $product->id ];
					$final_product->wc_id = $wc_product->get_id();
					$final_product->price = $wc_product->get_price();
				}

				$products[] = $final_product;
			}
		}

		$response->data = $products;

		return $response;
	}

	/**
	 * Create selfnamed products in WooCommerce.
	 *
	 * @param int $page The page number.
	 * @param int $size The products count per page.
	 *
	 * @return bool
	 */
	public function create_all_products( int $page = 1, int $size = 10 ): bool {
		$response = $this->get_products( $page, $size );

		if ( $response->error ) {
			return false;
		}

		foreach ( $response->data as $product ) {
			if ( ! $product->wc_id ) {
				$this->create_product( $product );
			}
		}

		return true;
	}

	/**
	 * Create a new product in WooCommerce.
	 *
	 * @param object $product The selfnamed product data.
	 *
	 * @return int|null
	 */
	public function create_product( object $product ): ?int {
		if ( ! $product ) {
			return null;
		}

		$product_id = wp_insert_post(
		/**
		 * Create a new product.
		 *
		 * @since 1.0.0
		 */
			apply_filters(
				'woocommerce_new_product_data',
				array(
					'post_type'      => 'product',
					'post_status'    => 'draft',
					'post_author'    => 1,
					'post_title'     => $product->name,
					'post_content'   => $product->description,
					'comment_status' => 'closed',
					'ping_status'    => 'closed',
					'post_date'      => current_time( 'mysql', false ),
					'post_date_gmt'  => current_time( 'mysql', true ),
					'post_name'      => $product->name,
				)
			),
			true
		);

		if ( $product_id && ! is_wp_error( $product_id ) ) {
			update_post_meta( $product_id, self::PRODUCT_IDENTIFIER_KEY, $product->id );

			return $this->update_product_meta( $product_id, $product );
		} else {
			return null;
		}
	}

	/**
	 * Update meta product information.
	 *
	 * @param int    $product_id The existing product id.
	 * @param object $product    The selfnamed products data.
	 *
	 * @return int
	 */
	public function update_product_meta( int $product_id, object $product ): int {
		if ( ! empty( $product->tags ) ) {
			wp_set_object_terms( $product_id, $product->tags, 'product_tag' );
		}

		$wc_product = wc_get_product( $product_id );
		$wc_product->set_manage_stock( false );

		$has_featured_image = (bool) get_post_thumbnail_id( $product_id );

		if ( $product->images && is_array( $product->images ) ) {
			$image_idx = 0;
			foreach ( $product->images as $image_url ) {
				$is_first_image = ($image_idx === 0);
				$image_idx++;
				$set_as_thumbnail = ( $is_first_image && !$has_featured_image );
				$this->set_image( $product_id, $image_url, $set_as_thumbnail );
			}
		} elseif ( $product->image ) {
			$this->set_image( $product_id, $product->image, !$has_featured_image );
		}

		$wc_product->set_price( $product->price );
		$wc_product->set_regular_price( $product->price );
		$wc_product->set_stock_status( $product->is_in_stock ? 'instock' : 'outofstock' );
		update_post_meta( $product_id, '_volume', $product->volume );

		$product_weight_kg = $product->weight / 1000;
		$wc_product->set_weight( $product_weight_kg );

		$wc_product->save();

		return $product_id;
	}

	/**
	 * Upload image for an existing product.
	 *
	 * @param int    $post_id          The product's id.
	 * @param string $image_url        The selfnamed image product's url.
	 * @param bool   $set_as_thumbnail If the image should be set as thumbnail.
	 *
	 * @return void
	 */
	private function set_image( int $post_id, string $image_url, bool $set_as_thumbnail = true ): void {
		$image_name       = substr( $image_url, strrpos( $image_url, '/' ) + 1 );
		$upload_dir       = wp_upload_dir(); // Set upload folder.
		$image_data       = wp_remote_retrieve_body( wp_remote_get( $image_url ) ); // Get image data.
		$unique_file_name = wp_unique_filename( $upload_dir['path'], $image_name ); // Generate unique name.
		$filename         = basename( $unique_file_name ); // Create image file name.

		// Check folder permission and define file location.
		if ( wp_mkdir_p( $upload_dir['path'] ) ) {
			$file = $upload_dir['path'] . '/' . $filename;
		} else {
			$file = $upload_dir['basedir'] . '/' . $filename;
		}

		$wp_file_system = new WP_Filesystem_Direct( null );

		// Create the image  file on the server.
		$wp_file_system->put_contents( $file, $image_data );

		$path_parts = pathinfo( $file );

		if ( substr( $file, - strlen( $path_parts['extension'] ) ) !== $path_parts['extension'] ) {
			$new_file = $file . '.' . $path_parts['extension'];
			$wp_file_system->move( $file, $new_file );
			$file     = $new_file;
			$filename = basename( $file );
		}

		// Check image file type.
		$wp_filetype = wp_check_filetype( $filename, null );

		// Set attachment data.
		$attachment = array(
			'post_mime_type' => $wp_filetype['type'],
			'post_title'     => sanitize_file_name( $filename ),
			'post_content'   => '',
			'post_status'    => 'inherit',
		);

		// Create the attachment.
		$attach_id = wp_insert_attachment( $attachment, $file, $post_id );

		// Include image.php.
		require_once ABSPATH . 'wp-admin/includes/image.php';

		// Define attachment metadata.
		$attach_data = wp_generate_attachment_metadata( $attach_id, $file );

		// Assign metadata to attachment.
		wp_update_attachment_metadata( $attach_id, $attach_data );

		if ( $set_as_thumbnail ) {
			// And finally assign featured image to post.
			set_post_thumbnail( $post_id, $attach_id );
		} else {
			$images_meta = get_post_meta( $post_id, '_product_image_gallery', true );
			$images_meta = $images_meta ? $images_meta . ',' . $attach_id : $attach_id;
			update_post_meta( $post_id, '_product_image_gallery', $images_meta );
		}
	}
}
