<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

/**
 * OrderSync class.
 */
class Orders_Sync {
	/**
	 * Api model.
	 *
	 * @var Api
	 */
	private $api;

	/**
	 * Args to get products.
	 *
	 * @var array
	 */
	private $args = array(
		'status'   => array( 'wc-processing' ),
		'orderby'  => 'date',
		'order'    => 'DESC',
		'limit'    => -1,
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->api = new Api();
	}

	/**
	 * Send orders from WooCommerce store to the Selfnamed.
	 *
	 * @param array $order_ids Array of order IDs.
	 * 
	 * @return void
	 */
	public function sync( array $order_ids = array() ) {
		$orders = $this->get_all( $order_ids );

		foreach ( $orders as $order ) {
			$response = $this->api->send_orders( $order['invoice'], $order['orderItems'], $order['shippingGroup'] );

			$order_class        = wc_get_order( $order['id'] );
			$message            = '';
			$planned_delivery   = '';
			$selfnamed_order_id = '';

			if ( is_wp_error( $response ) || $response->error ) {
				if ( is_array( $response->data ) ) {
					$errors_list = array();

					foreach ( $response->data as $error ) {
						if ( ! empty( $error->message ) ) {
							$message_without_dot = rtrim( $error->message, '.' );
							$errors_list[]       = $message_without_dot;
						}
					}

					$message = ! empty( $errors_list ) ? implode( ', ', $errors_list ) . '.' : null;
				} else {
					$message = 'Something went wrong when sending order to Selfnamed.';

					if ( is_wp_error( $response->response ) ) {
						$message .= ' ' . $response->response->get_error_message();
					}

					if ( is_string( $response->data ) ) {
						$message .= ' ' . $response->data;
					}
				}

				$status = 'CANCELLED';
				$order_class->update_status( 'cancelled' );
			} else {
				$status             = $response->data->status;
				$planned_delivery   = $response->data->plannedDeliveryAt;
				$selfnamed_order_id = $response->data->id;
			}

			$order_class->update_meta_data( '_selfnamed_order_id', $selfnamed_order_id );
			$order_class->update_meta_data( '_selfnamed_order_message', $message );
			$order_class->update_meta_data( '_selfnamed_order_status', $status );
			$order_class->update_meta_data( '_selfnamed_order_planned_delivery', $planned_delivery );
			$order_class->update_meta_data( '_selfnamed_order_updated_at', time() );

			$order_class->save();
		}
	}

	/**
	 * Remove the order message, update_at and planned_delivery meta data.
	 *
	 * @param object $order The existing order object.
	 *
	 * @return void
	 */
	public function reset_selfnamed_meta( object $order ): void {
		$order->delete_meta_data( '_selfnamed_order_message' );
		$order->delete_meta_data( '_selfnamed_order_updated_at' );
		$order->delete_meta_data( '_selfnamed_order_planned_delivery' );
		$order->save();
	}

	/**
	 * Get WooCommerce orders that contain selfnamed products.
	 *
	 * @param array $order_ids Array of order IDs.
	 *
	 * @return array
	 */
	private function get_all( array $order_ids ): array {
		$selfnamed_orders = array();

		$args = $this->args;
		$args['post__in'] = $order_ids;

		$orders = wc_get_orders( $args );

		foreach ( $orders as $order ) {
			$normalized_order = $this->convert_to_selfnamed_format( $order );

			if ( ! empty( $normalized_order ) ) {
				$selfnamed_orders[] = $normalized_order;
			}
		}

		return $selfnamed_orders;
	}

	/**
	 * Prepare the WooCommerce order to Selfnamed format
	 *
	 * @param object $order The WooCommerce order.
	 *
	 * @return array|null
	 */
	private function convert_to_selfnamed_format( object $order ): ?array {
		if ( $order->get_meta( '_selfnamed_order_id' ) ) {
			return null;
		}

		$order_items   = array();
		$invoice_items = array();

		$shipping = $order->get_address( 'shipping' );
		$billing  = $order->get_address( 'billing' );

		$items = $order->get_items();
		if ( ! empty( $items ) ) {
			foreach ( $items as $item ) {
				$product = $item->get_product();

				if ( ! $product ) {
					continue;
				}

				$sn_product_id = $product->get_meta( '_selfnamed_product_id' );
				if ( ! empty( $sn_product_id ) ) {
					$response   = $this->api->get_product_by_id( (int) $sn_product_id );
					$sn_product = null;

					if ( ! $response->error ) {
						$sn_product = $response->data;
					}

					$order_items[] = array(
						'productId' => $sn_product_id,
						'quantity'  => $item->get_quantity(),
					);

					$invoice_items[] = array(
						'productTitle'      => $product->get_name(),
						'quantity'          => $item->get_quantity(),
						'itemPrice'         => $product->get_price(),
						'itemDiscount'      => '0',
						'itemDiscountPrice' => '0',
						'totalPrice'        => $item->get_subtotal(),
						'weight'            => $sn_product ? $sn_product->weight : 0,
						'volume'            => $sn_product ? $sn_product->volume : 0,
					);
				}
			}
		}

		if ( count( $order_items ) < 1 ) {
			return null;
		}

		$merchant = unserialize( get_option( '_selfnamed_billing_information' ) );
		if ( ! $merchant['name']
			||
			! $merchant['country']
			||
			! $merchant['city']
			||
			! $merchant['street']
			||
			! $merchant['zip']
			|| ! $merchant['phone']
			|| ! $merchant['email']
		) {
			$order->update_meta_data(
				'_selfnamed_order_message',
				"Please fill your company's billing information on the Getting Started section."
			);
			$order->update_meta_data( '_selfnamed_order_updated_at', time() );
			$order->update_meta_data( '_selfnamed_order_status', 'CANCELLED' );
			$order->update_status( 'cancelled' );
			$order->save();

			return null;
		}

		$shipping_methods      = $order->get_shipping_methods();
		$shipping_method       = reset( $shipping_methods );
		$shipping_method_group = null;
		if ( false !== $shipping_method ) {
			$shipping_method_group = $shipping_method->get_meta( '_selfnamed_method_shipping_group' );
		}

		$grand_total_net = number_format(
            (float) $order->get_total() - (float) $order->get_total_tax(),
            2,
            '.',
            ''
        );

		$data = array(
			'id'            => $order->get_id(),
			'orderItems'    => $order_items,
			'invoice'       => array(
				'orderNr'             => $order->get_id(),
				'orderDate'           => $order->get_date_created()->format( 'Y-m-d H:i:s' ),
				'items'               => $invoice_items,
				'buyer'               => array(
					'firstName' => $billing['first_name'] && $billing['last_name'] ? $billing['first_name']
						: $shipping['first_name'],
					'lastName'  => $billing['first_name'] && $billing['last_name'] ? $billing['last_name']
						: $shipping['last_name'],
					'email'     => $order->get_billing_email(),
					'phone'     => $billing['phone'] ?? $shipping['phone'],
				),
				'merchant'            => $merchant,
				'shippingAddress'     => array(
					'country'     => $shipping['country'],
					'city'        => $shipping['city'],
					'street'      => $shipping['address_1'] . ' ' . $shipping['address_2'],
					'streetNo'    => '',
					'flatNo'      => '',
					'zip'         => $shipping['postcode'],
					'phone'       => $shipping['phone'],
					'stateRegion' => isset($shipping['state']) ? $shipping['state'] : '',
				),
				'billingAddress'      => array(
					'country'     => $billing['country'],
					'city'        => $billing['city'],
					'street'      => $billing['address_1'] . ' ' . $billing['address_2'],
					'streetNo'    => '',
					'flatNo'      => '',
					'zip'         => $billing['postcode'],
					'phone'       => $billing['phone'],
					'stateRegion' => isset($billing['state']) ? $billing['state'] : '',
				),
				'subtotalNet'         => $order->get_subtotal(),
				'shippingPriceNet'    => $order->get_shipping_total(),
				'shippingTaxPrice'    => $order->get_shipping_tax(),
				'grandTotalNet'       => $grand_total_net,
				'taxRate'             => number_format(
					$grand_total_net > 0 ? ( $order->get_total_tax()
																				/ $grand_total_net * 100 ) : 0
				),
				'taxPrice'            => $order->get_total_tax(),
				'grandTotal'          => $order->get_total(),
				'currency'            => $order->get_currency(),
				'shippingMethodTitle' => $order->get_shipping_method() ? $order->get_shipping_method() : '-',
				'paymentMethodTitle'  => $order->get_payment_method_title() ? $order->get_payment_method_title() : '-',
			),
			'shippingGroup' => $shipping_method_group,
		);

		return $data;
	}
}
