<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

/**
 * Order_Flags class.
 */
class Order_Flags {
	/**
 	 * Initialization.
 	 *
 	 * @return void
 	 */
	public static function create() {
		new self();
	}

	/**
 	 * Constructor.
 	 */
	public function __construct() {
		add_action( 'woocommerce_new_order_item', array( $this, 'on_new_order_item' ), 20, 3 );
		add_action( 'woocommerce_before_delete_order_item', array( $this, 'on_before_delete_order_item' ), 20, 1 );
	}

	/**
	 * Handle new order item added and re-check the flag.
	 *
	 * @param int                 $item_id  Item ID.
	 * @param \WC_Order_Item|null $item     Item object.
	 * @param int|null            $order_id Order ID.
	 *
	 * @return void
	 */
	public function on_new_order_item( $item_id, $item, $order_id ) {
		if ( ! empty( $order_id ) ) {
			$this->mark_if_selfnamed_order( $order_id );
			return;
		}
	}

	/**
	 * Before an order item is deleted, capture its order ID so we can process after deletion.
	 *
	 * @param int $item_id Item ID.
	 *
	 * @return void
	 */
	public function on_before_delete_order_item( $item_id ) {
		$order_id = (int) wc_get_order_id_by_order_item_id( $item_id );
		if ( $order_id ) {
			$this->mark_if_selfnamed_order( $order_id, array( $item_id ) );
		}
	}

	/**
 	 * Mark order as selfnamed if it contains any Selfnamed item.
 	 *
 	 * @param int $order_id The order ID.
	 * @param array $exclude_items Array of item IDs to exclude from the check.
 	 *
 	 * @return void
 	 */
	private function mark_if_selfnamed_order( $order_id, $exclude_items = array() ) {
		if ( empty( $order_id ) ) {
			return;
		}

		$order = wc_get_order( $order_id );
		if ( ! $order || ! is_a( $order, 'WC_Order' ) ) {
			return;
		}

		$items = $order->get_items();
		if ( ! empty( $exclude_items ) ) {
			$items = array_filter( $items, function( $item ) use ( $exclude_items ) {
				return ! in_array( $item->get_id(), $exclude_items );
			} );
		}

		if ( $this->contains_selfnamed_products( $items ) ) {
			$order->update_meta_data( '_is_selfnamed_order', 1 );
			$order->save();
		} else {
			$order->delete_meta_data( '_is_selfnamed_order' );
			$order->save();
		}
	}
	
	/**
	 * Determine whether an items contains any Selfnamed product.
	 *
	 * @param array $items Array of order items.
	 *
	 * @return bool
	 */
	private function contains_selfnamed_products( $items ) {
		if ( empty( $items ) ) {
			return false;
		}

		foreach ( $items as $item ) {
			if ( is_a( $item, 'WC_Order_Item_Product' ) ) {
				$product = $item->get_product();
				if ( $product && $product->get_meta( '_selfnamed_product_id' ) ) {
					return true;
				}
			}
		}

		return false;
	}
}


