<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

use Exception;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Notification class.
 */
class Notification {
	/**
	 * Api model.
	 *
	 * @var Api
	 */
	private $api;

	/**
	 * Product model.
	 *
	 * @var Product
	 */
	private $product_model;

	/**
	 * Initialization.
	 *
	 * @return void
	 */
	public static function create() {
		new self();
	}

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->api           = new Api();
		$this->product_model = new Product();

		add_action( 'rest_api_init', array( $this, 'register_custom_api_endpoint_notification' ) );
	}

	/**
	 * Register notification API endpoint.
	 *
	 * @return void
	 */
	public function register_custom_api_endpoint_notification() {
		register_rest_route(
			'sn',
			'notification',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'custom_api_endpoint_notification' ),
				'permission_callback' => function( $request ) {
					$body = json_decode( $request->get_body() );
					return isset($body->accessToken) &&
					$body->accessToken === get_option( 'selfnamed_platform_access_token' );
				}
			)
		);
	}

	/**
	 * Notification API endpoint.
	 *
	 * @param WP_REST_Request $request The request.
	 *
	 * @return WP_REST_Response
	 * @throws Exception Throws if request is wrong.
	 */
	public function custom_api_endpoint_notification( WP_REST_Request $request ): WP_REST_Response {
		try {
			$body = json_decode( $request->get_body() );

			switch ( $body->type ) {
				case 'PRODUCT':
					$this->update_product( (int) $body->{'relatedId'} );
					break;
				case 'ORDER':
					$this->update_order( (int) $body->{'relatedId'} );
					break;
				case 'STORE':
					$this->update_store( (string) $body->{'accessToken'} );
					break;
				default:
					return new WP_REST_Response(
						array(
							'success' => false,
							'error'   => 'The provided notification type is not supported!',
						),
						400
					);
			}
		} catch ( Exception $error ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'error'   => $error->getMessage(),
				),
				400
			);
		}

		return new WP_REST_Response( array( 'success' => true ), 200 );
	}

	/**
	 * Update product information based on selfnamed id from notification.
	 *
	 * @param int $selfnamed_id Selfnamed product's id.
	 *
	 * @return void
	 * @throws Exception Throws if failed to update the product.
	 */
	private function update_product( int $selfnamed_id ) {
		$product_id = $this->get_post_id_by_meta_key_and_value( '_selfnamed_product_id', esc_html( $selfnamed_id ) );

		if ( ! $product_id ) {
			throw new Exception( "The product hasn't been found with the id - " . esc_html( $selfnamed_id ) );
		}

		$response = $this->api->get_product_by_id( $selfnamed_id );

		if ( $response->error ) {
			throw new Exception( 'The update failed for the product id - ' . esc_html( $selfnamed_id ) );
		}

		$product = $response->data;

		$product->is_in_stock = $product->{'isInStock'};

		wp_update_post(
			array(
				'ID'           => $product_id,
				'post_title'   => $product->name,
				'post_content' => $product->description,
				'post_name'    => $product->name,
			)
		);

		$this->product_model->update_product_meta( (int) $product_id, $product );
	}

	/**
	 * Update order based on the selfnamed order id from notification.
	 *
	 * @param int $selfnamed_id The order's id from the Selfnamed API.
	 *
	 * @return void
	 * @throws Exception Throws if the update failed.
	 */
	private function update_order( int $selfnamed_id ) {
		$orders = wc_get_orders(
			array(
				'limit'        => 1,
				'orderby'      => 'date',
				'order'        => 'DESC',
				'meta_key'     => '_selfnamed_order_id',
				'meta_value'   => $selfnamed_id,
				'meta_compare' => '=',
			)
		);
		$order  = reset( $orders );

		$response = $this->api->get_order_by_id( $selfnamed_id );

		if ( $response->error ) {
			throw new Exception( 'The update failed for the order id - ' . esc_html( $selfnamed_id ) );
		}

		switch ( $response->data->status ) {
			case 'CANCELLED':
				$order->update_status( 'cancelled' );
				break;
			case 'SHIPPED':
				$order->update_status( 'completed' );
				break;
		}

		$order->update_meta_data( '_selfnamed_order_status', $response->data->{'status'} );
		$order->update_meta_data( '_selfnamed_order_planned_delivery', $response->data->{'plannedDeliveryAt'} );
		$order->update_meta_data( '_selfnamed_order_updated_at', time() );

		$tracking_ids  = $response->data->{'trackingIds'};
		$tracking_urls = $response->data->{'trackingUrls'};

		if ( isset( $tracking_ids ) && is_array( $tracking_ids ) && isset( $tracking_urls ) && is_array( $tracking_urls ) ) {
			$order->update_meta_data( 'selfnamed_order_tracking_numbers', implode( ', ', $tracking_ids ) );
			$order->update_meta_data( 'selfnamed_order_tracking_urls', implode( ', ', $tracking_urls ) );
		}

		$order->save();
	}

	/**
	 * Update the shipping profile option.
	 *
	 * @param string $access_token The platform access token generated on setup.
	 *
	 * @return void
	 * @throws Exception Throws if the update fails.
	 */
	private function update_store( string $access_token ) {
		$response = $this->api->get_stores();

		if ( $response->error ) {
			throw new Exception( 'The update failed for the store' );
		}

		$config = null;
		if ( ! empty( $response->data ) ) {
			foreach ( $response->data as $store ) {
				if ( $store->{'platformAccessToken'} === $access_token ) {
					$config = $store;
				}
			}
		}

		if ( ! is_null( $config ) ) {
			update_option( Setup::IS_USING_CUSTOM_SHIPPING_PROFILE, wp_json_encode( $config->{'isUsingCustomShippingProfile'} ) );
		}
	}

	/**
	 * Get post id by meta key and its value.
	 *
	 * @param string $key   The meta key.
	 * @param string $value The value of the meta key.
	 *
	 * @return bool|string
	 */
	private function get_post_id_by_meta_key_and_value( string $key, string $value ) {
		global $wpdb;
		$meta = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT * FROM ' . $wpdb->postmeta
													. ' WHERE meta_key=%s AND meta_value=%s',
				$key,
				$value
			)
		);
		if ( is_array( $meta ) && ! empty( $meta ) && isset( $meta[0] ) ) {
			$meta = $meta[0];
		}
		if ( is_object( $meta ) ) {
			return $meta->post_id;
		} else {
			return false;
		}
	}
}
