<?php
/**
 * This file is part of the Magebit_Selfnamed package.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future.
 *
 * Selfnamed: Cosmetics on demand extension is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @package Magebit_Selfnamed
 */

namespace Selfnamed\Model;

use Exception;
use Selfnamed\ViewModel\Pagination;
use stdClass;

/**
 * Api class.
 */
class Api {
	const API_URL     = 'https://ds-api.selfnamed.com/';
	const API_URL_KEY = 'selfnamed_api_url';

	const PER_PAGE      = 20;
	const MAX_PAGE_SIZE = 50;

	/**
	 * Setup model.
	 *
	 * @var Setup
	 */
	private $setup;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->setup = new Setup();
	}

	/**
	 * Initialization.
	 *
	 * @return void
	 */
	public static function create() {
		new self();
	}

	/**
	 * Request for the Selfnamed API endpoint.
	 *
	 * @param string $path   The Selfnamed API endpoint path.
	 * @param int    $page   The current page for pagination.
	 * @param int    $size   How many items per page.
	 * @param array  $params Query parameters (GET) to add to the URL.
	 * @param array  $args   The args for wp_remote_get.
	 *
	 * @return stdClass
	 *
	 * @see wp_remote_get() For more information on the args array.
	 */
	public function get_from_endpoint(
		string $path = '',
		int $page = 1,
		int $size = self::PER_PAGE,
		array $params = array(),
		array $args = array()
	): stdClass {
		$result             = new stdClass();
		$result->error      = false;
		$result->response   = null;
		$result->data       = null;
		$result->pagination = null;

		if ( ! $this->setup->is_ready() ) {
			$result->error    = true;
			$result->response = 'No API key provided!';

			return $result;
		}

		$default_params = array(
			'page'     => $page,
			'per-page' => $size,
		);

		$default_args = array(
			'timeout' => 30,
			'headers' => $this->get_auth_header(),
		);

		try {
			$response = wp_remote_get(
				self::get_api_url( $path, array_merge( $default_params, $params ) ),
				array_merge( $default_args, $args )
			);

			if ( is_wp_error( $response ) ) {
				$result->error    = true;
				$result->response = $response;
			} else {
				$result->data       = json_decode( wp_remote_retrieve_body( $response ) );
				$result->pagination = new Pagination(
					(int) $response['headers']['x-pagination-current-page'],
					(int) $response['headers']['x-pagination-page-count'],
					(int) $response['headers']['x-pagination-total-count'],
					(int) $response['headers']['x-pagination-per-page']
				);

				if ( is_object( $result->data ) && 'Unauthorized' === $result->data->message ) {
					$result->error = true;
				}
			}
		} catch ( Exception $e ) {
			$result->error    = true;
			$result->response = $e;
		}

		return $result;
	}

	/**
	 * Get products from the Selfnamed API.
	 *
	 * @param int $page The page number.
	 * @param int $size The items count.
	 *
	 * @return stdClass
	 */
	public function get_products( int $page = 1, int $size = self::PER_PAGE ): stdClass {
		return $this->get_from_endpoint( 'products', $page, $size );
	}

	/**
	 * Get stores from the Selfnamed API
	 *
	 * @param int $page The page number.
	 * @param int $size The items count.
	 *
	 * @return stdClass
	 */
	public function get_stores( int $page = 1, int $size = self::PER_PAGE ): stdClass {
		return $this->get_from_endpoint( 'stores', $page, $size );
	}

	/**
	 * Get product from the Selfnamed API by specific id.
	 *
	 * @param int $id The selfnamed product's id.
	 *
	 * @return stdClass
	 */
	public function get_product_by_id( int $id ): stdClass {
		$result           = new stdClass();
		$result->error    = false;
		$result->response = null;
		$result->data     = null;

		if ( ! $this->setup->is_ready() ) {
			$result->error    = true;
			$result->response = 'No API key provided!';

			return $result;
		}

		$args = array(
			'timeout' => 30,
			'headers' => $this->get_auth_header(),
		);

		try {
			$response = wp_remote_get(
				self::get_api_url( 'products/' . $id ),
				array_merge( $args )
			);

			if ( is_wp_error( $response ) || ! self::is_success_response_code( $response ) ) {
				$result->data     = json_decode( wp_remote_retrieve_body( $response ) );
				$result->error    = true;
				$result->response = $response;
			} else {
				$result->data = json_decode( wp_remote_retrieve_body( $response ) );
			}
		} catch ( Exception $e ) {
			$result->error    = true;
			$result->response = $e;
		}

		return $result;
	}

	/**
	 * Get shipping zones from Selfnamed API.
	 *
	 * @param int $page The page number.
	 * @param int $size The items count.
	 *
	 * @return stdClass
	 */
	public function get_shipping_zones( int $page = 1, int $size = self::PER_PAGE ): stdClass {
		return $this->get_from_endpoint( 'shipping-zones', $page, $size );
	}

	/**
	 * Get all shipping zones include results from all pages from Selfnamed API.
	 *
	 * @return stdClass
	 */
	public function get_all_shipping_zones(): stdClass {
		$result           = new stdClass();
		$result->error    = false;
		$result->response = null;
		$result->data     = array();

		$authorized     = true;
		$shipping_zones = array();

		$cur_page    = 1;
		$total_pages = INF;

		while ( $cur_page <= $total_pages && $authorized ) {
			$shipping_zones_result = $this->get_shipping_zones( $cur_page, self::MAX_PAGE_SIZE );

			if ( true === $shipping_zones_result->error ) {
				$authorized       = false;
				$result->error    = true;
				$result->response = $shipping_zones_result->response;
			} else {
				$shipping_zones = array_merge( $shipping_zones, $shipping_zones_result->data );
				$total_pages    = $shipping_zones_result->pagination->pages;
				++$cur_page;
			}
		}

		if ( $authorized ) {
			$result->data = $shipping_zones;
		}

		return $result;
	}

	/**
	 * Get api key for the auth header to use in the request for the Selfnamed API.
	 *
	 * @return string[]
	 */
	private function get_auth_header(): array {
		return array( 'Authorization' => 'Bearer ' . $this->setup->get_api_key() );
	}

	/**
	 * Get order by specific selfnamed order's id.
	 *
	 * @param int $id The selfnamed order's id.
	 *
	 * @return stdClass
	 */
	public function get_order_by_id( int $id ): stdClass {
		$result           = new stdClass();
		$result->error    = false;
		$result->response = null;
		$result->data     = null;

		if ( ! $this->setup->is_ready() ) {
			$result->error    = true;
			$result->response = 'No API key provided!';

			return $result;
		}

		$args = array(
			'timeout' => 30,
			'headers' => $this->get_auth_header(),
		);

		try {
			$response = wp_remote_get(
				self::get_api_url( 'orders/' . $id ),
				array_merge( $args )
			);

			if ( is_wp_error( $response ) || ! self::is_success_response_code( $response ) ) {
				$result->error    = true;
				$result->response = $response;
			} else {
				$result->data = json_decode( wp_remote_retrieve_body( $response ) );
			}
		} catch ( Exception $e ) {
			$result->error    = true;
			$result->response = $e;
		}

		return $result;
	}

	/**
	 * Send orders to the Selfnamed.
	 *
	 * @param array   $invoice        The invoice data.
	 * @param array   $order_items    The order items data.
	 * @param ?string $shipping_group The selfnamed shipping method group.
	 *
	 * @return stdClass
	 */
	public function send_orders( array $invoice, array $order_items, ?string $shipping_group ): stdClass {
		$result           = new stdClass();
		$result->error    = false;
		$result->response = null;
		$result->data     = null;

		try {
			$response = wp_remote_post(
				self::get_api_url( 'orders' ),
				array(
					'timeout' => 30,
					'headers' => $this->get_auth_header(),
					'body'    => array(
						'invoice'       => $invoice,
						'orderItems'    => $order_items,
						'shippingGroup' => $shipping_group,
					),
				)
			);

			if ( is_wp_error( $response ) ) {
				$result->error    = true;
				$result->response = $response;
			} else {
				if ( ! self::is_success_response_code( $response ) ) {
					$result->error    = true;
					$result->response = $response;
				}
				$result->data = json_decode( wp_remote_retrieve_body( $response ) );
			}
		} catch ( Exception $e ) {
			$result->error    = true;
			$result->response = $e;
		}

		return $result;
	}

	/**
	 * Create a new store on the Selfnamed.
	 *
	 * @param string $platform_store_id         The platform store id.
	 * @param string $storefront_url            The current storefront url.
	 * @param string $notification_callback_url The notification callback url.
	 * @param string $platform_access_token     The platform access token.
	 * @param string $api_key                   The api key.
	 * @param string $title                     The name of the store.
	 * @param string $type                      The type of the store.
	 *
	 * @return stdClass
	 */
	public static function send_api_key(
		string $platform_store_id,
		string $storefront_url,
		string $notification_callback_url,
		string $platform_access_token,
		string $api_key,
		string $title,
		string $type = 'WOOCOMMERCE'
	): stdClass {
		$result           = new stdClass();
		$result->error    = false;
		$result->response = null;
		$result->data     = null;

		try {
			$response = wp_remote_post(
				self::get_api_url( 'stores' ),
				array(
					'body' => array(
						'type'                    => $type,
						'platformStoreId'         => $platform_store_id,
						'storefrontUrl'           => $storefront_url,
						'notificationCallbackUrl' => $notification_callback_url,
						'platformAccessToken'     => $platform_access_token,
						'snAccessToken'           => $api_key,
						'title'                   => $title,
					),
				)
			);

			if ( is_wp_error( $response ) ) {
				$result->error    = true;
				$result->response = $response;
			} elseif ( ! self::is_success_response_code( $response ) ) {
				$result->error    = true;
				$result->response = $response;
			} else {
				$result->data = json_decode( wp_remote_retrieve_body( $response ) );
			}
		} catch ( Exception $e ) {
			$result->error    = true;
			$result->response = $e;
		}

		return $result;
	}

	/**
	 * Check if the response code equals 2xx.
	 *
	 * @param array $response The response from API.
	 *
	 * @return bool
	 */
	private static function is_success_response_code( array $response ): bool {
		if ( ! $response['response']
				|| ! is_array( $response['response'] )
				|| ! isset( $response['response']['code'] )
		) {
			return false;
		}

		return $response['response']['code'] >= 200 && $response['response']['code'] < 300;
	}

	/**
	 * Build query for requests.
	 *
	 * @param array $query The query.
	 *
	 * @return string
	 */
	private static function build_query( array $query = array() ): string {
		$query_pairs = array();

		foreach ( $query as $key => $val ) {
			$query_pairs[] = urlencode( $key ) . '=' . urlencode( $val );
		}

		return ( count( $query_pairs ) > 0 ? '?' : '' ) . implode( '&', $query_pairs );
	}

	/**
	 * Get selfnamed API endpoint.
	 *
	 * @param string $path  The path of the selfnamed endpoints.
	 * @param array  $query The query.
	 *
	 * @return string
	 */
	private static function get_api_url( string $path = '', array $query = array() ): string {
		$persisted_api_key = get_option( self::API_URL_KEY );

		if ( empty( $persisted_api_key ) ) {
			$api_url = self::API_URL;
		} else {
			$api_url = $persisted_api_key;
		}

		// Remove trailing slash from API_URL if added.
		if ( substr( $api_url, - 1 ) === '/' ) {
			$api_url = substr( $api_url, 0, - 1 );
		}

		// Remove leading slash from $path if added.
		if ( substr( $path, 0, 1 ) === '/' ) {
			$path = substr( $path, 1 );
		}

		return $api_url . '/' . $path . self::build_query( $query );
	}
}
